#!/usr/bin/env python3
# Copyright (c) 2008, 2009 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# in autobuild sometimes SimpleConfigFile is not found
# pylint: disable-msg=F0401

import argparse
import atexit
import os
import subprocess
import sys
import tempfile
import zipfile

import Swi
from Swi import sign, SwiOptions
import SimpleConfigFile

_blessVersion = ''

SUCCESS = 0
BLESS_FAIL = 1

def bless( swiFile, skipSigning=False, blessVersion='', printDebugs=None,
           blessVal=1 ):
   global _blessVersion
   if not _blessVersion:
      # This is just used for testing when we're calling bless by itself.
      # In normal execution we would call blessHandler which would set _blessVersion.
      _blessVersion = blessVersion

   blessImage( swiFile, blessVal )

   if not skipSigning:
      if printDebugs:
         sign.printDebugs = True
      useDevCA = ( blessVal == 0 )
      return sign.sign( swiFile, useDevCA=useDevCA,
                        forceSign=True )
   return SUCCESS

def blessImage( swiFile, blessVal ):
   """ Bless all versions in provided swi image

   Args:
      swiFile: str: Path to swi image
      blessVal: int: value to store as BLESSED=<val> in version files.
   """

   assert blessVal in ( 0, 1 ), "Bless val needs to be set to 0 or 1"

   if not zipfile.is_zipfile( swiFile ):
      print( swiFile + ' does not appear to be a swi image' )
      sys.exit( 1 )

   # Prepare temporary dir with cleanup
   tmpDir = tempfile.mkdtemp()
   def cleanupTmpDir():
      Swi.run( [ 'sudo', 'rm', '-rf', tmpDir ] )
   atexit.register( cleanupTmpDir )

   # Bless all found version files and put them back into swi
   with zipfile.ZipFile( swiFile, mode='r' ) as swi:
      for filename in swi.namelist():
         # Don't use zipfile library to extract files. It somehow messes up
         # permissions on binary files
         subprocess.check_call( [ "unzip", "-q", swiFile, filename, "-d", tmpDir ] )
         if not filename.endswith( 'version' ):
            continue

         fileToBless = os.path.join( tmpDir, filename )
         blessVersionFile( fileToBless, blessVal )

   # Recreate swi
   options = SwiOptions( dirname=tmpDir, squashfs=False, fast=False,
                         installfs_only=False, force=True, trace=True,
                         zstd=False, force_resquash=False, update_version=False )
   Swi.create.create( swiFile, options )

def blessVersionFile( swiVersion, blessVal ):
   Swi.run( [ "sudo", "chmod", "666", swiVersion ] )
   config = SimpleConfigFile.SimpleConfigFileDict( swiVersion )
   # Ensure blessVersion and the SWI_VERSION match if the blessVal is 1, which
   # indicates this isn't a dummy blessing
   if blessVal == 1 and config.get( 'SWI_VERSION' ) != _blessVersion:
      print( f"\nERROR: SWI version ({config.get( 'SWI_VERSION' )}) and "
             f"blessed version ({_blessVersion}) do not match. Please check "
             "the project setting 'swiRelease' and set the correct value or "
             "get a successful build\n" )
      sys.exit( BLESS_FAIL )
   if config.get( 'BLESSED' ) and config[ 'BLESSED' ] == '1':
      print( swiVersion, "already blessed" )
   else:
      print( "blessing", swiVersion )
      config[ 'BLESSED' ] = blessVal
   Swi.run( [ "sudo", "chmod", "644", swiVersion ] )

def maybeRerunAsRoot( args ):
   ''' Runs the same command as sudo if not su. '''
   if os.geteuid() != 0:
      os.execvp( 'sudo', [ 'sudo', 'swi', 'bless' ] + args )

def blessHandler( args ):
   if not args:
      args = args[1:]
   maybeRerunAsRoot( args )
   global _blessVersion
   parser = argparse.ArgumentParser( prog="swi bless" )
   parser.add_argument( 'swi', metavar='EOS.swi',
                        help="Path of the SWI to be blessed" )
   parser.add_argument( 'version',
                        help="EOS release version the SWI has to be blessed for" )
   parser.add_argument( '--skip-signing',
                        help="Skip signing the SWI.", action="store_true" )
   parser.add_argument( '--verbose',
                        help="Print some progress info.", action="store_true" )
   parser.add_argument( '--bless-val', type=int, default=1,
                        help="The value to store as BLESSED=<value> within "
                             "the version file. If --bless-val=0, the "
                             "image will be signed using the dev certificate." )

   args = parser.parse_args( args )
   if not args.version:
      parser.error( "Version is not specified. Typically, it's the value of "
                    "'swiRelease' project setting." )
   _blessVersion = args.version
   retCode = bless( args.swi, args.skip_signing, args.version, args.verbose,
                    args.bless_val )
   sys.exit( retCode )
