#!/usr/bin/env python3
# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import argparse
import os
import shutil
import sys
import tarfile
import zipfile

import ChecksumsGenLib
import SwimHelperLib

def genImageChecksumsJson( imagePath ):
   # swi checksum is run against various image files that are published
   # for releases. The typical filetypes we expect are zip ( swi/swix ),
   # tar ( e.g. cEOS-lab.tar.xz ), and vmdks. We accept all filetypes
   # so that the swi checksum tool is forward compatible with new filetypes
   # that may be published in the future.
   if tarfile.is_tarfile( imagePath ):
      print( "Generating checksum data for tar file image" )
      ChecksumsGenLib.genImageChecksums( imagePath )
      return

   if not zipfile.is_zipfile( imagePath ):
      # pylint: disable-next=consider-using-f-string
      print( "Generating checksum data for image %s" % imagePath )
      ChecksumsGenLib.genImageChecksums( imagePath )
      return

   workDir = "/tmp/optims-%d" % os.getpid() # pylint: disable=consider-using-f-string
   os.mkdir( workDir )
   try:
      # Check if image is in SWIM or legacy format
      if SwimHelperLib.isSwimImage( imagePath ):
         # pylint: disable-next=consider-using-f-string
         print( "Generating checksum data for SWIM image %s" % imagePath )
         ChecksumsGenLib.genSwimImageChecksums( imagePath, workDir )
      else:
         # pylint: disable-next=consider-using-f-string
         print( "Generating checksum data for SWI(X) image %s" % imagePath )
         ChecksumsGenLib.genImageChecksums( imagePath )
   finally:
      shutil.rmtree( workDir )

def checksumHandler( args ):
   args = sys.argv[ 1 : ] if args is None else args
   parser = argparse.ArgumentParser( prog="swi checksum" )
   parser.add_argument( "image",
                        help="Path of the image to generate checksum data for" )
   args = parser.parse_args( args )

   genImageChecksumsJson( args.image )
   # pylint: disable-next=consider-using-f-string
   print( "Finished generating checksum data for %s." % args.image )
