#!/usr/bin/env python3
# Copyright (c) 2008, 2009, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import optparse # pylint: disable=deprecated-module
import shutil
import subprocess
import sys, os
import tempfile
import time
import traceback
import zipfile
import platform
import SwimFlatten
import SwimHelperLib

from ImageVars import swimFlavorToFullOptimization

TIMESTAMP_FILE_NAME = 'swi.resquash.timestamp'


if platform.machine() == 'aarch64':  # A4NOCHECK
   farch = "aarch64"
else:
   farch = "i386"

def createTimestampFile( directory ):
   """ Creates a timestamp that will be used later to decide which
       squashes should be recreated. """
   timestamp = os.path.join( directory, TIMESTAMP_FILE_NAME )
   with open( timestamp, 'a' ):
      # Create or update existing timestamp
      os.utime( timestamp, None )

   time.sleep( 0.1 )

def extractRootfs( filename, opts=None, quiet=False, unsquashArgs=None ):
   assert filename.endswith( ( ".sqsh", ".gz" ) )

   if unsquashArgs is None:
      unsquashArgs = []

   if opts and opts.dirname:
      dirname = opts.dirname
   else:
      dirname = ".".join( filename.split( "." )[ :-1 ] ) + ".dir"

   if filename.endswith( ".sqsh" ):
      cmd = []
      if os.geteuid() != 0:
         cmd += [ "sudo" ]
      cmd += [ "unsquashfs" ]
      if quiet:
         cmd += [ "-q" ]
      cmd += [ "-n", "-d", dirname, filename ] + unsquashArgs
      subprocess.check_call( cmd )
   else:
      subprocess.check_call( [ "mkdir", "-p", dirname ] )
      shutil.move( filename, dirname )
      basename = os.path.basename( filename )
      curdir = os.path.abspath( os.curdir )
      os.chdir( dirname )
      subprocess.check_output( "gunzip -c {basename} | sudo cpio -ic".format( 
         basename=basename ), shell=True )
      os.remove( basename )
      os.chdir( curdir )

   createTimestampFile( os.path.dirname( filename ) )

def extractFilePathFromSqshInSwi( swiFile, sqshName, extDir, filePath,
                                  useRegexFilePath=False ):
   extractedSqshFile = unzipSqshFromSwi( swiFile, sqshName, extDir )

   unsqshCmd = [ "unsquashfs", "-f", "-q", "-n" ]
   if useRegexFilePath:
      unsqshCmd += [ "-r" ]
   unsqshCmd += [ "-d", extDir, extractedSqshFile, filePath ]

   subprocess.check_call( unsqshCmd )

def unzipSqshFromSwi( swiFile, sqshName, extDir ):
   extractedSqshFile = os.path.join( extDir, sqshName )
   try:
      with zipfile.ZipFile( swiFile, 'r' ) as swiZip:
         zippedFiles = swiZip.namelist()

         if sqshName not in zippedFiles:
            print( f"{sqshName} is not in {swiFile}. Unable to extract." )
            sys.exit( 1 )

         swiZip.extract( sqshName, path=extDir )
         if not os.path.isfile( extractedSqshFile ):
            print( f"Failed to extract {sqshName} from {swiFile}" )
            sys.exit( 1 )

   except zipfile.BadZipfile:
      print( "%s does not appear to be a swi image." % swiFile )
      sys.exit( 1 )
   except Exception: # pylint: disable=broad-except
      traceback.print_exc()
      print( "Error - unable to read from SWI" )
      sys.exit( 1 )
   return extractedSqshFile

def createVariantRootfs( swiDir, optimization ):
   _, optimizations = SwimHelperLib.getSwimSqshMapAndOverlayDirs( swiDir, None )

   # Allow Flavors (DEFAULT/DPE) as optimizations and if so
   # pick the full optimization in that flavor
   optimizationName = optimization 
   if optimizationName not in optimizations:
      optimizationName = swimFlavorToFullOptimization.get( optimizationName )
   if optimizationName not in optimizations:
      print( "ERROR: bad optimization '{}' should be one of '{}'".format(
             optimization, " ".join( optimizations.keys() ) ) )
      return
   # concat top squash (<optimizationName>.rootfs-i386) and lower squashes together
   squashes = optimizations[ optimizationName ][ "root" ] + \
              ":" + optimizations[ optimizationName ][ "lower" ]
   # using nonOverlay so people can simply "rm -rf" when done (no umount needed)
   SwimFlatten.nonOverlayDupe( optimizationName, squashes, swiDir,
                               "rootfs-%s.dir" % farch )

def imageDirIsExt4():
   p = subprocess.run( [ 'df', '-T', '/images' ], stdout=subprocess.PIPE,
                       check=False )
   return "ext4" in p.stdout.decode( "utf-8" )

# Image extraction requires a filesystem that supports
# overlay and xattrs.  The AutoTest environment, in its
# wisdom, creates a /tmp that does *not* satisfy these
# requirements.  We've been told that using /images is
# the right thing for this case - please refer to
# https://arista.discourse.team/t/swi-flavor-during-art-sanitize-is-failing-due-to-
#    xattr-problems-starting-around-2022-04-17-and-increasing-drastically/3363/14
# and regarding stest, see
# https://discourse.arista.com/t/
#    stest-runs-with-tmp-filesystem-that-does-not-support-xattr/12212/3
def setTempDirIfNeeded():
   if ( ( 'AUTOTEST' in os.environ or 'STEST' in os.environ ) and
        ( os.path.exists( '/images' ) and imageDirIsExt4() ) ):
      tempfile.tempdir = '/images'

def maybePrintAutoTestIncompatibleFsWarning( targetDir ):
   if ( 'AUTOTEST' in os.environ or 'STEST' in os.environ ) and \
         not os.path.realpath( targetDir ).startswith( '/images/' ):
      print( "Warning: Extracting SWI to a possibly incompatible "
             "filesystem %s" % targetDir )

def extract( _file, opts=None, quiet=False, rpmOp=False, readOnly=False,
             unsquash=True ):
   try:
      # pylint: disable-next=consider-using-with
      z = zipfile.ZipFile( os.path.abspath( _file ) )
   except zipfile.BadZipfile:
      print( "%s does not appear to be a swi image." % _file )
      return 1
   if opts and opts.dirname:
      _dir = opts.dirname
      if os.path.exists( _dir ) and not opts.use_existing:
         print( "%s already exists and I don't have the guts to run rm -rf" % _dir )
         print( "Please delete it so I don't have to." )
         return 1
   else:
      _dir = os.path.basename( _file ).rsplit('.', 1)[0]
      if os.path.exists( _dir ):
         print( "%s already exists, use -d to specify extract directory." % _dir )
         return 1
   # This warning message is printed when being run in an AutoTest workspace and
   # if the target dir doesn't start with '/images/' ( which should support xattrs
   # in AutoTest ). The motive here is to logre the warning message to find all
   # tests that are possibly affected by "AutoTest Outside of Workspace"
   # which has changed the filesystem of where swi extract typically extracts the
   # SWI to ( /tmp ), causing unsquashfs failures due to unsupported xattrs.
   maybePrintAutoTestIncompatibleFsWarning( _dir )

   subprocess.check_call( [ "mkdir", "-p", _dir ] )

   for f in z.namelist():
      subprocess.check_call( [ "unzip", "-q", _file, f, "-d", _dir ] )
      if not unsquash:
         continue
      if ( f.startswith( "rootfs" ) or
            ( f.endswith( ".sqsh" ) and
               ( not ( rpmOp and readOnly ) or
                  f.startswith( 'Default' ) or
                     '.rpmdb.sqsh' in f ) ) ):
         extractRootfs( os.path.join( _dir, f ), quiet=quiet,
                        unsquashArgs=[ "/var/lib/rpm" ] if ( rpmOp and readOnly )
                                                        else None )

   # If this is a modular swi, assemble the disparate squash dirs into rootfs-i386
   # dir of yore, for developer's convenience or to keep older tests happy.
   if SwimHelperLib.isSwimImage( _file ):
      try: # can be called from outside main with fake opts
         if opts and opts.optimization:
            createVariantRootfs( _dir, opts.optimization )
      except AttributeError:
         pass

   createTimestampFile( _dir )

   return 0

def extractHandler( args=None ):
   args = sys.argv[1:] if args is None else args
   op = optparse.OptionParser(
         prog="swi extract",
         usage="usage: %prog EOS.swi" )
   op.add_option( '-d', '--dirname', action='store',
                  help='Directory in which to place the results')
   op.add_option( '-r', '--rootfs', action='store_true',
                  help='Extract a rootfs-formatted file instead of a full swi file' )
   op.add_option( '--use_existing', action='store_true',
                  help='Use existing directory to extract into' )
   op.add_option( '-v', '--optimization', action='store',
                  help='modular swi: the optimization to assemble into '
                       'rootfs-%s.dir (default: None). use "DEFAULT"/"DPE" '
                       'for the full image'  % farch )
   opts, args = op.parse_args( args )

   if len( args ) != 1:
      op.error( 'Please give me exactly one swi file!' )
   if opts.rootfs:
      extractRootfs( args[0], opts )
   else:
      sys.exit( extract( args[0], opts ) )

if __name__ == "__main__":
   extractHandler()
