#!/usr/bin/env python3
# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

from pprint import pprint
import json
import os, re, sys
import Swi, Swi.rpm, Swi.update
import tempfile
import argparse

from AddInOrder import AddEnableDisableReposInOrder
from EosVersion import swiFlavorDPE, swiFlavorPdp, swiFlavorCloud, \
                       swiFlavorDPEContainer

import ImageVars
from RepoArg import createRepoArgs, RepoArgOp
import SwimFlatten
import SwimHelperLib
from SwimFlavoringLib import flavorSwimWithExtraPkgs
import yaml
import platform
from subprocess import CalledProcessError
# pkgdeps: rpm rpm

repoFileTemplate = '''[{repoName}]
name={repoName}
baseurl={url}
skip_if_unavailable=False
priority=1
enabled=0
'''

if platform.machine() == 'aarch64':  # A4NOCHECK
   farch = "aarch64"
else:
   farch = "i386"

internationalPackages = [ 'Eos-international' ]

extraPkgsContainer = [ 'docker-ce', 'docker-buildx-plugin', 'docker-compose-plugin' ]

# Copied from EosImage/Makefile.am
extraPkgsVEosLab = [ 'Eos-veoslab', 'Eos-etba' ]

dpePackages = ImageVars.dpeFlavorRpms

# Add this function to check for SHIPPING setting when adding -future packages
# for CloudEos.
def cachedProjectSetting( settingName ):
   """Checks CachedProjectSettings.yaml (created at build time) and returns the
   value of the requested setting."""
   arosTestPluginDataPath = '/usr/share/ArosTestPluginData/SupportedHardware/'
   cachedProjectSettingsPath = arosTestPluginDataPath + 'CachedProjectSettings.yaml'
   try:
      with open( cachedProjectSettingsPath ) as f:
         cachedSettings = yaml.safe_load( f )
         return cachedSettings.get( settingName, None )
   except OSError as e:
      assert 0, "WARNING: error reading project setting '{}' from cache: " \
         "{}'".format( settingName, e )

cloudPackages = [
   "Baremetal-EosImage",
   "Baremetal-IntelDrivers",
   "CloudHa",
   "dpdk-tools",
   "KernelFirewall",
   "KernelFirewall-cli",
   "KernelFirewall-lib",
   "LicenseLoader-cloud",
   "LicenseLoader-azureCloudCerts",
   "Pathfinder-lib",
   "Pathfinder-cli",
   "Sfe-cloud",
   "WALinuxAgent",
]

# List of rpms that need to be included in the
# cEOS docker container.
cEOSPackages = [ "CEosUtils-etc" ]
cEOSLabPackages = [ ]

# Generate list of unused packages
# - 'platform' specifies a list of platforms
# - 'isWhiteList' specifies if the list of platforms is white-listed or black-listed
def excludePlatformPackages( excludePackagesDataPath, platforms, isWhiteList=True ):
   '''
   Generate a list of excluded packages of excluded platforms based on
   /etc/EosPkgExclude.d
   '''
   excludes = []
   for filename in os.listdir( excludePackagesDataPath ):
      if filename.endswith( '.exclude' ):
         match = re.search( platforms, filename )
         if ( ( isWhiteList and match ) or
              ( not isWhiteList and not match ) ):
            with open( os.path.join( excludePackagesDataPath, filename ) ) as f:
               excludes.append( set( f.read().splitlines() ) )
   if excludes:
      return list( set.intersection( *excludes ) )
   else:
      return []

def sharedExcludesDpe( excludePackagesDataPath, rootDir ):
   return ImageVars.oliveFirmware

def sharedExcludesCloud( excludePackagesDataPath, rootDir ):
   return excludePlatformPackages( excludePackagesDataPath,
                                   "Sfa|Sfe|Baremetal|Ragnar" )

def sharedExcludesSand( excludePackagesDataPath, rootDir ):
   return excludePlatformPackages( excludePackagesDataPath,
                                   "Viper|Quartz4O" )

def sharedExcludesVEosLab( excludePackagesDataPath, rootDir ):
   # We don't yet have a platform for vEOS-lab, so we cheat by using Sfe
   # and then removing the CloudEOS packages.
   sharedExcludes = excludePlatformPackages( excludePackagesDataPath, "Sfe" )
   sharedExcludes.extend( [ 'Sfe' ] )
   # Remove vEOS related RPMs
   sharedExcludes.extend( [ 'CloudHa', 'KernelFirewall', 'KernelFirewall-lib' ] )
   # Remove bess & Sfe RPMs
   sharedExcludes.extend( [ 'BessMgr', 'BessMgr-lib', 'BessMgr-cli',
                            'Sfe-lib', 'Sfe-cli', 'SfeModules-lib', 'bess' ] )
   return sharedExcludes


def getRpmsToExclude( rootDir, pkgsToExclude ):
   # Find out which RPMs are actually present in the swi ...
   cmdOut = Swi.runAndReturnOutput( [ "rpm", "--root=%s" % rootDir, "-qa",
                                      "--queryformat=%{NAME}\n" ],
                                      printStdout=False )
   swiRpms = set( cmdOut.splitlines() )
   # ... and only consider those for removal
   setDiff = set( pkgsToExclude ) - swiRpms
   if setDiff:
      dashLine = 71 * '-'
      print( dashLine )
      print( "Warning: there are RPMs we want to remove that are not in image" )
      pprint( setDiff )
      print( dashLine )
   return list( swiRpms.intersection( pkgsToExclude ) )

tempRepo = None
def getRepoList( opts ):
   try:
      repoArgs = createRepoArgs( opts.ordered_args )
   except AttributeError as _: # User did not provide enablerepo or disablerepo
      repoArgs = []
     
   hasEnableRepo = False
   hasDisableRepo = False
   for repoArg in repoArgs:
      if repoArg.op == RepoArgOp.ENABLE:
         hasEnableRepo = True
      elif repoArg.op == RepoArgOp.DISABLE:
         hasDisableRepo = True

   if opts.repo:
      # When a repository has been explicitly specified, we do not do
      # any of the default behavior.  Instead, we create a repo here,
      # and explicitly enable it, and also respect anything that was
      # passed on the command line, but default to disabling *.

      # (It'd be nice if we could do this without this deep knowledge
      # of "a4 yum", i.e., that it puts its repos in /etc/Ayum.repos.d/)
      global tempRepo
      if tempRepo is None:
         # pylint: disable-next=consider-using-with
         tempRepoFile = tempfile.NamedTemporaryFile( dir='/etc/Ayum.repos.d',
               prefix='flavor-', suffix='.repo', delete=False, mode="w" )
         tempRepo = os.path.basename( tempRepoFile.name ).replace( '.repo', '' )
         tempRepoFile.write( repoFileTemplate.format( repoName=tempRepo,
                                                      url=opts.repo ) )
         tempRepoFile.close()

      repoArgs = ( createRepoArgs(
         [ ( RepoArgOp.ENABLE, tempRepo ) ] ) + repoArgs )
      if not hasDisableRepo:
         repoArgs = ( createRepoArgs(
            [ ( RepoArgOp.DISABLE, "*" ) ] ) + repoArgs )

      return repoArgs

   # The default repository to be used in build environments is 'local'.
   # For AUTOTEST/STEST environments we should use the Abuild repository.
   # Also check for enablerepo & disablerepo args which can be used to
   # override these defaults
   if 'AUTOTEST' in os.environ or 'STEST' in os.environ or 'A4_MOCK' in os.environ:
      enablerepo = createRepoArgs(
            [ ( RepoArgOp.ENABLE, "Abuild" ) ] )
   else:
      enablerepo = createRepoArgs(
            [ ( RepoArgOp.ENABLE, "local" ) ] )

   if not hasEnableRepo:
      repoArgs = enablerepo + repoArgs

   return repoArgs

def flavorCommon( opts, extraPkgs, extraExclude, sharedExcludesFunc,
                  swiMaxHwEpoch, image, swiFlavor=None ):
   '''
   Generate a new flavor SWI from EOS.swi.  The extraPkgs parameter is used to
   specify a list of extra RPM packages that should be installed.  If it is None,
   then no extra packages are required.  If the swiMaxHwEpoch parameter is not None,
   then the SWI_MAX_HWEPOCH of EOS.swi is updated to the new value.
   '''
   if opts.extraPkgs:
      if extraPkgs is None:
         extraPkgs = opts.extraPkgs
      else:
         extraPkgs = extraPkgs + opts.extraPkgs
      extraPkgs = list( set( extraPkgs ) )

   excludePackagesPath = opts.excludePkgs
   tmpdir = tempfile.mkdtemp()
   print( f"Extracting {image} to {tmpdir}" )

   try:
      ret = Swi.extract.extract( image, Swi.SwiOptions( dirname=tmpdir,
                                                        use_existing=True ) )
   except CalledProcessError as e:
      Swi.run( [ 'sudo', 'rm', '-rf', tmpdir ] )
      raise Exception() from e

   assert ret == 0, 'swi extract failed'

   if SwimHelperLib.isSwimDir( tmpdir ):
      SwimFlatten.swimFlatten( tmpdir )

   rootDir = os.path.join( tmpdir, 'rootfs-%s.dir' % farch )
   if not opts.excludePkgs:
      # Default to using the /etc/EosPkgExclude.d files in the swi if caller
      # does not specify one
      print( "excludePkgs not specified, using the one in %s" % image )
      excludePackagesPath = os.path.join( rootDir, 'etc/EosPkgExclude.d' )

   sharedExcludes = sharedExcludesFunc( excludePackagesPath, rootDir )

   sharedRpmExcludes = getRpmsToExclude( rootDir, sharedExcludes )

   repoArgs = getRepoList( opts )

   if swiFlavor is None:
      swiFlavor = opts.swiFlavor

   try:
      assert sharedRpmExcludes, 'Exclusion function %s returned nothing ' \
            'to exclude' % ( sharedExcludesFunc.__name__, )
      rpmArgs = [ '--nodeps', '-e' ]
      rpmArgs.extend( sharedRpmExcludes )
      # Erase the packages that this flavor excludes
      eraseFunc = Swi.rpm.rpmInSwiFunc( *rpmArgs )
      # Update swi flavor in the version file, yumConfig,
      # enableRepos and disableRepos are not used if packages is passed
      # as None.
      updateFunc = Swi.update.swiUpdate( swiVariant=opts.swiVariant,
                                         swiFlavor=swiFlavor,
                                         packages=extraPkgs, yumConfig=None,
                                         repoArgs=repoArgs,
                                         swiMaxHwEpoch=swiMaxHwEpoch,
                                         extraExclude=extraExclude )
      print( "Erasing uneeded RPMs and updating version file" )
      Swi.inSwi( image, [ eraseFunc, updateFunc ], outputfile=opts.output,
                 extImageDir=tmpdir, updateVersion=False )
   finally:
      print( "Cleaning up", tmpdir )
      Swi.run( [ 'sudo', 'rm', '-rf', tmpdir ] )

def flavorWithCloud( opts, extraPkgs, extraExclude, swiMaxHwEpoch, image ):
   flavorCommon( opts, extraPkgs, extraExclude, sharedExcludesCloud,
                 swiMaxHwEpoch, image )

def flavorWithVEosLab( opts, extraPkgs, extraExclude, swiMaxHwEpoch, image ):
   # The "Normal" vEOS-lab swi is created with installrootfs with no flavor,
   # so use DEFAULT instead of vEOS-lab for the flavor in the version file.
   flavorCommon( opts, extraPkgs, extraExclude, sharedExcludesVEosLab,
                 swiMaxHwEpoch, image, swiFlavor='DEFAULT' )

def flavorWithDpe( opts, extraPkgs, extraExclude, swiMaxHwEpoch, image ):
   flavorCommon( opts, extraPkgs, extraExclude, sharedExcludesDpe,
                 swiMaxHwEpoch, image )

def cleanup( workdir ):
   print( "Cleaning up", workdir )
   Swi.run( [ 'sudo', 'rm', '-rf', workdir ] )

# Remove unused packages
# - 'platform' specifies a list of platforms
# - 'isWhiteList' specifies if the list of platforms is white-listed or black-listed
def removeUnusedPackages( image, tmpdir, rootFsDir, platforms, isWhiteList=True ):
   excludePackagesPath = os.path.join( rootFsDir, 'etc/EosPkgExclude.d' )
   if not os.path.exists( excludePackagesPath ):
      print( "No etc/EosPkgExclude.d in image %s" % image )
   else:
      pkgsToExclude = excludePlatformPackages( excludePackagesPath,
                                               platforms,
                                               isWhiteList )
      cEOSRpmExcludes = getRpmsToExclude( rootFsDir, pkgsToExclude )
      if cEOSRpmExcludes:
         rpmArgs = [ '--nodeps', '-e' ]
         rpmArgs.extend( cEOSRpmExcludes )
         print( "Excluding packages:" )
         print( cEOSRpmExcludes )
         eraseFunc = Swi.rpm.rpmInSwiFunc( *rpmArgs )
         eraseFunc( rootFsDir )

# Utility function to modify the toggle value within a toggle file with multiple
# toggle definitions. This might result in a modified toggle file compared to the
# source toggle file due to indentation/spacing differences.
def modifyToggleInRootFs( tmpDir, rootFsDir, toggleFile, toggleName, toggleValue ):
   assert toggleValue in ( True, False )
   toggleFilePath = f"{rootFsDir}/etc/toggles.d/{toggleFile}"
   tmpToggleFilePath = f"{tmpDir}/{toggleFile}"
   toggleData = None
   with open( toggleFilePath ) as toggleFh:
      toggleData = json.load( toggleFh )
   toggleFound = False
   for toggleVal in toggleData:
      if toggleVal[ 'name' ] == toggleName:
         toggleFound = True
         toggleVal[ 'enabled' ] = toggleValue
         break
   assert toggleFound
   # Write the modified toggle file to a temporary file.
   with open( tmpToggleFilePath, "w" ) as toggleFh:
      json.dump( toggleData, toggleFh, indent=3 )
   # move the temporary file into /etc/toggles.d in the rootfs, which requires
   # root privileges.
   mvToggleFileToRootFs = [ "sudo", "mv", tmpToggleFilePath, toggleFilePath ]
   Swi.runAndReturnOutput( mvToggleFileToRootFs )

def flavorWithContainerBasedImage( opts, pkgList, outputFile, swiFlavor, image,
                                   rootFsFunc=None, eraseFunc=None ):
   tmpdir = tempfile.mkdtemp()

   try:
      Swi.extract.extract( image, Swi.SwiOptions( dirname=tmpdir,
                                                  use_existing=True ) )
   except CalledProcessError as e:
      Swi.run( [ 'sudo', 'rm', '-rf', tmpdir ] )
      raise Exception() from e

   if SwimHelperLib.isSwimDir( tmpdir ):
      SwimFlatten.swimFlatten( tmpdir )

   rootFsDir = f"{tmpdir}/rootfs-{farch}.dir"

   repoArgs = getRepoList( opts )

   updateFunc = Swi.update.swiUpdate( swiVariant=None, swiFlavor=swiFlavor,
                                      packages=pkgList, yumConfig=None,
                                      repoArgs=repoArgs )
   if eraseFunc:
      eraseFunc( rootFsDir )
   updateFunc( rootFsDir )

   # 'version' file has been updated with swiFlavor during updateFunc(),
   # so now we copy it to /etc/swi-version
   sourceVersion = os.path.join( tmpdir, 'version' )
   targetVersion = os.path.join( rootFsDir, 'etc', 'swi-version' )
   Swi.run( [ 'sudo', 'mv', sourceVersion, targetVersion ] )

   print( f" Removing core file setting in {swiFlavor}. The OS is responsible for "
          "this." )
   for fname in ( "etc/sysctl.d/99-eos.conf", "usr/lib/sysctl.d/50-coredump.conf" ):
      systemConfFileName = os.path.join( rootFsDir, fname )
      if os.path.exists( systemConfFileName):
         Swi.run( [ 'sudo', 'sed', '-i', '/kernel.core/d', systemConfFileName ] )

   systemUnitFiles = [
         # Udevd is not supported currently by even latest release of systemd for
         # containers. Disabling udevd.
         "systemd-udevd-kernel.socket",
         "systemd-udevd-control.socket",
         "systemd-udev-trigger.service",
         "systemd-udev-settle.service",
         "systemd-udevd.service",

         # Disable Login and agetty services
         "console-getty.service",
         "getty.target",
         ]
   for sFile in systemUnitFiles:
      filePath = os.path.join( rootFsDir, "lib/systemd/system", sFile )
      Swi.run( [ 'sudo', 'rm', filePath ] )

   # Disable ZTP
   with tempfile.NamedTemporaryFile( mode="w" ) as cfg:
      cfg.write( 'DISABLE=True\n' )
      cfg.flush()
      flashDir = os.path.join( rootFsDir, 'mnt', 'flash' )
      Swi.run( [ 'sudo', 'mkdir', '-p', '%s' % flashDir ] )
      Swi.run( [ 'sudo', 'cp', cfg.name,
         os.path.join( flashDir, 'zerotouch-config' ) ] )

   if rootFsFunc:
      rootFsFunc( tmpdir, rootFsDir )

   try:
      etcCeosPath = "%s/etc/cEOS-release" % rootFsDir
      # cEOS-release is part of the CEosUtils-etc package; the rest of
      # CEosUtils-etc is only needed on the deprecated CEos-on-hardware.
      # If it does not exist, then simply create an empty file.
      if not os.path.exists( etcCeosPath ):
         Swi.run( [ 'sudo', 'touch', etcCeosPath ] )
      # Iptables are for temporary hack to get ssh working.
      # Please refer to BUG188760.
      # Also disabling reboot commands inside the container, since semantics don't
      # really make sense.
      filesToDelete = [ 'lib/systemd/system/iptables.service',
                        'lib/systemd/system/ip6tables.service',
                        'usr/sbin/reboot',
                        'usr/sbin/poweroff',
                        'usr/sbin/shutdown',
                        'usr/sbin/halt' ]
      for fileToDelete in filesToDelete:
         Swi.run( [ 'sudo', 'rm', os.path.join( rootFsDir, fileToDelete ) ] )

      # pylint: disable-next=consider-using-with
      tarFile = tempfile.NamedTemporaryFile().name
      Swi.repackContainerImage( rootFsDir, outputFile, tarFile )
      Swi.run( [ 'sudo', 'rm', tarFile ] )
   except AssertionError:
      # Debug info for BUG239880 before cleanup
      # Are we out of space in our Autotest workspaces?
      _outputDf = Swi.runAndReturnOutput( [ 'df', '-h' ] )
      _outputDu = Swi.runAndReturnOutput( [ 'sudo', 'du', '-h', '/tmp' ] )
      raise
   finally:
      cleanup( tmpdir )

def flavorWithCEOS( opts, image, extraPackages=None ):
   extraPackages = [] if extraPackages is None else extraPackages
   outputFile = opts.output
   if not outputFile:
      outputFile = '/images/cEOS.tar.xz'

   if opts.swiFlavor in ( 'cEOS', 'cEOSAristaSAI' ):
      pkgList = cEOSPackages + extraPackages
   else:
      pkgList = cEOSLabPackages + extraPackages

   def rootFsFunc( tmpdir, rootFsDir ):
      if opts.swiFlavor == 'cEOSAristaSAI':
         # We need to enable the AristaSAI toggle just for this image
         modifyToggleInRootFs( tmpdir, rootFsDir, "AristaSAI.json",
               "AristaSAIMode", True )
         # We need to disable TH450GReleaseOnShut toggle so that
         # 1. Logical ports gets allocated to /1 interfaces by default
         # 2. Logical ports doesn't get freed when a port is shutdown
         # This helps in creating required hostif's during bootup by Strata
         # duts go past PortInitDone stage. It also allows us to unshut intf fron
         # Sonic.
         modifyToggleInRootFs( tmpdir, rootFsDir, "AlePhy.json",
               "TH450GReleaseOnShut", False )

         # Temporarily disable the L1Subdomain toggle. Changes need to be made in SAI
         # Fru plugin to provide xcvr status info for Strata to create interfaces
         # BUG745341
         modifyToggleInRootFs( tmpdir, rootFsDir, "AlePhy.json",
               "L1TopoEthPhySmSubdomainSupport", False )
         # Enables SpeedGroupAuto to automatically switch PLL speed groups
         modifyToggleInRootFs( tmpdir, rootFsDir, "AlePhy.json",
               "SAISpeedGroupAuto", True )

         # Remove disable ipv6 sysctl setting. This should be controlled by Sonic.
         removeDisableIpv6Setting = [ "sudo", "sed", "-i",
                                      "/Disable ipv6/d;/disable_ipv6/d",
                                      "%s/etc/sysctl.d/99-eos.conf" % rootFsDir ]
         Swi.runAndReturnOutput( removeDisableIpv6Setting )

         # We only care about BlackHawk, Lodoga, Marysville and Smartsville
         removeUnusedPackages( image, tmpdir, rootFsDir,
                               "Lodoga|Blackhawk|Marysville|Smartsville|Catalina" )

      if opts.swiFlavor == 'cEOSDsfSim':
         # Enable LyonsvilleDsfClusterMode toggle in the container image.
         modifyToggleInRootFs( tmpdir, rootFsDir, "Sand.json",
               "LyonsvilleDsfClusterMode", True )
         # Enable DsfPhase1 toggle for DsfAgent runnability.
         modifyToggleInRootFs( tmpdir, rootFsDir, "Tunnel.json", "DsfPhase1", True )
         # Enable Sand platform changes for DSF cluster support.
         modifyToggleInRootFs( tmpdir, rootFsDir, "Sand.json",
               "SandDsfClusterMode", True )
         modifyToggleInRootFs( tmpdir, rootFsDir, "Sand.json",
               "ArmOrderedSwTable", True )
         # The DSF simulation currently only runs on Lyonsville duts, whitelist the
         # Lyonsville SKU package.
         removeUnusedPackages( image, tmpdir, rootFsDir, "Lyonsville" )

   flavorWithContainerBasedImage( opts, pkgList, outputFile, 'cEOS', image,
                                  rootFsFunc=rootFsFunc )

def flavorWithCEOSSingleHW( opts, image ):
   outputFile = opts.output
   if not outputFile:
      outputFile = image
   pwd = os.getcwd()
   rootFsDir = "rootfs-%s.dir" % farch
   Swi.run( [ 'sudo', 'mkdir', '-p', rootFsDir ] )
   Swi.run( [ 'sudo', 'tar', '-Jxf', image , '-C', rootFsDir,
              '--warning=no-timestamp' ] )
   if not opts.keep:
      Swi.run( [ 'sudo', 'rm', '-f', image ] )
   rootFsDir = os.path.abspath( rootFsDir )

   removeUnusedPackages( image, pwd, rootFsDir, opts.hw, True )
   Swi.repackContainerImage( rootFsDir, outputFile, outputFile + '.tmp' )
   cleanup( rootFsDir )
   Swi.runAndReturnOutput( [ 'sudo', 'rm', '-f', outputFile + '.tmp' ] )

def flavorWithCEOSLab( opts, image ):
   outputFile = opts.output
   if not outputFile:
      opts.output = '/images/cEOS-lab.tar.xz'
   flavorWithCEOS( opts, image, extraPackages=[ 'Eos-ceoslab' ] )

def flavorWithCEOSR( opts, image ):
   outputFile = opts.output
   if not outputFile:
      opts.output = '/images/cEOSR.tar.xz'
   flavorWithCEOS( opts, image )

def flavorWithCEOSCloudLab( opts, image ):
   outputFile = opts.output
   if not outputFile:
      opts.output = '/images/cEOS-cloudlab.tar.xz'
   flavorWithCEOS( opts, image, extraPackages=[ 'Eos-ceoslab',
                                                'Bfd-ceoslab',
                                                'EosLabInit-ceoslab',
                                                'SfeShared-cloudlab' ] )

def flavorWithSandLab( opts, image ):
   outputFile = opts.output
   if not outputFile:
      outputFile = '/images/SandLab.tar.xz'

   # Including SandCmodelDut will implicitly include other required RPMs
   # EosLabInit-ceoslab modifies several systemd services
   pkgList = [ 'SandCmodelDut', 'EosLabInit-ceoslab' ]

   def eraseFunc( rootFsDir ):
      excludePackagesPath = os.path.join( rootFsDir, 'etc/EosPkgExclude.d' )
      sharedExcludesFunc = sharedExcludesSand
      sharedExcludes = sharedExcludesFunc( excludePackagesPath, rootFsDir )
      sharedRpmExcludes = getRpmsToExclude( rootFsDir, sharedExcludes )
      assert sharedRpmExcludes, 'Exclusion function %s returned nothing ' \
            'to exclude' % ( sharedExcludesFunc.__name__, )
      rpmArgs = [ '--nodeps', '-e' ]
      rpmArgs.extend( sharedRpmExcludes )
      # Erase the packages that this flavor excludes
      swiEraseFunc = Swi.rpm.rpmInSwiFunc( *rpmArgs )
      swiEraseFunc( rootFsDir )

   flavorWithContainerBasedImage( opts, pkgList, outputFile, 'SandLab', image,
                                  eraseFunc=eraseFunc )

def flavorWithExtraPkgs( opts, extraPkgs, extraExclude, swiMaxHwEpoch, image ):
   '''
   Generate a new flavor SWI from EOS.swi.  The extraPkgs parameter is used to
   specify a list of extra RPM packages that should be installed.  If it is None,
   then no extra packages are required.  If the swiMaxHwEpoch parameter is not None,
   then the SWI_MAX_HWEPOCH of EOS.swi is updated to the new value.
   '''

   repoArgs = getRepoList( opts )

   updateFunc = Swi.update.swiUpdate( swiVariant=opts.swiVariant,
                                      swiFlavor=opts.swiFlavor,
                                      packages=extraPkgs, yumConfig=None,
                                      repoArgs=repoArgs,
                                      swiMaxHwEpoch=swiMaxHwEpoch,
                                      extraExclude=extraExclude )

   print( "Updating image, %s, with flavor=%s, swiVariant=%s" %
      ( image, opts.swiFlavor, opts.swiVariant ) )
   isSwim = SwimHelperLib.isSwimImage( image )
   Swi.inSwi( image, [ updateFunc ], outputfile=opts.output,
              flattenSwim=isSwim, updateVersion=False )

def flavorHandler( args=None ):
   args = sys.argv[ 1: ] if args is None else args
   op = argparse.ArgumentParser(
      prog="swi flavor",
      description="Create a flavor of EOS from full EOS.swi",
      usage="usage: %(prog)s EOS.swi [--with swiflavor] [-o target] " +
            "[--swiVariant International] [--hw product] [--excludePkgs path] " )

   op.add_argument( '--with',
                    help='desired flavor',
                    dest='swiFlavor' )
   op.add_argument( "--enablerepo", action=AddEnableDisableReposInOrder,
                    help="enable repository (may be specified multiple times)" )
   op.add_argument( "--disablerepo", action=AddEnableDisableReposInOrder,
                    help="disable repository (may be specified multiple times)" )
   op.add_argument( "--repo", action="store",
                    help="specify URL to repository to use (disables default "
                         "repository list)" )
   op.add_argument( '-o', '--output',
                    help="target swi for swiFlavor EOS",
                    action='store' )
   op.add_argument( '--swiVariant', action='store',
                    help="Update EOS SWI variant" )
   op.add_argument( '--hw', action='store',
                    help="product for which cEOS image size should be optimized" )
   op.add_argument( '--excludePkgs',
                    help="path to dir containing package exclude info, "
                         "default uses /etc/EosPkgExclude.d/ within EOS.swi",
                    action='store' )
   op.add_argument( '--extraPkgs', action='append',
                    help='extra RPM packages that should be installed '
                         '(may be specified multiple times)' )
   op.add_argument( '-k', '--keep',
                    help="keep the original cEOS image",
                    action='store_true' )
   op.add_argument( '-m', '--outputMetadata', action='store',
                    help="write metadata for installed packages to an output "
                         "file for derived flavors like DPE-CTNR" )

   opts, args = op.parse_known_args( args )
   if len( args ) != 1:
      op.error( 'Please give me exactly one swi file!' )
   Swi.extract.setTempDirIfNeeded()
   extraExclude = None
   if opts.swiFlavor is None:
      assert opts.swiVariant == 'International'
      flavorWithExtraPkgs( opts, internationalPackages,
                           extraExclude, None, *args )
   elif opts.swiFlavor == swiFlavorCloud:
      # Include BareMetal RPMs in the CloudEOS.swi
      extraPkgs = cloudPackages
      extraExclude = None
      if opts.swiVariant == 'International':
         extraPkgs += internationalPackages
      flavorWithCloud( opts, extraPkgs, extraExclude, None, *args )
   elif opts.swiFlavor == swiFlavorDPE:
      assert opts.swiVariant != 'International'
      flavorWithDpe( opts, dpePackages, None, None, *args )
   elif opts.swiFlavor == swiFlavorPdp:
      assert opts.swiVariant != 'International'
      flavorWithExtraPkgs( opts, None, None, None, *args )
   elif opts.swiFlavor == swiFlavorPdp + '-' + swiFlavorDPE:
      assert opts.swiVariant != 'International'
      flavorWithDpe( opts, dpePackages, None, None, *args )
   elif opts.swiFlavor == swiFlavorDPEContainer:
      assert opts.swiVariant != 'International'
      image = args[ 0 ]
      assert image.endswith( "EOS-DPE.swi" ), (
            "flavoring as %s only supported with EOS-DPE.swi" % 
            swiFlavorDPEContainer )
      repoArgs = getRepoList( opts )
      flavorSwimWithExtraPkgs( image, ImageVars.SwimOptimizations.Dpe,
                               swiFlavorDPEContainer, extraPkgsContainer,
                               repoArgs, opts.output, opts.outputMetadata )
   elif opts.swiFlavor == 'vEOS-lab':
      # Until we make sure that we delete all of the packages
      # that we don't want to distribute.  For now this feature
      # is just to generate images to run in autotest.
      print( '-------------------------------------------------------' )
      print( 'Warning: images generated this way are only for testing' )
      print( '         and should not be distributed to customers' )
      print( '-------------------------------------------------------' )
      flavorWithVEosLab( opts, extraPkgsVEosLab, None, None, *args )
   elif opts.swiFlavor == 'cEOS':
      flavorWithCEOS( opts, *args )
   elif opts.swiFlavor == 'cEOSLab':
      flavorWithCEOSLab( opts, *args )
   elif opts.swiFlavor == 'cEOSR':
      flavorWithCEOSR( opts, *args )
   elif opts.swiFlavor == 'cEOSCloudLab':
      flavorWithCEOSCloudLab( opts, *args )
   elif opts.swiFlavor == 'cEOSSingleHW':
      if not opts.output and opts.keep:
         op.error( 'Cannot use --keep without --output' )
      flavorWithCEOSSingleHW( opts, *args )
   elif opts.swiFlavor == 'cEOSAristaSAI':
      flavorWithCEOS( opts, *args,
                      extraPackages=[ "AristaSAI-lib",
                                      "AristaSAI",
                                      "sonic-sairedis" ] )
   elif opts.swiFlavor == 'cEOSDsfSim':
      flavorWithCEOS( opts, *args )
   elif opts.swiFlavor == 'SandLab':
      flavorWithSandLab( opts, *args )
   else:
      op.error( 'Invalid flavor!' )

if __name__ == "__main__":
   flavorHandler()
