#!/usr/bin/env python3
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Swi
import Swi.flavor
import SwimHelperLib
import argparse
import os
import re
import subprocess
import sys

def stripSwi( image, output, products, rpms, verbose=False, opts=None ):
   input_bytes = os.stat( image ).st_size if os.path.exists( image ) else 0
   def deleteRpms( rootDir ):
      excludePackagesPath = os.path.join( rootDir, 'etc/EosPkgExclude.d' )

      rpmExcludes = []
      productRe = "^$"
      if products:
         productRe = "|".join( products )
         excludes = []
         for f in os.listdir( excludePackagesPath ):
            if f.endswith( ".exclude" ):
               if re.search( productRe, f ):
                  # Matching list, read its rpms
                  print( "Matched " + f.split( '.exclude' )[ 0 ] )
                  with open( os.path.join( excludePackagesPath, f ) ) as fr:
                     excludes.append( set( fr.read().splitlines() ) )

         if not excludes:
            print( "Found no matching products supported in this swi. Aborting" )
            sys.exit( 1 )
         rpmExcludes += list( set.intersection( *excludes ) )

      if rpms:
         rpmExcludes += rpms

      for f in os.listdir( excludePackagesPath ):
         if f.endswith( ".exclude" ) and not re.search( productRe, f ):
            # Non-matching list. If rpmExcludes is not a subset of the
            # RPMs in this file, we delete it to indicate this
            # platform is not supported
            with open( os.path.join( excludePackagesPath, f ) ) as fr:
               fileRpms = set( fr.read().splitlines() )
            if not set( rpmExcludes ).issubset( fileRpms ):
               Swi.run( [ "sudo", "rm", os.path.join( excludePackagesPath, f ) ] )

      # filter for rpms that actually exist in the image
      rpmExcludes = Swi.flavor.getRpmsToExclude( rootDir, rpmExcludes )
      if not rpmExcludes and not opts.sh:
         print( "Output swi would be identical to input swi. Aborting" )
         sys.exit( 1 )

      print( "Removing " + str( len( rpmExcludes ) ) + " rpms" )
      if verbose:
         print( '\n'.join( sorted( rpmExcludes ) ) )
      # pylint: disable-next=consider-using-f-string
      Swi.run( [ "sudo", "rpm", "--root=%s" % rootDir, "--nodeps", "-e" ] +
               rpmExcludes )

   def runShellCommands( rootDir ):
      if not opts.sh:
         return
      for shCmd in opts.sh:
         shCmd = shCmd.replace( "ROOTDIR", rootDir )
         if verbose:
            print( "Running:", shCmd )
         subprocess.check_call( shCmd, shell=True )

   # SWIM images must be flattened into a single rootfs before we can
   # strip the SWI
   isSwim = SwimHelperLib.isSwimImage( image )
   Swi.inSwi( image,
              [ deleteRpms, runShellCommands ],
              outputfile=output,
              flattenSwim=isSwim,
              readOnly=opts and opts.readOnly,
              fast=opts and opts.fast,
              zstd=opts and opts.zstd )
   output_bytes = os.stat( output ).st_size if output else os.stat( image ).st_size
   input_mbytes = int( round( input_bytes / 1024 / 1024 ) )
   output_mbytes = int( round( output_bytes / 1024 / 1024 ) )
   # pylint: disable-next=consider-using-f-string
   print( "Reduced swi size from %sMB to %sMB" %
          ( str( input_mbytes ), str( output_mbytes ) ) )

def stripHandler( args=None ):
   if args is None:
      args = sys.argv[ 1: ]

   op = argparse.ArgumentParser(
      prog="swi strip",
      description="Create a stripped swi from a full swi that only supports "
                  "specified hardware",
      usage="%(prog)s EOS.swi [-o target] [-p product] [--rpm rpm]" )
   op.add_argument( '-p', '--product',
                    help="product to support in the stripped swi"
                    " (may be specified multiple times)",
                    action='append' )
   op.add_argument( '--rpm',
                    help="rpm to remove from the stripped swi"
                    " (may be specified multiple times)",
                    action='append' )
   op.add_argument( '-v', '--verbose',
                    help="print all rpms removed from the stripped swi",
                    action='store_true' )
   op.add_argument( '--sh',
                    help="shell commands to run after removal of RPMs, use"
                    " ROOTDIR to refer to the squashfs root"
                    " (may be specified multiple times)",
                    action='append' )
   Swi.rpm.addRpmOptions( op )

   opts, args = op.parse_known_args( args )

   if len( args ) != 1:
      op.error( "Please provide exactly one input swi" )
   if not opts.product and not opts.rpm and not opts.sh:
      op.error( "Please specify the product(s) you wish to support with "
                "'-p product' or rpm(s) to remove with '--rpm rpm' or "
                "shell commands to run with '--sh sh'" )

   stripSwi( args[ 0 ], opts.file, opts.product, opts.rpm, verbose=opts.verbose,
             opts=opts )

if __name__ == "__main__":
   stripHandler()
