#!/usr/bin/env python3
# Copyright (c) 2008, 2009, 2016, 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import sys
import argparse
import Swi
from Swi.installrootfs import RepoArgOp
from SwimHelperLib import onSupportedKernel, isSwimImage

def swiUpdate( swiVariant=None, swiFlavor=None, packages=None, yumConfig=None,
               repoArgs=None, swiMaxHwEpoch=None,
               targets=None, extraExclude=None, zipRoot=False ):
   if extraExclude is None:
      extraExclude = {}
   def updateWithinExtractedSwi( rootDir ):

      def updateSwiConfig( label, config ):
         printData = { "config": config,
                       "label": label }
         print( "Updating %(label)s=%(config)s" % printData )
         Swi.run( [ 'sed', '-i',
                    's/%(label)s=.*$/%(label)s=%(config)s/g' % printData,
                    '%s/../version' % rootDir ] )

      if swiVariant:
         updateSwiConfig( "SWI_VARIANT", swiVariant )
      if swiFlavor:
         updateSwiConfig( "SWI_FLAVOR", swiFlavor )
      if swiMaxHwEpoch:
         updateSwiConfig( "SWI_MAX_HWEPOCH", swiMaxHwEpoch )

      if packages:
         # yum install packages into rootfs
         cmd = [ "sudo" ]
         if not yumConfig:
            cmd += [ "env", "A4_YUM_NATIVE_ARCH=1", "a4", "yum" ]
         else:
            cmd += [ "yum", "-c", yumConfig ]
         for repoArg in repoArgs:
            cmd += [ f"--{ repoArg.op.value }repo={ repoArg.repo }" ]
         Swi.run( cmd + [ "-y", "--installroot=%s" % rootDir, "install" ] +
                  packages )
         # Make sure packages were really installed
         Swi.run( [ "rpm", "--root=%s" % rootDir, "-q" ] + packages )
         # Clean up yum cache
         Swi.run( [ "sudo", "sh", "-c", "rm -rf %s/var/cache/yum/*" % rootDir ] )
         # Clean up dnf cache
         Swi.run( [ "sudo", "sh", "-c", "rm -rf %s/var/cache/dnf/*" % rootDir ] )

      for originalPackage, extraPackages in extraExclude.items():
         newPackages = '\\n'.join( [ originalPackage ] + extraPackages )
         sedStr = f"s/{originalPackage}$/{newPackages}/g"
         path = '%s/etc/EosPkgExclude.d/*.exclude' % rootDir
         Swi.run( [ 'sudo', 'sed', '-i', sedStr, path ] )

      if targets:
         for f in targets:
            source, dest = f 
            baseDir = ( rootDir + "/.." ) if zipRoot else rootDir
            Swi.run( [ "sudo", "cp", "-ar", source,
                       f"{baseDir}/{dest}" ] )
   return updateWithinExtractedSwi

def updateHandler( args=None ):
   args = sys.argv[1:] if args is None else args
   parser = argparse.ArgumentParser( description="Updates files or attributes of a"
                                                 " swi" )

   class AddInOrder( argparse.Action ):
      """Adds arguments in ordered fashion. This is needed for saving the
      enable/disable repo arguments order"""

      def __call__( self, parser, namespace, values, option_string=None ):
         if not 'ordered_args' in namespace:
            setattr( namespace, 'ordered_args', [] )
         previous = namespace.ordered_args
         if self.dest == "enablerepo":
            previous.append( ( RepoArgOp.ENABLE, values ) )
         elif self.dest == "disablerepo":
            previous.append( ( RepoArgOp.DISABLE, values ) )
         else:
            raise Exception( f"Unexpected argument: { self.dest }" )

         setattr( namespace, 'ordered_args', previous )

   parser.add_argument( "image", type=str, nargs=1, help="EOS.swi image to update" )
   parser.add_argument( "--swiVariant", help="Update EOS SWI variant" )
   parser.add_argument( '--swiFlavor', help="Update EOS SWI flavor" )
   parser.add_argument( "--swiMaxHwEpoch", help="Update EOS SWI max HW epoch" )
   parser.add_argument( "--yum-config", "-c", metavar="CONFIG",
                        help="use yum configuration file CONFIG" )
   parser.add_argument( "--enablerepo", action=AddInOrder,
                        help="enable repository (may be specified multiple times)" )
   parser.add_argument( "--disablerepo", action=AddInOrder,
                        help="disable repository (may be specified multiple times)" )
   parser.add_argument( "--zipRoot", action="store_true",
                        help="Update file(s) in zip root instead of rootfs" )
   parser.add_argument( "--overrideRpmOp", action="store_true",
                       help="Disables the rpmOp optimization by unsquashing "
                            "everything in the SWI" )
   parser.add_argument( "--forceSign",
                        help="Force updating the swi signature after update "
                             "commands run",
                        action="store_true" )
   parser.add_argument( '--skipVersionUpdate', action='store_true',
                        help='skip refresh build date & serial number in all of the '
                             'version files' )
   group = parser.add_mutually_exclusive_group()
   group.add_argument( "--target", "-t", type=str, nargs=2, action="append",
                        help="SOURCE and DESTINATION of a file or directory (may be"
                             " specified multiple times" )
   group.add_argument( "--package", "-p", action="append",
                     help="install PACKAGE (may be specified multiple times)" )
   args = parser.parse_args( args )

   # Provide defaults for enablerepo and disablerepo, don't use
   # op.set_defaults(), as then anything provided by the user gets
   # appended to the defaults, the user specified ones does not
   # replace the defaults.
   if args.enablerepo or args.disablerepo:
      repoArgs = Swi.installrootfs.createRepoArgs( args.ordered_args )
   else:
      repoArgs = Swi.installrootfs.createRepoArgs(
            [ ( RepoArgOp.ENABLE, "local" ) ] )

   # The zipRoot option does not work with specifying a package.  There isn't
   # an easy way to express this with argparse, so check that here:
   if args.zipRoot and args.package:
      raise ValueError( "--zipRoot can not be used with --package" )

   if isSwimImage( args.image[ 0 ] ) and not onSupportedKernel():
      print( "WARNING: Cannot overlay mount image on unsupported kernel 3.18" )
      print( "Attempting `swi update` command anyways" )

   rpmOp = args.package and not args.overrideRpmOp

   swiUpdateFn = swiUpdate( swiVariant=args.swiVariant,
                            swiFlavor=args.swiFlavor,
                            packages=args.package,
                            yumConfig=args.yum_config,
                            repoArgs=repoArgs,
                            swiMaxHwEpoch=args.swiMaxHwEpoch,
                            targets=args.target,
                            zipRoot=args.zipRoot )

   # Disable fastInSwi until BUG760149/BUG765159 are resolved
   #
   # if ( rpmOp and isSwimImage( args.image[ 0 ] ) ):
   #    Swi.fastInSwi( args.image[ 0 ], [ swiUpdateFn ], fast=False,
   #                   forceSign=args.forceSign, zipRoot=args.zipRoot,
   #                   targets=args.target,
   #                   updateVersion=not args.skipVersionUpdate )
   # else:

   Swi.inSwi( args.image[ 0 ], [ swiUpdateFn ], fast=False,
              rpmOp=rpmOp, forceSign=args.forceSign,
              updateVersion=not args.skipVersionUpdate )

if __name__ == "__main__":
   updateHandler()
