#!/usr/bin/env python3
# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-with
# pylint: disable=consider-using-f-string

import argparse
import collections
import errno
import glob
import operator
import os
import re
import shutil
import stat
import subprocess
import sys
import tempfile

from six.moves import urllib

from Swi.extract import extract as swiExtract, setTempDirIfNeeded
from Swi.chroot import mountedIs as swiMountedIs
import Swi
import SwimHelperLib

NOREPO = "None"
PRINT_PREFIX = "[swi workspace]"
useDnf = False

def cmdPrint( s ):
   """
   Add a prefix to each informational print from the swi workspace command. This
   differentiates the informational print from other debug output.
   """
   print( PRINT_PREFIX, s )

# BUG388437
# We have copied getRepoFromLocation from A4.rpmrepo to avoid an
# import dependency.  When we move A4.rpmrepo to a new location,
# we should go back to importing.
def getRepoFromLocation( locationFmt, candidateRepos ):
   '''Picks the first repo from the candidateRepos list that is valid from the
   location specified in locationFmt.'''
   if candidateRepos:
      for candidateRepo in candidateRepos:
         repoURL = locationFmt % candidateRepo
         try:
            # Determine whether or not a URL specifies a yum repository
            # (looks for repoURL/repoData/repomd.xml)
            repoMetadataURL = "%s/repodata/repomd.xml" % repoURL
            metadataObj = urllib.request.urlopen( repoMetadataURL )
            metadataObj.close()
            return repoURL
         except urllib.error.URLError as e:
            cmdPrint( f"Repo meta-data for repository {repoURL}: {e}." )
   return None
# end BUG388437

class SwiException( Exception ):
   '''Exceptions of this class may contain extra (error) data to print.'''
   def __init__( self, msg, data=None ):
      super().__init__( msg )
      self.data = data

class Package:
   '''Encapsulates a package to install.'''
   def __init__( self, name, commandToVerifyInstall, lookupPkg=False ):
      '''lookupPkg: whether or not to run dnf/yum whatprovides to look up the package
      that provides the program denoted by "name".'''
      if lookupPkg:
         cmd = "dnf" if useDnf else "yum"
         rpmData = run( [ cmd, "whatprovides", "*/%s" % name ],
                        captureStdout=True ).splitlines()
         rpm = None
         for line in rpmData:
            match = re.search( r"\.i686\s+:\s+", line )
            # -devel rpms may show up which we don't want.
            if match and "-devel" not in line:
               rpm = re.match( "[^-]+", line ).group()
               break
         if rpm:
            self.name_ = rpm
         else:
            cmdPrint( "Package not found for program '%s'" % name )
            self.name_ = name
      else:
         # denotes the name of the package
         self.name_ = name
      # the program the package installs, to verify that the package was installed
      self.commandToVerifyInstall_ = commandToVerifyInstall
      self.output_ = None

   def install( self ):
      '''Install the given package.'''
      self.output_ = None
      cmdPrint( "Installing %s" % self.name_ )
      try:
         cmd = "dnf" if useDnf else "yum"
         self.output_ = run( [ cmd, "-y", "install", self.name_ ],
                             captureStderr=True )
      except SwiException as e:
         self.output_ = e.data
   def installed( self ):
      '''Returns whether or not installation was successful.'''
      print( f"{PRINT_PREFIX} Verifying install of {self.name_}", end=' ' )
      if self.commandToVerifyInstall_ is not None:
         try:
            run( self.commandToVerifyInstall_, captureStderr=True )
         except SwiException:
            print( "... not installed" )
            return False
      print( "... installed" )
      return True

   def output( self ):
      return self.output_

   def __str__( self ):
      return self.name_
   def __repr__( self ):
      return self.name_

class ArtoolsPackage( Package ):
   '''Encapsulates an Artools package, which requires special treatment to
   not install everything. The only debugging tool we want is a4 debug, so
   only extract the necessary files for it.'''
   def __init__( self ):
      super().__init__( "Artools", [ "which", "a4" ] )

   def generateA4Code( self ):
      a4Code = '''#!/usr/bin/env python3
import sys
if len( sys.argv ) >= 2 and sys.argv[ 1 ] == "debug":
   import A4.debug
   A4.debug.debugHandler( args=sys.argv[ 2: ] )
else:
   print(  "this a4 command only supports a4 debug." )
   sys.exit( 1 )
'''
      a4Filename = "/usr/bin/a4"
      with open( a4Filename, "w" ) as a4File:
         a4File.write( a4Code )
      os.chmod( a4Filename, stat.S_IXUSR | stat.S_IXGRP | stat.S_IXOTH )

   def install( self ):
      output = []
      destDir = "/tmp"
      try:
         if useDnf:
            run( [ "dnf", "download", "-h" ], captureStderr=True )
         else:
            run( [ "which", "yumdownloader" ], captureStderr=True )

      except SwiException as e:
         output.append( "dnf/yum download needed: " + str( e ) )
      else:
         success = False
         for bootstrapRpm in [ False, True ]:
            rpmName = self.name_ + ( "-bootstrap" if bootstrapRpm else "" )
            try:
               if useDnf:
                  cmd = [ 'dnf', 'download' ]
               else:
                  cmd = [ 'yumdownloader' ]
               cmd += [ '--destdir', destDir, rpmName ]
               output.append( run( cmd, captureStdout=True, captureStderr=True ) )
            except SwiException as e:
               # Continue best we can if Artools rpm can't be found.
               output.append( "%s Unable to download Artools rpm:" % PRINT_PREFIX )
               output.append( e.data )
               continue

            rpmGlob = glob.glob( f"/{destDir}/{rpmName}*" )
            if rpmGlob:
               success = True
               rpm = rpmGlob[ 0 ]
               output.append( "Running rpm2cpio:" )
               filesToExtract = [ "*/A4/DebugLib.py",
                                  "*/A4/Helpers.py",
                                  "*/A4/debug.py",
                                  "*/A4/__init__.py",
                                  "*/A4/rpmrepo.py" ]
               output.append( run( [ "bash", "-c", "rpm2cpio %s | cpio -ivd %s" %
                                     ( rpm, " ".join( filesToExtract ) ) ] ) )
         if success:
            self.generateA4Code()
      self.output_ = "\n".join( output )

def yumRepoPath( root ):
   '''Given a root directory, returns the path to the yum repositories.'''
   return os.path.join( root, "etc", "yum.repos.d" )

# Copied from A4.Helpers.run and stripped of un-needed features.
def run( argv, captureStdout=True, captureStderr=False ):
   p = subprocess.Popen( args=argv,
                         stdout=captureStdout and subprocess.PIPE or None,
                         stderr=captureStderr and subprocess.PIPE or None,
                         stdin=None, universal_newlines=True )
   try:
      ( stdout, stderr ) = p.communicate()
   except:
      while True:
         try:
            p.wait()
            break
         except OSError as e:
            if e.errno == errno.ECHILD:
               break
         except: # pylint: disable-msg=W0702
            pass
      raise
   rc = p.returncode
   assert rc is not None
   if rc:
      raise SwiException( "'%s': error code %d" % ( " ".join( argv ), rc ),
                          ( stdout or '' ) + ( stderr or '' ) )
 
   return (stdout or '') + (stderr or '')

class SwiChrootContext:
   '''A context manager that handles entering and exiting a chroot environment.
   Allows an instantiator to run commands while chroot-ed in the swi workspace.'''
   def __init__( self, root ):
      self.root = root
      self.outsideRoot = None
   def __enter__( self ):
      self.outsideRoot = os.open( "/", os.O_RDONLY )
      swiMountedIs( self.root, True, False )
      os.chroot( self.root )
      os.chdir( "/" )
      if "A4_CHROOT" in os.environ:
         # Update A4_CHROOT, if it exists, as it points to the outer directory which
         # will not be accessible in this chroot.
         os.environ[ "A4_CHROOT" ] = os.getcwd()

   def __exit__( self, exc_type, exc_value, traceback ):
      os.fchdir( self.outsideRoot )
      os.chroot( "." )
      os.close( self.outsideRoot )
      swiMountedIs( self.root, False, False )

def parseVersionFile( versionFilePath ):
   versionDict = {}
   with open( versionFilePath ) as versionFile:
      for line in versionFile:
         keyValue = line.split( '=', 1 )
         if len( keyValue ) == 2:
            versionDict[ keyValue[ 0 ] ] = keyValue[ 1 ].strip()
   return versionDict

def getSwiRepoUsingRelease( versionDict, options ):
   '''Tries to get the proper repo using just the release field of the swi image.'''
   if "SWI_RELEASE" not in versionDict:
      cmdPrint( "No SWI_RELEASE key in the version file." )
      return NOREPO
   swiRelease = versionDict[ "SWI_RELEASE" ]
   # parse SWI_RELEASE to get project data. Apparently many release names
   # encode a change number and project name.
   projectData = swiRelease.split( ".", 2 )
   # EOS.swi images that are manually built are formatted:
   # [changeNum].[project name].XX, where XX is a number.
   # Don't include the ".XX" in the project name.
   if len( projectData ) < 2:
      cmdPrint( "Unable to parse SWI_RELEASE and extract project and change "
                  "number." )
      return NOREPO
   projectChangeNo = projectData[ 0 ]
   projectName = projectData[ 1 ]
   cmdPrint( "Project name: %s, project change number: %s." %
               ( projectName, projectChangeNo ) )
   flag = []
   if options.force:
      flag = [ "-" + "f" * options.force ]
   output = run( [ "a4", "rpmrepo" ] + flag +
                 [ "-c", projectChangeNo, "-p", projectName ] )
   cmdPrint( output[ : -1 ] )
   return output.splitlines()[ -1 ]

def getSwiRepo( versionDict, options, swiFile=None ):
   u = swiUrl( swiFile )
   if u:
      repoURL = getRepoFromLocation( '%s', [ u.replace( 'images', 'RPMS' ) ] )
      if repoURL is None:
         # getRepoFromLocation returns python None when the repo does not
         # exist, but NOREPO is the string "None", so we need to convert.
         repoURL = NOREPO
   else:
      repoURL = NOREPO
   if repoURL == NOREPO:
      if 'SERIALNUM' not in versionDict:
         raise SwiException( "No SERIALNUM key in the version file." )
      serialNum = versionDict[ 'SERIALNUM' ]
      cmdPrint( "Serial number: %s." % serialNum )
      output = run( [ "a4", "rpmrepo", "-s", serialNum ] )
      cmdPrint( output[ : -1 ] )
      repoURL = output.splitlines()[ -1 ]
   if repoURL == NOREPO:
      repoURL = getSwiRepoUsingRelease( versionDict, options )
   if repoURL == NOREPO:
      raise SwiException( "Unable to find RPMs corresponding to "
                          "the .swi file specified." )
   cmdPrint( "Using repository %s." % repoURL )
   return repoURL

def getAndWriteSwiRepo( root, versionDict, options, swiFile=None ):
   repoURL = getSwiRepo( versionDict, options, swiFile )
   yumReposD = yumRepoPath( root )
   with open( os.path.join( yumReposD, "Abuild.repo" ), "w" ) as abuildRepo:
      writeRepoEntry( abuildRepo, "Abuild", "Abuild", repoURL, True )

def writeToolsV2Repo( root ):
   yumReposD = yumRepoPath( root )
   repoURL = "http://tools/ToolsV2/repo/$basearch/RPMS/"
   with open( os.path.join( yumReposD, "ToolsV2.repo" ), "w" ) as toolsV2Repo:
      writeRepoEntry( toolsV2Repo, "ToolsV2", "ToolsV2", repoURL, True )

def maybeWriteRepo( repoFile, name, repoURL, enabled ):
   try:
      # Determine whether or not a URL specifies a yum repository
      # (looks for repoURL/repoData/repomd.xml)
      repoMetadataURL = "%s/repodata/repomd.xml" % repoURL
      metadataObj = urllib.request.urlopen( repoMetadataURL )
      metadataObj.close()
   except urllib.error.URLError as e:
      cmdPrint( f"Skipping repo with name '{name}' at {repoURL}: {e}." )
   else:
      writeRepoEntry( repoFile, name, name, repoURL, enabled )

def writeSwiRepos( root, repoURIs ):
   '''Create repo files for each repository specified by the user.'''
   repoNum = 0
   with open( os.path.join( yumRepoPath( root ), "repoMounts" ), "w" ) as repoMounts:
      with open( os.path.join( yumRepoPath( root ), "user.repo" ), "w" ) as userRepo:
         for repoURI in repoURIs:
            repoName = "userRepo" + str( repoNum )
            if repoURI.startswith( ( "http://", "https://", ) ):
               # only write the entry if the http repo is present
               maybeWriteRepo( userRepo, repoName, repoURI.rstrip( "/" ), True )
            else:
               # the repo specified is a local path. Write the path to a file if it
               # is a valid repo, and have swi chroot mount it later.
               pathToRepo = repoURI[ len( "file://" ): ] \
                   if repoURI.startswith( "file://" ) else repoURI
               pathToRepo = os.path.abspath( pathToRepo )

               if os.path.exists( pathToRepo + "/repodata/repomd.xml" ):
                  userRepoDir = os.path.join( yumRepoPath( root ), repoName )
                  os.mkdir( userRepoDir )
                  repoMounts.write( f"{pathToRepo} --> {userRepoDir}\n" )
                  writeRepoEntry(
                     userRepo, repoName, repoName,
                     "file://" + os.path.join( yumRepoPath( "/" ), repoName ), True )
               else:
                  cmdPrint( "Skipping repo at '%s': repository not found" %
                              pathToRepo )
            repoNum += 1

def checkAndWriteFedoraRepos( fedoraRepoFile, version, arch ):
   # get the fedora debug releases URL
   repoReleaseURL = "http://mirrors.aristanetworks.com/fedora/linux/releases/" \
       "%s/Everything/%s/debug" % ( version, arch )
   repoName = "fedora-releases-fc%s-debug" % version
   maybeWriteRepo( fedoraRepoFile, repoName, repoReleaseURL, False )
   # get the fedora debug updates URL
   repoUpdateURL = "http://mirrors.aristanetworks.com/fedora/linux/updates/" \
       "%s/%s/debug" % ( version, arch )
   repoName = "fedora-updates-fc%s-debug" % version
   maybeWriteRepo( fedoraRepoFile, repoName, repoUpdateURL, False )
   # Add the releases and updates URLs.
   writeRepoEntry( fedoraRepoFile, "Fedora-releases", "Fedora releases",
                   "http://mirrors.aristanetworks.com/fedora/linux/"
                   "releases/%s/Everything/%s/os" % ( version, arch ), False )
   writeRepoEntry( fedoraRepoFile, "Fedora-updates", "Fedora updates",
                   "http://mirrors.aristanetworks.com/fedora/linux/updates/"
                   "%s/%s" % ( version, arch ), False )

def checkAndWriteCentosRepos( centosRepoFile, version, arch ):
   # Add the appropriate Centos debuginfo repo for the version
   repoReleaseURL = f"http://debuginfo.centos.org/{version}/{arch}"
   repoName = "CentOS-el%s-debug" % version
   maybeWriteRepo( centosRepoFile, repoName, repoReleaseURL, False )
   # Add the releases and updates URLs.
   if arch == 'x86_64':
      mirror = "http://mirrors.aristanetworks.com/centos"
   else:
      # We don't mirror CentOS altarch yet
      mirror = "http://mirror.centos.org/altarch"
   writeRepoEntry( centosRepoFile, "CentOS-releases", "CentOS releases",
                   f"{mirror}/{version}/os/{arch}", False )
   writeRepoEntry( centosRepoFile, "CentOS-updates", "CentOS updates",
                   f"{mirror}/{version}/updates/{arch}", False )

def createUpstreamRepos( arch ):
   '''Parse `dnf/yum list` and from the release names, extract elX.
   Then add debug repos of fedora version XX for fedora.'''
   # Would like to use the following command:
   # dnf repoquery -c /etc/Adnf/Adnf.conf -a --pkgnarrow=installed --nvr
   # but dnf-plugins-core not yet installed
   # pkgdeps: rpm dnf
   if useDnf:
      yumList = run( [ "dnf", "list", "installed" ] )
   else:
      yumList = run( [ "yum", "list", "installed" ] )

   centosRepo = open( os.path.join( yumRepoPath( "/" ), "CentOS-Debug.repo" ), "w" )
   extractVersion = re.compile( r"\.(el|fc)(\d+)(\.|$)" )
   centosVersions = collections.defaultdict( int )
   for line in yumList.splitlines():
      yumData = line.split()
      if len( yumData ) < 3:
         continue # will miss some packages due to line wrap, ignored
      reObj = re.search( extractVersion, yumData[ -2 ] )
      if reObj:
         flavor = reObj.group( 1 )
         if flavor == "el":
            version = reObj.group( 2 )
            centosVersions[ version ] += 1
   # Only create repos for the most common version of rpms installed.
   if centosVersions:
      version = max( centosVersions.items(), key=operator.itemgetter( 1 ) )[ 0 ]
      checkAndWriteCentosRepos( centosRepo, version, arch )
   centosRepo.close()

def writeRepoEntry( repoFile, repoId, repoName, baseUrl, enabled ):
   '''writes the .repo file necessary to add a yum repository'''
   enabledVal = 1 if enabled else 0
   repoFile.write( '''[{}]
name={}
baseurl={}
enabled={}
gpgcheck=0
'''.format( repoId, repoName, baseUrl, enabledVal ) )

def copyEtcResolv( root ):
   with open( "/etc/resolv.conf" ) as resolvSrc:
      with open( os.path.join( root, "etc", "resolv.conf" ), "w" ) as resolvDst:
         resolvDst.write( resolvSrc.read() )

class FedoraRepoContext:
   '''Context that, when entered, creates the default fedora repos, and, when exited,
   deletes them.'''
   def __init__( self, arch ):
      self.arch_ = arch
      self.repoFileVersion_ = 0
      self.repoFilename_ = None
   def __enter__( self ):
      self.repoFilename_ = os.path.join(
         yumRepoPath( "/" ), "fedora%d.repo" % self.repoFileVersion_ )
      with open( self.repoFilename_, "w" ) as repoFile:
         checkAndWriteFedoraRepos( repoFile, "18", self.arch_ )
   def __exit__( self, exc_type, exc_value, traceback ):
      os.remove( self.repoFilename_ )
      self.repoFileVersion_ += 1

def installPrograms():
   '''Install programs that help with debugging in the swi workspace that are not
   present on the DUT.'''
   pkgs = [ Package( "file", [ "which", "file" ] ),
                     Package( "binutils", [ "which", "readelf" ] ),
            # NOTE: a4 provided by Artools-bootstrap will be overwritten since
            # a4 functionality in swi workspace is reduced.
            Package( "Artools-bootstrap", [ "which", "a4" ] ) ]
   for pkg in pkgs:
      pkg.install()
      if not pkg.installed():
         cmdPrint( "Unable to install the %s package" % pkg )

def runInSwiWorkspaceChroot( root, arch, noAuto, installArt ):
   with SwiChrootContext( root ):
      # We need to make sure that /etc/mtab exists.
      if not os.path.exists( "/etc/mtab" ):
         open( "/etc/mtab", "w" ).close()

      # Dnf is used for newer release images, but won't be installed for older
      # releases
      global useDnf
      useDnf = bool( shutil.which( "dnf" ) )

      # not including "EosImage-ArosTestPlugins" just yet: it would download ~250
      # packages, and running product tests in swi workspace will come later.
      if useDnf:
         packages = [ Package( "dnf-plugins-core",
                               [ "dnf", "debuginfo-install", "-h" ] ) ]
      else:
         packages = [ Package( "yum-utils", [ "which", "debuginfo-install" ] ) ]

      if noAuto:
         # if repos were manually specified, install file using those repos.
         packages.append( Package( "file", [ "which", "file" ] ) )
      else:
         createUpstreamRepos( arch )
         installPrograms()
      if installArt:
         packages.append( Package( "Art", [ "which", "Art" ], lookupPkg=True ) )
         # in older EOS.swis, the dependencies aren't 100% correct, so manually
         # attempt the install of RdamDatabase just in case.
         packages.append( Package( "RdamDatabase", None ) )
         # Art uses P4 which uses mx.DateTime.Parser.
         packages.append( Package( "mx", [ sys.executable, "-c",
                                           "import mx.DateTime.Parser" ] ) )
      # This rpm provides the elftools module needed by a4 debug.
      packages.append( Package( "python3-pyelftools", [ "a4", "debug", "-h" ] ) )
 
      # Used for pretty printing in gdb.
      packages.append( Package( "Marco-gdb", [ "rpm", "-q", "Marco-gdb" ] ) )
      packages.append( Package( "ArSlab-gdb", [ "rpm", "-q", "ArSlab-gdb" ] ) )
      packages.append( Package( "tacgdb", [ "rpm", "-q", "tacgdb" ] ) )

      # Artools must be installed last to overwrite any Artools changes installed
      # by the previous packages.
      packages.append( ArtoolsPackage() )

      packagesToRetry = []

      for package in packages:
         package.install()
         if not package.installed():
            packagesToRetry.append( package )

      if packagesToRetry:
         failedPackages = []
         # If any install fails, retry with Fedora repos.
         with FedoraRepoContext( arch ):
            for package in packagesToRetry:
               package.install()
               if not package.installed():
                  failedPackages.append( package )
                  print( package.output() )
         if failedPackages:
            cmdPrint( "The following packages could not be installed: %s" %
                      str( failedPackages ) )

def moveRootfs( tempRootfs, userDir ):
   '''move all files and directories in tempRoofs into userDir'''
   allFilePaths = glob.glob( os.path.join( tempRootfs, "*") )
   for filePath in allFilePaths:
      filename = os.path.basename( filePath )
      # os.rename does not work across mount points
      # shutil.move does not preserve some metadata
      Swi.run( [ "mv", filePath, os.path.join( userDir, filename ) ] )

def findRootfsInExtractedSwiDir( extractedSwiDir ):
   potentialRootfses = glob.glob( os.path.join( extractedSwiDir, "rootfs*.dir" ) )
   tempRootfs = None
   if potentialRootfses:
      potentialRootfs = potentialRootfses[ 0 ]
      if os.path.isdir( potentialRootfs ):
         tempRootfs = potentialRootfs
   if not tempRootfs:
      raise SwiException( 'No chroot environment found in swi file.' )
   return tempRootfs

def swiUrl( swiFile ):
   if swiFile.startswith( ( "http:", "https:", ) ):
      return os.path.dirname( swiFile )
   else:
      return None

def maybeRetrieveSwiFile( swiFile ):
   '''if the swiFile specifies an http URL, try to retrieve it'''
   if swiUrl( swiFile ):
      handle, tempPath = tempfile.mkstemp( suffix='.swi' )
      cmdPrint( "Swi file looks like a URL. Downloading file." )
      try:
         with os.fdopen( handle, "wb" ) as localSwi:
            remoteSwi = urllib.request.urlopen( swiFile )
            while True:
               filePart = remoteSwi.read( 1048576 )
               if filePart:
                  localSwi.write( filePart )
               else: break
            remoteSwi.close()
      except Exception as e:
         os.remove( tempPath )
         # pylint: disable-next=raise-missing-from
         raise SwiException( "Retrieving SWI file: %s" % e )
      return tempPath
   return swiFile

def getSwiArch( swiFile ):
   swiData = run( [ "swi", "info", swiFile ] )
   versionMatch = re.search( r"SWI_ARCH=(\S+)", swiData )
   if versionMatch:
      arch = versionMatch.group( 1 )
      if arch == "i686":
         return "i386"
      return arch
   # Assume older EOS.swis that don't have a SWI_ARCH variable use i386/
   return "i386"

def deleteWorkspace( workspaceRoot ):
   if os.path.isdir( workspaceRoot ):
      success = swiMountedIs( workspaceRoot, False, False )
      if success:
         Swi.run( [ "rm", "-rf", workspaceRoot ] )
      else:
         cmdPrint( "not removing swi workspace: unmounts failed." )
   else:
      cmdPrint( "%s is not a directory." % workspaceRoot )

def ensureA4Installed():
   '''test that a4 is installed'''
   try:
      run( [ "which", "a4" ], captureStderr=True )
   except SwiException:
      # pylint: disable-next=raise-missing-from
      raise SwiException( "'swi workspace' needs a4 to be installed, "
                          "which it is not. " )

def workspaceHandler( args ):
   argv = args
   parser = argparse.ArgumentParser(
      prog="swi workspace", formatter_class=argparse.RawDescriptionHelpFormatter,
      description=
"""Creates a new workspace from a (EOS).swi file. This includes adding yum
repositories containing Arista-specific and fedora-specific RPMS that are consistent
with those installed in the specified .swi file. Also deletes a workspace using the
-d option.

There are cases where the repository is known, but due to infrastructure problems,
it is not found for an EOS.swi. For example, an EOS.swi at
http://dist/Abuild/ga.marietta-dev/i386_14/latest/images/EOS.swi uses
http://dist/Abuild/ga.marietta-dev/i386_14/latest/RPMS as the respoitory, but may not
be found due to the corresponding entries not being filled in in the mysql database.
In this case, use the -r option to manually specify the repository.

When using the -r option above, you may also need to specify the repository to the
centos rpms, at
http://mirrors.aristanetworks.com/centos/[centos version]/<arch>
""", usage="%%prog [--noauto] "
"""[-f[f]] [ -r repository ]* DIRECTORY [http://]SWI-FILE

%%prog -d DIRECTORY""" )
   parser.add_argument( "-f", dest="force", action="count", default=0,
                        help="""Allows the use of repositories from workspaces of
successful abuilds, as opposed to just repositories from http://dist/* (the default).
By default they are not used, as this method does the right thing most of the time,
but not 100%%. -ff goes further, allowing the use of repositories from workspaces of
failed abuilds.""" )
   parser.add_argument( "-r", dest="repos", action="append", default=[],
                        help="""Specifies a URI pointing to a yum repository to look
for packages to install. This option can be repeated to add multiple repositories.
Note that no priorities are given to the repositories specified and/or generated. The
file:// and http:// schemas are supported.""" )
   parser.add_argument( "--noauto", action="store_true", default=False,
                        help="""Disables the automatic search for a repository
corresponding to the EOS.swi file. If this option is specified, use the -r option to
specify the necessary repositories.""" )
   parser.add_argument( "-d", "--delete", action="store_true", default=False,
                        help="""Delete the specified swi workspace.""" )
   parser.add_argument( "--noArt", action="store_true",
                        help="""Does nothing. Art is  not installed by default.""" )
   parser.add_argument( "--art", action="store_true",
                        help="""Install Art in the swi workspace.""" )
   # Don't use this option; it is used by the command internally to denote whether
   # or not swi workspace is in its own mount namespace.
   parser.add_argument( "--in-mount-namespace", dest="inMountNamespace",
                        action="store_true", help=argparse.SUPPRESS )
   parser.add_argument( "directory", nargs=1,
                        help="Directory to create the swi workspace." )
   parser.add_argument( "swi", nargs="?",
                        help="""Swi file, or http:// path to swi file, to create the
workspace from (not used when deleting a workspace).""" )
   args = parser.parse_args( argv )

   newCmd = None
   newCmdArgs = []

   if os.geteuid() != 0:
      cmdPrint( "Running swi workspace as root." )
      sys.stdout.flush()
      newCmd = "sudo"
      newCmdArgs = [ "sudo" ]
      # this option shouldn't be specified if we are not running as root, as it
      # is hidden.
      assert not args.inMountNamespace

   if not args.inMountNamespace:

      if newCmd is None:
         newCmd = "chns"
      newCmdArgs += [ "chns", "-m", "/usr/bin/swi", "workspace",
                      "--in-mount-namespace" ]

   if newCmd is not None:
      os.execvp( newCmd, newCmdArgs + argv )
      sys.exit( 1 )

   workspaceRoot = args.directory[ 0 ]

   if args.delete:
      if args.swi:
         parser.error( "Too many arguments for deleting workspace." )
      deleteWorkspace( workspaceRoot )
      return

   if not args.swi:
      parser.error( "Missing .swi file to use to create workspace." )
   swiFile = args.swi

   alreadyExists = os.path.isdir( workspaceRoot )
   if alreadyExists:
      if os.listdir( workspaceRoot ):
         parser.error( "Directory specified is not empty (appears to be in use)." )
   else:
      os.makedirs( workspaceRoot )
   rootPath = os.path.abspath( workspaceRoot )

   newSwiFile = maybeRetrieveSwiFile( swiFile )
   arch = getSwiArch( newSwiFile )
   workspaceArch = run( [ "uname", "-m" ] ).strip()
   if ( "x86_64" in workspaceArch ) != ( "x86_64" in arch ):
      cmdPrint( "WARNING: EOS image specified is of a different architecture (%s) "
                "than the workspace (%s). Recommend using a workspace or image that "
                "are consistent in architecture." % ( arch, workspaceArch ) )

   setTempDirIfNeeded()
   extractedSwiDir = tempfile.mkdtemp()
   try:
      ensureA4Installed()
      swiExtract( newSwiFile, Swi.SwiOptions( dirname=extractedSwiDir,
                                              use_existing=True ) )
      versionFilePath = os.path.join( extractedSwiDir, "version" )
      if not os.path.isfile( versionFilePath ):
         raise SwiException( "%s does not appear to be a swi image "
                             "(no version file)." % newSwiFile )
      versionDict = parseVersionFile( versionFilePath )

      if SwimHelperLib.isSwimDir( extractedSwiDir ):
         tempRootfs = None
         try:
            # Mount the first superset overlayFS found and copy it to rootPath
            tempRootfs = SwimHelperLib.mountSwimWorkspaceRootfs( extractedSwiDir )
            moveRootfs( tempRootfs, rootPath )
         finally:
            # umount tempRootfs
            if tempRootfs:
               SwimHelperLib.umountSwimFs( tempRootfs, ignoreReturnCode=True )
      else:
         tempRootfs = findRootfsInExtractedSwiDir( extractedSwiDir )
         moveRootfs( tempRootfs, rootPath )

      copyEtcResolv( rootPath )

      writeSwiRepos( rootPath, args.repos )
      if not args.noauto:
         getAndWriteSwiRepo( rootPath, versionDict, args, swiFile )
      writeToolsV2Repo( rootPath )

      runInSwiWorkspaceChroot( rootPath, arch, args.noauto, args.art )
      cmdPrint( "Run 'swi chroot %s' to use the new workspace." %
                workspaceRoot )
   except SwiException as e:
      print( "Error:", e )
      print( "Other useful data:", e.data )
      if 'SWI_BREADTH_TEST' not in os.environ:
         deleteWorkspace( rootPath )
   except:
      if 'SWI_BREADTH_TEST' not in os.environ:
         deleteWorkspace( rootPath )
      raise
   finally:
      if newSwiFile != swiFile:
         os.remove( newSwiFile )
      Swi.run( [ "rm", "-rf", extractedSwiDir ] )

def workspaceHelp():
   workspaceHandler( [ "-h" ] )

if __name__ == "__main__":
   workspaceHandler( sys.argv[ 1: ] )
