#!/usr/bin/env python3
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import glob
import os
import re
import shutil
import sys
import tempfile
import platform

from EosVersion import swimUSDefault, swimUSDPE, swiLatestImageFormatVersion, \
                       sidToOptimizationFile, swimHdrFile
from Swi.installrootfs import updateImageFormatVersion, \
                              updateSwiOptimization
import SwimHelperLib
import Tac

sqshFlavors = [
   swimUSDefault,
   swimUSDPE,
]

# Duplicates an overlayFS by overlay mounting all lowerdirs
# that make up the overlayFS and then duplicates the merged/union dir
# to rootfs-i386.dir using the rsync command
def overlayDupe( flavor, lowerDirsArg, swiDir, rootfsDir ):
   # tmpdir to hold our unioned rootfs-i386.dir
   tmpDir = tempfile.mkdtemp( prefix="union-tmp-" )
   tmpRootfsDir = os.path.join( tmpDir, rootfsDir )
   Tac.run( [ "mkdir", "-p", tmpRootfsDir ] )

   try:
      print( f"Mounting {flavor} OverlayFS on {tmpRootfsDir}:" )
      print( "lowerdir=%s" % lowerDirsArg )
      print()
      SwimHelperLib.mountSwimfsReadOnly( tmpRootfsDir, lowerDirsArg, flavor )
      # Duplicate overlayFS into single rootFS
      print( "Duplicating %s OverlayFS to a single rootfs." % flavor )
      print( f"rsyncing {tmpRootfsDir} to {swiDir}" )
      print()
      Tac.run( [ "rsync", "-axHAX", tmpRootfsDir, swiDir ], asRoot=True )
   finally:
      SwimHelperLib.umountSwimFs( tmpRootfsDir )
      shutil.rmtree( tmpDir )

# Duplicates an overlayFS by copying each lowerdir, from lowest to highest, to
# the rootfs-i386.dir
def nonOverlayDupe( flavor, lowerDirsArg, swiDir, rootfsDir ):
   swiRootfs = os.path.join( swiDir, rootfsDir )
   Tac.run( [ "mkdir", "-p", swiRootfs ], asRoot=True )

   # lowerdirs are colon separated paths
   lowerDirs = lowerDirsArg.split( ":" )
   # Copy overlay filesystem dirs from lowest to highest
   print( "Duplicating %s OverlayFS to a single rootfs." % flavor )
   for fsDir in reversed( lowerDirs ):
      print( f"Copying {fsDir} to {swiRootfs}" )
      for fpath in os.listdir( fsDir ):
         Tac.run( [ "cp", "-a", os.path.join( fsDir, fpath ), swiRootfs ],
                    asRoot=True )

   # Remove known files which have been deleted in the upper dirs of the OverlayFS.
   # These files will still be in the resulting rootfs-i386.dir if they
   # existed in one or more of the lower dirs.
   Tac.run( [ "rm", "-f",
              os.path.join( swiRootfs, 'usr/lib/locale/locale-archive' ) ],
            asRoot=True )

# Flatten an extracted SWIM by converting an overlayFS into
# a single rootfs and removing SWIM related files
def swimFlatten( swiDir, nonOverlay=False ):
   # Get overlayDirs - pass in empty string for workdir since
   # it's not applicable here
   _, overlayDirs = SwimHelperLib.getSwimSqshMapAndOverlayDirs( swiDir, None )

   # Determine which overlay flavor we will be mounting (whichever is available!)
   optimizations = [ "Default", "Default-DPE", "DEFAULT", "DPE",
                     "Sand-4GB", "Sand-4GB-DPE", "Strata-4GB", "Strata-4GB-DPE" ]
   for optim in optimizations:
      if optim in overlayDirs:
         flavor = optim
         break
   else:
      sys.stderr.write( "Error: input image appears to be an improper SWIM image\n" )
      sys.exit( 1 )

   if platform.machine() == "aarch64":  # A4NOCHECK
      rootfsDir = "rootfs-aarch64.dir"
   else:
      rootfsDir = "rootfs-i386.dir"
   # a SWIM flavor overlayFS will mounted and duplicated to rootfs-i386.dir, so
   # here we ensure the  rootfs dir doesn't already exist in the swiDir
   assert not os.path.exists( os.path.join( swiDir, rootfsDir ) ),\
         "Error: %s already exists in the extracted SWI dir" % rootfsDir

   # Retrieve lowerdirs from overlayDirs map, prepend flavor rootfs
   # to lowerDirs and include swiDir in all lowerdir paths
   lowerDirs = overlayDirs[ flavor ][ "lower" ]
   lowerDirsArg = os.path.join( swiDir, overlayDirs[ flavor ][ "root" ] )
   lowerDirsArg += ":%s" % lowerDirs

   # Overlay mounting is not supported on kernel versions 3.x.x
   # so force the non-overlay duplication if we're on 3.x.x
   unameOutput = Tac.run( [ "uname", "-r" ], stdout=Tac.CAPTURE )
   kernel = re.match( r'(\d+\.\d+\.\d+)-', unameOutput ).group( 1 )
   forceNonOverlay = kernel.startswith( "3." )

   if forceNonOverlay:
      print( "WARNING: Kernel version %s detected" % kernel )
      print( "Non-overlay duplication will be forced due to kernel version" )
      print( "incompatibility with overlay mounting." )
      dupeFunc = nonOverlayDupe
   elif nonOverlay:
      dupeFunc = nonOverlayDupe
   else:
      dupeFunc = overlayDupe

   dupeFunc( flavor, lowerDirsArg, swiDir, rootfsDir )

   # version and signature files from other rootfs/optimizations
   signatureFiles = glob.glob( '%s/*.signature' % swiDir )
   versionFiles = glob.glob( '%s/*.version' % swiDir )

   cleanup = [ swimHdrFile, sidToOptimizationFile ] + signatureFiles + versionFiles
   # Remove all .dir & .sqsh files within the extracted SWI dir
   for fname in os.listdir( swiDir ):
      if re.match( r'(\S+)\.(dir|sqsh)', fname ) and fname != rootfsDir:
         cleanup.append( fname )

   # Update image format to SWI
   updateImageFormatVersion( os.path.join( swiDir, 'version' ),
                             swiLatestImageFormatVersion )
   # Update optimization. As we are flatenning, optimization is lost
   updateSwiOptimization( os.path.join( swiDir, 'version' ),
                          'None' )

   print( "Cleaning up the following SWIM sqshes & dirs:" )
   print( " ".join( cleanup ) )

   for cleanupFile in cleanup:
      Tac.run( [ "rm", "-rf", os.path.join( swiDir, cleanupFile ) ], asRoot=True )

   print( "SWIM flattening complete." )
