#!/usr/bin/env python3
# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import Tracing
import weakref
import StageHelper

__defaultTraceHandle__ = Tracing.Handle( 'SysdbSwitchover' )

t0 = Tracing.trace0
t3 = Tracing.trace3
t8 = Tracing.trace8
t9 = Tracing.trace9

# Remember to save a ref to the Switchover instance.

class SwitchoverStageHandler( Tac.Notifiee ):
   """A quick and dirty wrapper to simplify handling of switchover stages.
   Interface is fine --- it captures a common case --- but this should 
   be rewritten pretty simply in tacc.  If we need to wait for 
   locallyReadOnly to be false, then we should pass in a filtered 
   redundancyStatus rather than the raw redundancyStatus"""

   notifierTypeName = 'Redundancy::RedundancyStatus'

   def __init__( self, redStatus, entityManager,
                 agent, stage, helper, handler, completes=False ):
      t0( 'SwitchoverStageHandler:__init__' )
      self.agent_ = agent
      self.stage_ = stage
      self.eventCfg_ = None
      self.helper_ = weakref.proxy( helper )
      self.handler_ = handler
      self.em_ = entityManager
      self.completes_ = completes
      self.finished_ = False
      Tac.Notifiee.__init__( self, redStatus )
      self.handleSwitchoverStage( None ) # what if stage active before start?

   def close( self ):
      t0( 'close' )
      self.helper_ = None
      self.em_ = None
      self.handler_ = None
      Tac.Notifiee.close( self )

   @Tac.handler( 'mode' )
   def handleMode( self ):
      n = self.notifier_
      t0( 'handleProtocol', n.mode )
      if n.mode == 'switchover':
         # Possibly first time, so check that switchover stage hadn't been
         # set before mode
         self.handleSwitchoverStage( None )

   @Tac.handler( 'protocol' )
   def handleProtocol( self ):
      n = self.notifier_
      t0( 'handleProtocol', n.protocol )
      if n.protocol == 'sso':
         self.handleSwitchoverStage( None )

   @Tac.handler( 'switchoverStage' )
   def handleSwitchoverStage( self, key=None ):
      # ignore key
      t0( 'handleSwitchoverStage', key )
      n = self.notifier_
      if self.finished_:
         t9( "Already executed handler for", self.agent_, self.stage_,
             "returning" )
         return
      # Don't check for stage_ being active until we recurse, so that we
      # get one shot at trying to find our event config
      if( ( ( key == "" ) or ( key is None ) ) and
         # pylint: disable-next=superfluous-parens
         not ( self.stage_ in n.switchoverStage ) ):
         t9( "Expected stage", self.stage_, "not yet active" )
         return
      if( not self.eventCfg_ ): # pylint: disable=superfluous-parens
         t9( 'eventCfg set from helper outside of init,',
             'consider switching to use SimpleSwitchover' )
         self.eventCfg_ = \
             self.helper_.switchoverEventConfig( self.agent_, self.stage_,
                                                 create=False )
      event = self.eventCfg_
      if not event:
         t8( " event, agentConfig, not ready for", self.agent_, self.stage_ )

         def handleEventCfg( eventCfg ):
            # pylint: disable-next=superfluous-parens
            if( not self.helper_.agentMounted( self.agent_ ) ):
               self.handleSwitchoverStage( "" )
         # Be careful not to create an event if it has been deleted.  If it
         # *does* get created, though, we want to pass handleEventCfg
         self.helper_.switchoverEventConfig( self.agent_, self.stage_,
                                             handleEventCfg,
                                             create=False, reactOnly=True )
         # Will get called back in handleEventCfg
         return
      t8( " handling event for", self.agent_, self.stage_ )
      # check for complete/finished *after* call to handleMode, because
      # that may have called into us already.

      eventComplete = self.helper_.switchoverCompleteCheck( event )
      if eventComplete or self.finished_:
         t9( "Event already completed" )
         return
      if n.mode != 'switchover' or n.protocol != 'sso':
         t9( "Not in SSO switchover; deferring execution" )
         return
      if self.stage_ in n.switchoverStage:
         self.handler_()
         self.finished_ = True
         if self.completes_:
            self.helper_.switchoverEventCompleted( self.agent_, self.stage_ )
      else:
         t9( "Expected stage", self.stage_, "not yet active" )

class SimpleSwitchover:
   '''This class is used when the caller wants an instance of the 
      SwitchoverStageHandler SM without instantiating the Switchover class.
      This class implements all helper calls from SwitchoverStageHandler'''
   def __init__( self, entityManager, agentConfig, agentStatus ):
      self.agentConfig = agentConfig
      self.agentStatus = agentStatus
      redStatus = entityManager.redundancyStatus()
      self.redStatusManager = \
         Tac.newInstance( "Redundancy::Agent::RedundancyStatusManager",
                          entityManager.cEm_, redStatus )

   def switchoverEventConfig( self, agent, stage, *args, **kwargs ):
      t0( 'SimpleSwitchover EventConfig' )
      return self.agentConfig.event[ stage ]

   def switchoverEventCompleted( self, agent, stage ):
      t0( 'SimpleSwitchover EventCompleted' )
      statusKey = Tac.Value( 'Stage::AgentStatusKey', self.agentStatus.name,
                             stage, 'default' )
      self.agentStatus.complete[ statusKey ] = True

   def switchoverCompleteCheck( self, eventConfig ):
      t0( 'SimpleSwitchover CompleteCheck' )
      assert eventConfig.agent == self.agentStatus.name, \
         'EventConfig agent name differs from SimpleSwitchover agent name'
      stage = eventConfig.stage
      statusKey = Tac.Value( 'Stage::AgentStatusKey', self.agentStatus.name,
                             stage, 'default' )
      stageComplete = self.agentStatus.complete
      return statusKey in stageComplete and stageComplete[ statusKey ]

# Important note: if you expect to use an object returned by one of these
# routines you must have called Switchover() early enough to allow its
# mounts to have completed before you call any other method.
# Further, each time you specify a new agent, you must allow the agent config
# to be mounted, too.
class Switchover( StageHelper.StageHelper ):
   # A python class that wraps the Switchover tac model in an easier to
   # use interface.
   def __init__( self, entityManager,
                 mountStageClassStatus=True,
                 # if you know you will define events in a set of agents,
                 # pass their names here, so the agent configs can be mounted
                 # at instantiation.
                 agent=() ):
      t0( 'Switchover.__init__' )
      status = None
      if not mountStageClassStatus:
         status = 'Skipped'

      StageHelper.StageHelper.__init__( self, entityManager, 'switchover',
                                        agent=agent, status=status )
      self.redStatusManager_ = None

   def switchoverEventConfig( self, agent, stage, finish=None, create=True,
                              readOnly=False, reactOnly=False ):
      t0( 'switchoverEventConfig' )
      return self.eventConfig( agent, stage, finish, create, readOnly,
                               reactOnly )

   def findSwitchoverEventConfig( self, agent, stage ):
      return self.findEventConfig( agent, stage )

   def onSwitchoverEventConfig( self, agent, stage, handler ):
      self.onEventConfig( agent, stage, handler )

   # pylint: disable-next=dangerous-default-value
   def registerSwitchoverStage( self, agent, stage, dependencies=[],
                                force=False ):
      return self.registerStage( agent, stage, dependencies, force )

   def unregisterSwitchoverStage( self, agent, stage ):
      self.unregisterStage( agent, stage )

   def _redundancyStatus( self, sysdbIndependent=False ):
      redStatus = self.em_.redundancyStatus()
      if not sysdbIndependent:
         if not self.redStatusManager_:
            self.redStatusManager_ = \
                Tac.newInstance( "Redundancy::Agent::RedundancyStatusManager",
                                 self.em_.cEm_, redStatus )
         redStatus = self.redStatusManager_.filteredStatus
      return redStatus

   def registerSwitchoverEventHandler( self, agent, stage, function,
                                       sysdbIndependent=False ):
      """Create a reactor that will call function when STAGE is the
      active switchoverStage, and will mark event as complete when
      done.  It goes without saying that this routine can only be used
      if function is able to avoid returning until all of the relevant
      switchover operations are completed.  As this is called from a
      reactor, all of the work must be synchronous.
      If some of the work needs to be done in reactors, and may involve
      asynchrony, consider registerSwitchoverEventReactor"""
      t0( 'registerSwitchoverEventHandler' )
      self.eventReactors_.append(
         SwitchoverStageHandler( self._redundancyStatus( sysdbIndependent ),
                                 self.em_,
                                 agent, stage, self,
                                 function,
                                 completes=True ) )

   def unregisterSwitchoverEventHandler( self, agent, stage ):
      self.unregisterEventHandler( agent, stage )

   def registerSwitchoverEventReactor( self, agent, stage, function,
                                       sysdbIndependent=False ):
      """Create a reactor that will call function when STAGE is the
      active switchoverStage, but will NOT mark event as complete when
      the function returns.  There may still be more work to be done.
      Use this instead of registerSwitchoverEventHandler if some 
      of the work needs to be done in reactors, and may involve
      asynchrony.  It is your responsibility to call 
      switchoverEventCompleted when the event is done."""
      t0( 'registerSwitchoverEventReactor' )
      self.eventReactors_.append(
         SwitchoverStageHandler( self._redundancyStatus( sysdbIndependent ),
                                 self.em_,
                                 agent, stage, self,
                                 function,
                                 completes=False ) )

   def switchoverEventCompleted( self, agent, stage ):
      self.eventCompleted( agent, stage )

   def switchoverCompleteCheck( self, eventConfig ):
      return eventConfig.complete

   def registerSwitchoverStageDependency( self, stage, dependencies,
                                          force=False ):
      return self.registerStageDependency( stage, dependencies, force )

   def registerSwitchoverEvent( self, agent, stage, dependencies, function,
                                force=False, completes=False,
                                sysdbIndependent=False ):
      """Define an event in stage STAGE, list dependencies between stages, and
      create reactor that will call function when stage is active.  See 
      registerSwitchoverEventHandler for restrictions on FUNCTION"""
      t0( 'registerSwitchoverEvent' )
      cfg = self.stageIs( agent, stage, dependencies, force=force )
      self.eventReactors_.append(
         SwitchoverStageHandler( self._redundancyStatus( sysdbIndependent ),
                                 self.em_,
                                 agent, stage, self,
                                 function,
                                 completes=completes ) )
      return cfg
