# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import argparse
import subprocess
import os
import sys
import shutil

from Swix.Lowmem.payload import createPayload
from Swix.Lowmem.rpmspec import createSpec
from Swix.Lowmem.config import loadConfig

def setupBuildTree( workdir ):
   dirs = [ "BUILD", "RPMS", "SOURCES", "SPECS", "SRPMS", "BUILDROOT", "tmp" ]
   absdirs = [ os.path.join( workdir, name ) for name in dirs ]
   for absd in absdirs:
      os.makedirs( absd, exist_ok=True )

def rpmbuildExtension( workdir, archname, spec, sqfs, outputFile, isInSwix ):
   setupBuildTree( workdir )

   specsdir = os.path.join( workdir, "SPECS" )

   shutil.copy( spec, specsdir )
   shutil.copy( sqfs, specsdir )

   cwd = os.getcwd()
   os.chdir( workdir )

   rpmbuild = 'rpmbuild --define "_topdir `pwd`" ' \
              '--define "_sourcedir `pwd`/SPECS" ' \
              '--define "_tmppath `pwd`/tmp" ' \
              '--define "_rpmdir `pwd`/.." ' \
              '--define "_rpmfilename %%{NAME}.%%{ARCH}.rpm"'
   if isInSwix:
      rpmbuild += ' --define "rpm_in_swix 1"'
   rpmbuild += f' --target={archname} -v -ba {spec}'

   proc = subprocess.run( rpmbuild, check=False, capture_output=True,
                          text=True, shell=True )
   if proc.returncode != 0:
      print( f"Process {rpmbuild} failed with exit code {proc.returncode}.\n"
             f"Output:\n{proc.stdout}" )
      sys.exit( -1 )
   os.chdir( cwd )

def createExtension( outputFile, cfg, workdir="workdir" ):
   print( f"Output: {outputFile}, config: {cfg}, workdir: {workdir}" )

   absOutFile = os.path.abspath( outputFile )
   absWorkdir = os.path.abspath( workdir )

   archname = cfg[ "archname" ]
   isSwix = cfg[ "isSwix" ]

   payloadConfig = cfg[ "payload" ]
   rpmPayload = absOutFile[ : -len( ".rpm" ) ] + ".sqsh"
   rpms = payloadConfig[ "rpms" ].split()
   repoargs = payloadConfig[ "repoargs" ].split()
   createPayload( absWorkdir, archname, rpms, rpmPayload, repoargs )

   rpmSpecConfig = cfg[ "rpmSpec" ]
   rpmSpec = absOutFile + ".spec"
   instCmdsFile = rpmPayload + ".install"
   uninstCmdsFile = rpmPayload + ".uninstall"
   metadataTagsFile = rpmPayload + ".deps"
   listFile = rpmPayload + ".list"
   createSpec( rpmSpecConfig, listFile, metadataTagsFile, instCmdsFile,
               uninstCmdsFile, rpmSpec )

   rpmbuildExtension( workdir, archname, rpmSpec, rpmPayload, outputFile,
                      isSwix )

if __name__ == "__main__":
   parser = argparse.ArgumentParser()
   parser.add_argument( "-c", "--config", help="Config YAML file",
                        required=True )
   parser.add_argument( "-w", "--workdir", help="Work directory",
                        default="workdir" )
   parser.add_argument( "output", help="Output filename" )
   args = parser.parse_args()

   createExtension( args.output, args.config, args.workdir )
