#!/usr/bin/env python3
# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import json
import os
import weakref

import Tac
import Toggles.SysdbAgentToggleLib
import Tracing

defaultSysdbAirStreamConfigFilePath = \
      '/usr/share/SysdbAgent/SysdbAirStreamConfig.json'

def getAirStreamConfig():
   # Parse the file that contains additional configuration for Sysdb's AirStream
   # server, including:
   # - a TCP port for the AirStream server to open
   # - a set of specific prefixes to handle
   # - additional tracing to enable
   debugPort = None
   specificPrefixes = []
   traceSettingToAppend = ''

   # An stest sets a custom config file. Read the custom file, if specified.
   customFilePath = os.environ.get( 'SYSDB_AIRSTREAM_CONFIG_FILE' )
   sysdbAirStreamConfigFilePath = \
         customFilePath if customFilePath else defaultSysdbAirStreamConfigFilePath
   with open( sysdbAirStreamConfigFilePath ) as f:
      config = json.load( f )
      debugPort = config.get( 'AirStreamDebugPort' )
      specificPrefixes = config.get( 'SpecificPrefixes' )
      traceSettingToAppend = config.get( 'TraceSettingToAppend' )

   # In a btest environment, read environment variables to get additional
   # specificPrefixes. The format is a comma-separated string of prefixes.
   if btestSpecificPrefixes:= os.environ.get( 'SYSDB_AIRSTREAM_SPECIFIC_PREFIX' ):
      specificPrefixes += btestSpecificPrefixes.split( ',' )

   return ( debugPort, specificPrefixes, traceSettingToAppend )

def createAirStreamSm( em, scheduler ):
   try:
      # create path discovery entity for Sysdb
      socketsPath = em.lookup( 'airstream/sockets' )
      socketsPath.newEntity( "AirStream::SocketAndPaths", "Sysdb" )
   except NameError:
      print( "Failed to lookup AirStream paths.",
             "Skipping instantiating AirStream components" )
      return None

   debugPort, specificPrefixes, traceSettingToAppend = getAirStreamConfig()
   if debugPort:
      os.environ[ 'SYSDB_AIRSTREAM_DEBUG_PORT' ] = str( debugPort )
   if traceSettingToAppend:
      traceSetting = Tracing.traceSetting()
      Tracing.traceSettingIs( f'{traceSetting},{traceSettingToAppend}' )

   try:
      # Create /<sysname>/airstream directory to be used by Path-Type export
      em.root().parent.mkdir( 'airstream' )
      # Create a task group that will be paused when there is congestion during
      # the servicing of a gNMI subscribe request.
      # TODO this should be revisited when the AirStreamCoalescingQueue toggle is
      # enabled.
      priorityNormal = Tac.newInstance( 'Ark::TaskPriority', 'normal' )
      priorityEnforcementAny = \
            Tac.newInstance( 'Ark::TaskGroupPriorityEnforcement', 'allowAny' )
      notifyTaskGroup = Tac.newInstance( 'Ark::TaskGroup', 'Sysdb-GnmiNotify',
            priorityNormal, priorityEnforcementAny, scheduler )
      providerConfig = Tac.newInstance( 'AirStream::BaseCustomProviderConfig',
            notifyTaskGroup )
      # Use low priority when sending gnmi messages through AirStream
      priorityLow = Tac.newInstance( 'Ark::TaskPriority', 'low' )
      providerConfig.walkTaskPriority = priorityLow

      # Add specific prefixes that Sysdb will handle instead of AirStream agent.
      for prefix in specificPrefixes:
         providerConfig.specificPrefix.add( prefix )

      runnabilityDir = em.lookup( 'airstream/runnability' )

      # Unconditionally create the runnabilty SM, which is not runnable until the
      # "AirStream::Config" entity appears for Sysdb. There are some btests that
      # create the entity manually.
      entManRef = weakref.proxy( em )
      entManRef.airStreamReactor = \
         Tac.newInstance( "AirStream::AirStreamRunnabilitySm",
                          runnabilityDir,
                          em.cEm_,
                          em.sysname(),
                          "Sysdb",
                          providerConfig )
      if Toggles.SysdbAgentToggleLib.toggleSysdbAirStreamServerEnabled():
         # Create an entity that makes the AirStream server runnable
         runnabilityDir.newEntity( "AirStream::Config", "Sysdb" )
         # Enable the work queue locally
         entManRef.airStreamReactor.serverWrapper.server.locallyEnableCoalescing()
         return entManRef.airStreamReactor.serverWrapper.server
      else:
         return None

   except TypeError as e:
      print( "Failed to instantiate AirStream::Server.",
             "Skipping AirStream components", e )
      return None
