#!/usr/bin/env python3
# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import SimpleConfigFile
import Tracing
import EntityManager
import Cell
import Tac
import os

t0 = Tracing.trace0

redundancyConstants = Tac.Value( "SysdbUtil::RedundancyFileConstants" )

DEFAULT_REDUNDANCY_PROTOCOL = redundancyConstants.defaultRedundancyProtocol
REDUNDANCY_PROTOCOL_ENV_VAR = redundancyConstants.redundancyProtocolEnvVar
REDUNDANCY_PROTOCOL_FILE = redundancyConstants.redundancyProtocolFile
VERSION_FILE = '/etc/swi-version'

# This is the connection timeout for (SSO) standby because standby Sysdb
# needs to wait for active Sysdb to finish loading startup-config.
# It needs to be higher than EntityManager::sysdbInitWaitTime.
#
# However, right now we keep it as 1200 until we can handle switchover
# during this phase.
standbyConnectionTimeout = 1200

def localVersion( versionFile=VERSION_FILE ):
   try:
      config = dict( SimpleConfigFile.SimpleConfigFileDict(
            versionFile ).items() )
      version = config.get( 'SWI_VERSION' )
      release = config.get( 'SWI_RELEASE' )
      t0( "local version:", version, release )
      return ( version, release )
   except OSError:
      return ( None, None )

class RedundancyProtocolFile:
   def __init__( self ):
      self.rpFile = Tac.Value( "SysdbUtil::RedundancyProtocolFile" )

   def oprProtocol( self ):
      if self.rpFile.oprProtocol == "":
         return None
      return self.rpFile.oprProtocol

   def oprProtocolIs( self, protocol ):
      # Operational redundancy protocol may be different from negotiated protocol.
      # Ex: Peer negotiated sso protocol and then it was powered off.
      #     In this case negotiated protocol field will be sso and represents last
      #     negotiated protocol whereas opr protocol will be simplex.
      self.rpFile.updateOprProtocol( protocol )

   def negotiatedProtocol( self ):
      if self.rpFile.negotiatedProtocol == "":
         return None
      return self.rpFile.negotiatedProtocol

   def negotiatedProtocolIs( self, protocol ):
      # Write the redundancy protocol negotiated during NorCalInit phase to file.
      self.rpFile.updateNegotiatedProtocol( protocol )

   def swiVersionMismatch( self ):
      # 1 if swi were found incompatible during protocol negotiation, 0 otherwise.
      return self.rpFile.swiVersionMismatch

   def swiVersionMismatchIs( self, swiVersionMismatch_ ):
      swiVersionMismatch_ = 1 if bool( swiVersionMismatch_ ) else 0
      self.rpFile.updateSwiVersionMismatch( swiVersionMismatch_ )

   def hwEpoch( self ):
      if self.rpFile.hwEpoch == "":
         return None
      return self.rpFile.hwEpoch

   def hwEpochIs( self, hwEpoch ):
      self.rpFile.updateHwEpoch( hwEpoch )

   def swiVariant( self ):
      if self.rpFile.swiVariant == "":
         return None
      return self.rpFile.swiVariant

   def swiVariantIs( self, swiVariant ):
      self.rpFile.updateSwiVariant( swiVariant )

   def maxHwEpoch( self ):
      if self.rpFile.maxHwEpoch == "":
         return None
      return self.rpFile.maxHwEpoch

   def maxHwEpochIs( self, maxHwEpoch ):
      self.rpFile.updateMaxHwEpoch( maxHwEpoch )

class LockFile:
   def __init__( self ):
      self.lockFile = Tac.Value( "SysdbUtil::LockFile" )

   def create( self ):
      return self.lockFile.create()

   def fileName( self ):
      return self.lockFile.fileName

def redundancyProtocol( ):
   protocol = os.environ.get( REDUNDANCY_PROTOCOL_ENV_VAR )
   if protocol:
      assert protocol in ( 'sso', 'rpr', 'simplex' )
      return protocol
   try:
      protocol = RedundancyProtocolFile().rpFile.oprProtocol
      if protocol in ( 'sso', 'rpr', 'simplex' ):
         return protocol
   except OSError:
      pass
   return 'simplex' # default to simplex.

def onStandbySupervisor():
   ''' Return true if the caller is on the standby supervisor and false
   otherwise. Determines supervisor status by comparing current cell id and the
   active cell id. '''
   return Cell.cellType() == 'supervisor' and Cell.activeCell() != Cell.cellId()

def onStandbySsoSupervisor():
   return onStandbySupervisor() and redundancyProtocol() == 'sso'

def addLogFacility( em, facilityName ):
   """
   Used by DefaultConfig plugins to register a new log facility.
   `em` should be an instance of EntityManager.Local or
   EntityManager.Remote
   """
   logConfig = em.lookup( 'logging/config' )
   if facilityName not in logConfig.facilityConfig:
      logConfig.newFacilityConfig( facilityName )

def assertNoCircularReferences( sysname, plugins ):
   ''' Creates an EntityManager.Local object, then clears what should
   be the only reference to it. Then check that the object is
   destroyed. plugins has the same meaning as Local's plugins. '''
   _entMan = EntityManager.Local( sysname, plugins=plugins )
   oldDeletedCount = EntityManager.Local.deletedCount
   _entMan = None
   import gc # pylint: disable=import-outside-toplevel
   print( "collect", gc.collect() )
   assert EntityManager.Local.deletedCount == oldDeletedCount + 1, \
       "Reference count did not go to zero. Look for circular references"

   # If this check fails with plugins==None, try running this code
   # block to discover which plugin is creating the circular
   # reference:
   #
   # allPlugins = ['AgentMonitor', 'Artflow', # etc...
   # toLoad = []
   # for i, p in enumerate( allPlugins ):
   #    print "Adding plugin", p
   #    toLoad.append( p )
   #    assertNoCircularReferences( str( i ), plugins=toLoad )
