# Copyright (c) 2008, 2009, 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=superfluous-parens

import io
import math
from collections import namedtuple
import fcntl, termios, struct, os, sys
import functools

def terminalWidth():
   """Return the width of the terminal attached to stdout."""
   cols = 0
   if 'COLUMNS' in os.environ and os.environ[ 'COLUMNS' ].isdigit():
      cols = int( os.environ[ 'COLUMNS' ] )
   else:
      # Use the first file descriptor that works
      fileDescriptors = []
      for stream in ( sys.stdin, sys.stdout, sys.stderr ):
         if hasattr( stream, 'fileno' ):
            try:
               fileDescriptors.append( stream.fileno() )
            except ( NotImplementedError, io.UnsupportedOperation ):
               # This can happen when Pytest redirects a stream to a pseudo-file
               # where there is no file descriptor even though `fileno` is a valid
               # method.
               continue
      for fd in fileDescriptors:
         try:
            data = fcntl.ioctl( fd, termios.TIOCGWINSZ, '1234' )
            _rows, cols = struct.unpack( 'HH', data )
            break
         except Exception: # pylint: disable=broad-except
            pass
   # Never return 0.
   if cols == 0:
      cols = 80
   return cols

#----------------------------------------------------------------
# formatting routines to help print output neatly.
#----------------------------------------------------------------

def _listGet( listOfItems, i, default=None ):
   if( len ( listOfItems ) <= i ):
      return( default )
   else:
      return( listOfItems[ i ] )

def _isFloat( num ):
   try:
      float( num )
      return True
   except ValueError:
      return False

def _splitDotAlign( string ):
   """Splits a string of the form M.N U into a triple.

   Examples:
      Input = "10.2 kB"; Output = ("10", "2", "kB")
      Input = "10 bytes"; Output = ("10", "", "bytes")
      Input = "1.1"; Output = ("1", "1", "")

   Raises a ValueError if the input string is not parseable into form M.N U.
   """
   split_by_space = string.split( " ", 1 )
   if not _isFloat( split_by_space[ 0 ] ):
      raise ValueError( string + " is not of form M.N U" )
   split_by_dot = split_by_space[ 0 ].split( ".", 1 )
   n = "" if len( split_by_dot ) == 1 else split_by_dot[ 1 ]
   u = "" if len( split_by_space ) == 1 else split_by_space[ 1 ]
   return split_by_dot[ 0 ], n, u

class Format:
   def __init__( self, isHeading=False, justify="right", border=False,
                 maxWidth=None, minWidth=None, pad=" ", noBreak=False,
                 terminateRow=False, padding=0, align="top",
                 wrap=False, dotAlign=False ):
      """Format, as its name indicates, specifies how a column should be 
         formatted. Use formatRows and formatColumns method in TableFormatter to
         convey the format to TableFormatter. 
         The following are ways in which Format can be used :
             a) Truncating and inflating column widths.
                TableFormatter uses variable width columns and assign these
                width at the time of output. If you dont prefer this and want
                to specify fixed static widths, use minWidth and maxWidth
                arguments in the constructor

             b) TableFormatter adds whitespaces to beautify the output. 
                If instead you prefer compact spacing then use the
                following options :
                noPadLeftIs( True ) will do strict left justification to the left
                padLimitIs( True ) will remove extra padding that gets added when
                                   the table is samll
                noTrailingSpaceIs( True ) will remove the extraneous space that 
                                gets added at the end of a right justified cell
                                in lieu of a border. (If the column has
                                a right border, then this arg has no effect).
             c) justification -- left, center, right, and right-float
                                 justification supported. The justification
                                 passed to Format is used for all the cells in
                                 the column. right-align is an alias for
                                 Format(justify="right", dotAlign=True)
             d) Columns whose entries are of the form M.N U (eg: 32.5 Mbps) are
                more legible when aligned by unit and decimal place. You can set
                dotAlign=True in the constructor to enable this setting for a
                given column."""

      if minWidth and maxWidth:
         # pylint: disable-next=consider-using-f-string
         exp = "minWidth(%d) cannot be > maxWidth(%d)" % ( minWidth, maxWidth )
         assert minWidth <= maxWidth, exp
      assert isinstance( padding, int )
      # external attributes
      self.isHeading = isHeading
      self.border = border
      self.noBreak = noBreak
      self.noPadLeft = False
      self.minPadding = padding
      self.wrap_ = wrap
      # external attributes that are not initable:
      self.padLimit = False 
      self.noTrailSpace = False 
      #
      assert justify in [ "center", "right", "left", "right-float" ]
      self.justify_ = justify
      self.dotAlign_ = dotAlign
      if justify == "right-float":
         self.justify_ = "right"
         self.dotAlign_ = True
      self.pad_ = pad
      self.padding_ = 0
      self.terminateRow_ = terminateRow
      self.minWidth_ = minWidth
      self.maxWidth_ = maxWidth
      assert align in [ "top", "bottom", "middle" ]
      self.align_ = align
      self.borderStartPos_ = 0

   def noPadLeftIs( self, value ):
      self.noPadLeft = value

   def padLimitIs( self, value ): 
      self.padLimit = value

   def noTrailingSpaceIs( self, value ):
      self.noTrailSpace = value

   def wrap( self ):
      return( self.wrap_ )

   def borderStartPosIs( self, pos ):
      """When the border will start."""
      self.borderStartPos_ = pos

class FormatMnuInternal:
   # Internal representation of a column with M.N U (eg: 10.2 kbps) formatting
   def __init__( self, mw, nw, uw, hasDot, hasUnit ):
      self.mWidth = mw
      self.nWidth = nw
      self.uWidth = uw
      self.hasDot = hasDot
      self.hasUnit = hasUnit

class FormatInternal:
   # delegate to self.format_.  Don't need __setattr__ because
   # TableFormatter shouldn't be writing to any attributes of Format, just to
   # FormatInternal
   def __getattr__( self, attrName ):
      return( getattr( self.format_, attrName ))
   def __init__( self, internalFormat=None ):
      self.format_ = internalFormat or Format()
      self.reset()

   def reset( self ):
      # pylint: disable-msg=W0201
      self.minWidth_ = self.format_.minWidth_
      self.maxWidth_ = self.format_.maxWidth_
      self.width_ = None
      self.padding_ = self.minPadding 
      self.leftBorder_ = False
      self.nRows_ = None
      # For dotAlign columns
      self.mnu_ = FormatMnuInternal( 0, 0, 0, False, False )
      # pylint: enable-msg=W0201

   def newPadding( self, padding ):
      self.padding_ += padding

   def padding( self ):
      if self.format_.padLimit:
         return 0
      else:
         return( self.padding_ )

   def borderWidth( self ):
      borderWidth = 0
      if self.leftBorder_:
         borderWidth += 1
      if self.format_.border:
         borderWidth += 1
      elif self.format_.noTrailSpace:
         borderWidth -= 1
      return borderWidth

   def width( self ):
      return( self.borderWidth() + self.printableWidth() + self.padding() )

   def minWidth( self ):
      if self.minPrintableWidth():
         return( self.borderWidth() + self.minPrintableWidth() + self.padding() )
      else:
         return None

   def compressedWidth( self ):
      return( self.borderWidth() + self.compressedPrintableWidth() + self.padding() )

   def printableWidth( self ):
      return( self.width_ )

   def minPrintableWidth( self ):
      if self.minWidth_:
         return( self.minWidth_ )
      else:
         # Replace with heuristic for minWidth when it isn't specified.
         # Consider using the longest word from the column.
         return None

   def compressedPrintableWidth( self ):
      if self.wrap() and self.minPrintableWidth():
         return self.minPrintableWidth()
      else:
         return self.printableWidth()

   # Set width at output time when you know the widest item in column
   def widthIs( self, width ):
      if( self.minWidth_ and (width < self.minWidth_) ):
         width = self.minWidth_
      if( self.maxWidth_ and (width > self.maxWidth_) ):
         width = self.maxWidth_
      self.width_ = width # pylint: disable-msg=W0201

   def height( self ):
      return( self.numRows() )

   def numRows( self ):
      return( self.nRows_ )

   # Set numRows at output time when you know the tallest item in row
   def numRowsIs( self, nrows ):
      self.nRows_ = nrows # pylint: disable-msg=W0201

   def _spacePadLeft( self ):
      # A border already provides some separation, so need less whitespace
      # Put the bulk of the padding on the side away from the border.
      if self.leftBorder_ and not self.format_.border:
         spacePadLeft = min( max( self.padding() - 2, 0), 1 )
      elif not self.leftBorder_ and self.format_.border:
         spacePadLeft = self.padding() - min( max( self.padding() - 2, 0), 1 )
      else:
         spacePadLeft = int( math.floor( self.padding() / 2.0 ) )

      if self.format_.noPadLeft:          # force left justify
         spacePadLeft = 0
      return spacePadLeft

   def format( self, cell ):
      string = str( cell )

      if self.dotAlign_:
         # M.N U dot alignment formatting
         try:
            m, n, u = _splitDotAlign( string )
            if self.mnu_.hasDot:
               m += "." if len( n ) > 0 else " "
            string = " " * ( self.mnu_.mWidth - len( m ) ) + m
            string += n + " " * ( self.mnu_.nWidth - len( n ) )
            string += " " if self.mnu_.hasUnit else ""
            string += u + " " * ( self.mnu_.uWidth - len( u ) )
         except ValueError:
            pass # no formatting change if the string is not of the form M.N U

      # truncate printed area
      if len(string) > self.width_:
         string = string[0:self.width_]
      padLen = self.width_ - len(string)

      if self.format_.padLimit:
         self.padding_ = 0 # pylint: disable-msg=W0201

      if self.format_.justify_ == "right":
         string = self.format_.pad_*padLen + string
      elif self.format_.justify_ == "center":
         padLeft = int( math.floor( padLen/2.0 ))
         string = ( self.format_.pad_*padLeft + 
                    string + 
                    self.format_.pad_*(padLen - padLeft) )
      else:
         # "left"
         string = string + self.format_.pad_*padLen
      spacePadLeft = self._spacePadLeft()

      string = " "*spacePadLeft + string + " "*(self.padding() - spacePadLeft)
      if self.leftBorder_:
         string = " "+string
      if self.format_.border:
         string += " "
      return( string )

class MultiCellFormatInternal( FormatInternal ):
   def __init__( self, table, startCol, nCols=1, internalFormat=None ):
      FormatInternal.__init__( self, internalFormat=internalFormat )
      self.table_ = table
      self.startCol_ = startCol
      self.nCols_ = nCols
   # If a multiCellFormat is justified left and the leftmost underlying cell is
   # also justified left, then it needs to have its padding aligned with 
   # the inner cell, too.  Same goes for "right" justification.
   def _spacePadLeft( self ):
      # pylint: disable=protected-access
      if self.format_.noPadLeft:
         return 0
      elif self.format_.justify_ == "left":
         innerFormat = self.table_.columnFormats_[ self.startCol_ ]
         if innerFormat.format_.justify_ == "left":
            innerLeft = innerFormat._spacePadLeft()
            if innerFormat.leftBorder_ and not self.leftBorder_:
               innerLeft += 1
            elif self.leftBorder_ and not innerFormat.leftBorder_:
               innerLeft -= 1
            return min( self.padding(), max( innerLeft, 0 ))
      elif self.format_.justify_ == "right":
         endCol = self.startCol_ + self.nCols_ - 1
         innerFormat = self.table_.columnFormats_[ endCol ]
         if innerFormat.format_.justify_ == "right":
            rightPadding = innerFormat.padding() - innerFormat._spacePadLeft()
            if innerFormat.border and not self.border:
               rightPadding += 1
            elif self.border and not innerFormat.border:
               rightPadding -= 1
            
            return self.padding() - min( self.padding(), max( rightPadding, 0 ))
      return FormatInternal._spacePadLeft( self )
         
         
   



class _TerminateRow( Exception ):
   """Internal exception class used below to mark if a column is a final 
      column in the table. Set terminateRow in Format if you want/need to 
      break up the Table into multiple tables because the width is too long"""
   def __init__( self, row ):
      self.row = row
      Exception.__init__( self )

FormattedCell = namedtuple( 'FormattedCell', 'content nCols format' )

class CellInternal:
   def reset( self ):
      # pylint: disable-msg=W0201
      self.wrapWidth_ = None
      self.wrappedText_ = None
      # pylint: enable-msg=W0201
   def __init__( self, rawCell ):
      string = str( rawCell )
      self.strings_ = string.split( "\n" )
      self.reset()
   def __len__( self ):
      return( max( map( len, self.strings_ ) ) )
   def __str__( self ):
      return( "<CellInternal: " + "\n".join( self.strings_ ) + ">" )
   def height( self ):
      return( max( len( self.wrappedText_ if self.wrappedText_ \
                           else self.strings_ ), 1 ))
   def string( self, rowNum=0 ):
      if self.wrappedText_:
         if 0 <= rowNum < len( self.wrappedText_ ):
            return( self.wrappedText_[ rowNum ] )
         else:
            return( "" )
      else:
         if 0 <= rowNum < len( self.strings_ ):
            return( self.strings_[ rowNum ] )
         else:
            return( "" )
   def wrap( self, wrapWidth ):
      if self.wrapWidth_ != wrapWidth: # pylint: disable=too-many-nested-blocks
         self.wrapWidth_ = wrapWidth # pylint: disable-msg=W0201
         self.wrappedText_ = []      # pylint: disable-msg=W0201
         for outputLine in self.strings_:
            if len( outputLine ) <= wrapWidth:
               self.wrappedText_.append( outputLine )
            else:
               # Need to split the entry into multiple lines. We attempt
               # to do this based on whitespace separate in the entry.
               words = outputLine.split( )
               while words:
                  # At least one (possibly truncated) word per line
                  line = words.pop( 0 )
                  while words:
                     word = words.pop( 0 )
                     # pylint: disable-next=no-else-break
                     if len( line ) + len( word ) + 1 > wrapWidth:
                        # Can't add the next word
                        words = [ word ] + words
                        break
                     else:
                        line += " " + word
                  if len( line ) > wrapWidth: 
                     # first word on line was too long
                     words = [ line[ wrapWidth-1 : ] ] + words
                     line = line[ : wrapWidth-1 ] + "\\"
                  self.wrappedText_.append( line )

class TableFormatter:
   """TableFormatter class :

      TableFormatter formats data in tabular format without requiring the
      caller to know, in advance, the number or content (e.g. width) of the
      columns/rows in the table.

      Some Cli 'show' commands, for example 'show lacp internal' and 'show
      lacp port', output different sets of columns depending on Cli
      arguments (brief, all-ports, ...).  Other commands have columns that
      are often all 0's, or all ''s, but sometimes can be long
      strings. ('Up' vs. 'MuxStateCollectingDistributing').  If there are
      two or more such columns, it is possible that the table expands so
      that it is too wide for the line-width.

      In all of these cases there are no good, fixed, set of column widths
      one can apply to the table.  They all depend on the specific data
      being output in the table.

      TableFormatter.output() will return a string displaying the tabular
      data. It will automatically choose appropriate column widths, and in
      the rare occasions when it is necessary, it will split the output into
      two (or more) tables.      
      
      For examples on how to use TableFormatter, please take a look at :

               a) src/Ark/test/TableFormatterTest.py
               b) src/Lag/CliPlugin/LagCli.py
               c) src/Mlag/CliPlugin/MlagShowCli.py
               d) src/Stp/CliPlugin/StpShowLib.py

      The simplest way to use TableFormatter is as follows :

          table = createTable() 
          table.newRow( row content here )
          table.newRow( row content here )
          ...
          print table.output()

          For example: 

          t = createTable( ("heading1", "heading2", "heading3", 
                            "heading4", "heading5") )

          for key in ...:
             t.newRow( key, *values[key] )
          print t.output()

          There are plenty of ways to tune the table. The recommended way is to use 
          Format; Check Format desc string to see how to use it with TableFormatter

      If tableWidth is None (the default), the class will use the value of the 
      COLUMNS environment variable, if set.  Otherwise, it will try to figure out 
      how wide the terminal is, and use that as the table width. If no terminal is
      attached (for instance because the output is redirected to a file) then the
      table is made 80 columns wide.
   """

   def __init__( self, indent=0, tableWidth=None ):
      if not tableWidth:
         tableWidth = terminalWidth()
      self.indent_ = indent
      self.width_ = tableWidth - indent
      self.byRowOrColumn_ = None # can be either None, or "row" or "column"
      # if by row, then rows have cells and columns have Formats.
      self.activeEntry_ = -1
      self.entries_ = []
      self.columnFormats_ = []
      # Only care about isHeading and border.  If we ever deal with
      # variable height, then something to think about
      self.rowFormats_ = []
      self.multiColumnCells_ = {}
      self.columnBreaks_ = []

   def printableWidth( self ):
      return( self.width_ )

   def formatColumns( self, *formats):
      """A list of Formats, one for each column.  A None can be used
      in place of a Format to specify all defaults"""
      # Make sure we get a local, mutatable copy of format for each col
      self.columnFormats_ = list( map( FormatInternal, formats ) )

   def formatRows( self, *formats):
      """A list of Formats, one for each row.  A None can be used
      in place of a Format to specify all defaults"""
      # Make sure we get a local, mutatable copy of format for each row
      self.rowFormats_ = list( map( FormatInternal, formats ) )

   def startRow( self, rowFormat=None ):
      descript = "Must define a table either by rows or by columns, not both."
      assert self.byRowOrColumn_ != "column", descript
      self.byRowOrColumn_ = "row"
      self._startEntry()
      if rowFormat:
         formatLen = len( self.rowFormats_ )
         nRows = self.activeEntry_ + 1
         self.rowFormats_ += ( [ None ] * (nRows - formatLen ))
         self.rowFormats_[ self.activeEntry_ ] = FormatInternal( rowFormat )

   def startColumn( self, colFormat=None ):
      descript = "Must define a table either by rows or by columns, not both."
      assert self.byRowOrColumn_ != "row", descript
      self.byRowOrColumn_ = "column"
      self._startEntry()
      if colFormat:
         formatLen = len( self.columnFormats_ )
         nColumns = self.activeEntry_ + 1
         self.columnFormats_ += ( [ None ] * (nColumns - formatLen ))
         self.columnFormats_[ self.activeEntry_ ] = FormatInternal( colFormat )

   def _startEntry( self ):
      self.activeEntry_ += 1
      self.entries_.append( list() ) # pylint: disable=use-list-literal

   def newRow( self, *cell ):
      self.startRow()
      for item in cell:
         if isinstance( item, FormattedCell ):
            self.newFormattedCell( item.content, item.nCols, item.format )
         else:
            self.newCell( item )
   
   def newColumn( self, *cell ):
      self.startColumn()
      self.entries_[ self.activeEntry_ ] += list ( map( CellInternal, cell ) )

   def newCell( self, *cell ):      
      self.entries_[ self.activeEntry_ ] += list( map( CellInternal, cell ) )

   def newFormattedCell( self, cell, nCols=1, 
                         format=None ): # pylint: disable-msg=W0622
      """Creates Multi-Column cells. Multi-Column cells span across 
         many columns. Arguments specify the number of columns that this 
         cell should span and also the format. Multi-Column cells can 
         have a format (e.g., justification etc.,) which is independent
         of other cells in the same column."""
      exp = "Multi-column cells only supported when defining table by rows."
      assert self.byRowOrColumn_ == "row", exp
      startCol = len( self.entries_[ self.activeEntry_ ] )
      self.multiColumnCells_[( self.activeEntry_, startCol )] = \
                               ( CellInternal( cell ), 
                                 nCols, 
                                 MultiCellFormatInternal( self, startCol,
                                                          nCols, format ) )
      if( nCols == 1 ) and ( format==None ): # pylint: disable=singleton-comparison
         self.newCell( cell )
      elif( nCols == 1 ):
         self.newCell( cell )
      else:
         # We'll deal with widths in a second pass.
         self.newCell( "" )
      for i in range( 1, nCols ):
         self.multiColumnCells_[( self.activeEntry_, startCol+i )] = None
         self.newCell( "" )

   def _rowsAndColumns( self ):
      default = CellInternal( "" )
      if self.byRowOrColumn_ == "column":
         columns = self.entries_
         nRows = functools.reduce( max, map( len, self.entries_ ), 0 )
         rows = [ list( map ( lambda col:
                              # pylint: disable-next=cell-var-from-loop
                              _listGet( col, i, default ),
                              self.entries_ ) )
                  for i in range( nRows )]
      elif self.byRowOrColumn_ == "row":
         rows = self.entries_
         nCols = functools.reduce( max, map( len, self.entries_ ), 0 )
         columns = [ list( map( lambda row :
                                 # pylint: disable-next=cell-var-from-loop
                                 _listGet( row, i, default ),
                                 self.entries_ ) )
                     for i in range( nCols ) ]
      else: # Table not initialized yet
         rows = []
         columns = []
      return( ( rows, columns ) )

   def _dotAlignSplitColumns( self, columns, col_index ):
      """Returns the largest total width and the largest individual widths
      for each of the three columns M, N, and U when a string is formatted
      as M.N U (eg: 10.56 kbps). Strings that do not follow this format are
      ignored for M, N, and U.

      Example:
      Input column: ["24.2 Mbps", "3 kbps", "1.111", "bad_string"]
      Return value: 10, 2, 3, 4
      """
      width = 0
      isDecimal = False # True if at least one num has a decimal
      hasUnit = False # True if at least one row has a unit
      mw, nw, uw = 0, 0, 0 # width of M, N, and U in M.N U
      for cell in columns[ col_index ]:
         try:
            m, n, u = _splitDotAlign( cell.string() )
            isDecimal |= len( n ) > 0
            hasUnit |= len( u ) > 0
            mw = max( mw, len( m ) )
            nw = max( nw, len( n ) )
            uw = max( uw, len( u ) )
            width = max( width, mw + isDecimal + nw + hasUnit + uw )
         except ValueError:
            width = max( width, len( cell.string() ) )
      return width, mw, nw, uw, isDecimal, hasUnit

   def _initColumnFormats( self, columns ):
      lf = len( self.columnFormats_ )
      l = len( columns )
      if lf < l:
         self.columnFormats_ += [ None ] * ( l - lf )
      elif lf > l:
         for col in range( l, lf ):
            colFormat = self.columnFormats_[ col ]
            colFormat.reset()
            colFormat.widthIs( 0 )
      prevBorder = False
      for col in range( l ):
         width = functools.reduce( max, map( len, columns[ col ] ), 0 )
         colFormat = self.columnFormats_[ col ] or  FormatInternal()
         colFormat.reset() # reset computed values to initial state
         # handle dotAlign if applicable
         if colFormat.format_.dotAlign_:
            width, mw, nw, uw, hasDot, hasUnit = \
                  self._dotAlignSplitColumns( columns, col )
            colFormat.mnu_ = FormatMnuInternal( mw, nw, uw, hasDot, hasUnit )
         self.columnFormats_[ col ] = colFormat
         colFormat.widthIs( width )
         colFormat.leftBorder_ = prevBorder
         prevBorder = colFormat.border
      self._multiColumnCells()
      self._computeColumnBreaks()

   def _columnFormats( self, activeCols ):
      # distribute widths and padding properly
      self._handleWidthDistribution( activeCols )
      self._breakMultiColumnCells()
      return( self.columnFormats_ )

   # This depends on _columnFormats running first, and multiColumnCells
   # being set up correctly.  In the future it will need to know what width 
   # to wrap the lines to.  For now, though, we only
   # wrap to an explicit maxWidth, so only multi-column cells relevant.
   def _initRowFormats( self, rows ):
      lf = len( self.rowFormats_ )
      l = len( rows )
      if lf < l:
         self.rowFormats_ += [ None ]*(l - lf)
      elif lf > l:
         for row in range( l, lf ):
            rowFormat = self.rowFormats_[ row ]
            rowFormat.reset()
            rowFormat.numRowsIs( 1 )

   def _rowFormats( self, rows, activeCols ):
      for row in range( len( rows ) ): # pylint: disable=consider-using-enumerate
         nCols = len( rows[ row ] )
         for col in range( nCols ):
            cell, cellFormat = \
                self._cellAndFormat( row, col, self.columnFormats_, rows[row] )
            if cellFormat and cell and cellFormat.wrap():
               cell.wrap( cellFormat.printableWidth() )
         activeRow = [ rows[ row ][ col] for col in range( nCols )
                       if col in activeCols ]\
                          if activeCols else rows[ row ]
         height = functools.reduce( max, map( CellInternal.height,
                                                    activeRow ), 1 )
         rowFormat = self.rowFormats_[ row ] or  FormatInternal()
         rowFormat.reset() # reset computed values to initial state
         self.rowFormats_[ row ] = rowFormat
         rowFormat.numRowsIs( height )
      return( self.rowFormats_ )

   def _columnBreakIn( self, start, end ):
      for c in self.columnBreaks_:
         if c > start and c < end: # pylint: disable=chained-comparison
            return( c )
      return( False )

   def _distribute( self, excess, columns, padding=True ):
      # if columns is empty, this is a no-op
      if not columns:
         return
      ncols = len( columns )
      if padding and ( excess > ncols*6 ):
         # too much whitespace makes the tables look ridiculous
         excess = ncols*6
      remState = [ excess, ncols ]
      def share( ):
         g = math.floor( remState[0]/float(remState[1]))
         remState[1] -= 1
         remState[0] -= g
         return( int( g ))
      cfs = [ self.columnFormats_[ x ] for x in columns ]
      if padding:
         for f in cfs:
            f.newPadding( share() )
      else:
         for f in cfs:
            f.widthIs( f.printableWidth() + share() )

   # All columns must agree on whether they are headers or not. Check?
   def _multiColumnCells( self ):
      for key, entry in self.multiColumnCells_.items():
         if entry: # ignore empty placeholder cells
            _row, col = key
            string, ncols, cellFormat = entry
            cellFormat.reset()
            cellFormat.leftBorder_ = self.columnFormats_[col].leftBorder_
            cellFormat.widthIs( len( string ) )
            mLen = cellFormat.printableWidth()
            cLen = functools.reduce( lambda x, y: x+y,
                                     map( lambda x: x.printableWidth() + 1,
                                          self.columnFormats_[ col:col + ncols ] ),
                  0 ) - 1
            if( cLen > mLen ):
               cellFormat.maxWidth_ = None
               cellFormat.widthIs( cLen )
            elif( cLen < mLen ):
               columns = list( range( col, col + ncols ) )
               # need to widen the columns to make room for the multi-column.
               self._distribute( mLen - cLen, columns, padding=False )

   # All columns must be headers or not. Check?
   # Not only breaks the cells, but expands if padding was added...
   def _breakMultiColumnCells( self ):
      # pylint: disable=unused-argument
      def handleCell( row, col, ncols, cellFormat, eol=False ):
         cellLen = cellFormat.printableWidth()
         cLen = functools.reduce( lambda x, y: x+y,
                                  map( lambda x: x.printableWidth() + 1,
                                       self.columnFormats_[ col:col + ncols ] ),
               0 ) - 1
         cFullLen = functools.reduce( lambda x, y: x+y,
                                      map( lambda x: x.width() + 1,
                                           self.columnFormats_[ col:col + ncols ] ),
               0 ) - 1
         newCellLen = max( cLen, cellLen )
         cellFormat.widthIs( newCellLen )
         if( cFullLen >= cellFormat.width() ):
            cellFormat.maxWidth_ = None
         cellFormat.padding_ = 0
         cellFormat.newPadding( max( 0, cFullLen-cellFormat.width() ))
         if( cFullLen < cellLen ):
            # This is going to be ugly.  Must have split in the middle, and
            # not enough space for multiColumnCell.
            if not eol: # may have to truncate
               cellFormat.padding_ = 0
               cellFormat.widthIs( cFullLen )
      headings = [ i for i in range( 0, len( self.columnFormats_ ))
                   if self.columnFormats_[i].isHeading ]
      newCells = {}
      remCells = []
      for key, entry in self.multiColumnCells_.items():
         row, col = key
         if entry: # ignore empty placeholder cells
            string, ncols, cellFormat = entry
            lineBreak = self._columnBreakIn( col, col+ncols )
            if lineBreak:
               cellFormat.widthIs( len( string ) )
               ncols1 = lineBreak - col
               eol = not headings or ( max( headings ) < ( col+ncols ))
               handleCell( row, col, ncols1, cellFormat, eol=eol )
               if cellFormat.noBreak:
                  for rcol in range( lineBreak, col+ncols ):
                     remCells.append( ( row, rcol ) )
               else:
                  ncols2 = ncols-ncols1
                  f = Format( justify=cellFormat.justify_,
                              border=cellFormat.border )
                  copy = MultiCellFormatInternal( self, lineBreak, ncols2, f )
                  copy.widthIs( len( string ) )
                  entry = ( string, ncols2, copy )
                  newCells[ ( row, lineBreak ) ] = entry
                  handleCell( row, lineBreak, ncols2, copy )
            else: # may have had padding added
               handleCell( row, col, ncols, cellFormat )

      for key in remCells:
         del self.multiColumnCells_[ key ]
      self.multiColumnCells_.update( newCells )

   def _computeColumnBreaks( self ):
      self.columnBreaks_ = [ 0 ]

      # include headings first
      headings = [ i for i in range( 0, len( self.columnFormats_ ))
                  if self.columnFormats_[i].isHeading ]
      hwid = functools.reduce( lambda x, y: x+y,
                     map( lambda x:  1 + self.columnFormats_[ x ].compressedWidth(),
                          headings ),
            0 )
      wid = hwid
      first = True
      # pylint: disable-next=consider-using-enumerate
      for c in range( len( self.columnFormats_ ) ):
         if c in headings: continue    # pylint: disable-msg=C0321
         fmt = self.columnFormats_[ c ]
         width = 1 + fmt.compressedWidth()
         # if the last character is ' ' we can go past the width by 1
         extraWidth = 1 if not ( fmt.border or fmt.noTrailSpace ) else 0
         if wid + width > extraWidth + self.width_ and not first:
            self.columnBreaks_.append( c )
            wid = hwid + width
         else:
            wid += width
            first = False

   def _setWrapMinWidths( self, columns ):
      def computeShare():
         g = math.floor( remState[0] / float( remState[1] ) )
         remState[1] -= 1
         remState[0] -= g
         return( int( g ) )
      minWrap = lambda f: f.minWidth() and f.wrap()
      cols = [ i for i in columns if minWrap( self.columnFormats_[ i ] ) ]
      compressedWidth = sum( 1 + self.columnFormats_[ x ].compressedWidth()
                             for x in columns )
      pool = self.width_ - compressedWidth
      newWidths = {}
      for x in cols:
         newWidths[ x ] = self.columnFormats_[ x ].minWidth()
      while pool > 0 and len( cols ) > 0:
         remState = [ pool, len( cols ) ]
         for x in cols:
            f = self.columnFormats_[ x ]
            share = computeShare()
            pool -= share
            newWidth = newWidths[ x ] + share
            if newWidth >= f.printableWidth():
               pool += newWidth - f.printableWidth()
               cols.remove( x )
            else:
               newWidths[ x ] = newWidth
      for x, newWidth in newWidths.items():
         self.columnFormats_[ x ].widthIs( newWidth )

   def _handleWidthDistribution( self, columns ):
      # reset headings padding
      for x in columns:
         f = self.columnFormats_[ x ]
         if f.isHeading:
            f.padding_ = f.minPadding

      splitTableWidth = functools.reduce( lambda x, y: x + y,
            [ 1 + self.columnFormats_[ x ].width()
               for x in columns ],
            0 )
      if splitTableWidth > self.width_:
         # fairly set new width's for wrappable columns with minWidth_ specified
         self._setWrapMinWidths( columns )
      else:
         excess = self.width_ - splitTableWidth
         if excess > 0:
            # distribute excess whitespace, but do not include last column if it was
            # last column in a row not ending in a column break
            newColumns = columns
            if columns and columns[ -1 ] >= self.columnBreaks_[ -1 ]:
               newColumns = columns[ : -1 ]
            self._distribute( excess, newColumns )

   def _columnsPerRow( self, startColumn, stopColumn ):
      progress = False
      # include headings first
      columns = [ i for i in range( 0, len( self.columnFormats_ ))
                  if self.columnFormats_[i].isHeading ]
      for c in range(startColumn, stopColumn or len( self.columnFormats_ )):
         if c in columns: continue    # pylint: disable-msg=C0321
         columns.append(c)
         progress = True
      if not progress:
         # nothing other than headers to print
         return( [] )
      else:
         return( sorted( columns ))

   def _cellAndFormat( self, rowNum, colNum, formats, cells ):
      key = ( rowNum, colNum )
      if key in self.multiColumnCells_:
         entry = self.multiColumnCells_[ key ]
         if entry:
            return( entry[0], entry[2] )
         else:
            # if entry is None, then it was handled by a previous column,
            # so nothing to do
            return( CellInternal( "" ), None )
      else:
         if colNum < len( cells ):
            return( cells[ colNum ], formats[ colNum ] )
         else:
            return( CellInternal( "" ), formats[ colNum ] )

   def _formatCell( self, rowNum, colNum, formats, cells, perCellRow=0 ):
      string = ""
      rowFormat = self.rowFormats_[ rowNum ]
      cell, colFormat = self._cellAndFormat( rowNum, colNum, formats, cells )
      if colFormat is None: # covered by previous multi-column cell
         return ""
      if colFormat.align_ == "top":
         r = perCellRow
      elif colFormat.align_ == "bottom":
         r = perCellRow + cell.height() - rowFormat.height()
      else: # "middle"
         r = perCellRow + ( cell.height() - rowFormat.height() ) // 2
      string += colFormat.format( cell.string( r ))
      if colFormat.border:
         if colFormat.borderStartPos_ <= rowNum:
            string += "|"
         else:
            string += " "
      elif (colFormat.noTrailSpace==False): # pylint: disable=singleton-comparison
         string += " "
      if colFormat.terminateRow_: # pylint: disable=no-else-raise
         raise _TerminateRow( string )
      else:
         return string
   # I know that I don't use self here...
   # pylint: disable-msg=R0201
   def _resetCells( self, cellLists ):
      for colOrRow in cellLists:
         for cell in colOrRow:
            cell.reset()
   # pylint: enable-msg=R0201

   def output( self ):
      tableString = ""
      rows, columns = self._rowsAndColumns()
      nColumns = len( columns ) # pylint: disable-msg=W0612
      nRows = len( rows ) 
      self._resetCells( columns )
      # Really unnecessary, because non-trivial cells should be same as in cols
      self._resetCells( rows ) 
      self._initColumnFormats( columns )
      self._initRowFormats( rows )
      indent = " " * self.indent_
      # pylint: disable-next=too-many-nested-blocks,consider-using-enumerate
      for i in range( len( self.columnBreaks_ )):
         startColumn = self.columnBreaks_[ i ]
         stopColumn = ((i < len(self.columnBreaks_) - 1) and
                       self.columnBreaks_[i+1] )
         # if table is too wide to print on a single line, we break the table
         # up into multiple tables (but preserve all header columns).
         activeColumns = self._columnsPerRow( startColumn, stopColumn )
         if not activeColumns:
            # Cannot be split. Return the current columns and formats
            # without any adjustment
            activeColumns = range( 0, len( self.columnFormats_ ) )
            columnFormats = self.columnFormats_
         else:
            columnFormats = self._columnFormats( activeColumns )
         # Any multiline cells that need handling?
         rowFormats = self._rowFormats( rows, activeColumns )
         for rowNum in range( nRows ):
            rowFormat = _listGet( rowFormats, rowNum )
            row = rows[ rowNum ]
            for rn in range( rowFormat.height() ):
               rowstr = indent
               border = indent
               for colNum in activeColumns:
                  try:
                     col = self._formatCell( rowNum, colNum, columnFormats, row,
                                                 perCellRow=rn )
                  except _TerminateRow as r:
                     col = r.row
                     rowstr += col
                     break # skip rest of columns
                  rowstr += col
                  if len( col ): # pylint: disable=use-implicit-booleaness-not-len
                     if rowFormat.borderStartPos_ <= colNum:
                        borderChar = "-"
                     else:
                        borderChar = " "
                     # Add the last character of col, in case it ends in '|'.
                     if col[ -1 ] == '|' or col[ -1 ] == ' ':
                        border += borderChar * ( len( col ) - 1 ) + col[ -1 ]
                     else:
                        border += borderChar * len( col )
               # Trim off a trailing blank in case the columns barely fit the width
               if rowstr and rowstr[ -1 ] == ' ':
                  rowstr = rowstr[ : -1 ]
               tableString += rowstr + "\n"
            if( rowFormat and rowFormat.border ):
               if border and border[ -1 ] == ' ':
                  border = border[ : -1 ]
               tableString += border + "\n"
         # needs to be 1+largest *NON-HEADER* column, not just largest column
         if activeColumns: # ignore if there are no non-headers.
            _startingColumn = activeColumns[ -1 ] + 1
            if( i < len( self.columnBreaks_ ) - 1 ):
               tableString += "\n"
      return( tableString )


justifiers = set("lcr")
valid = set("hblcr")

def parseFormatString( formatString, default="right" ):
   """ Format strings for headings and subheadings can be passed as a tuple
       argument to Headings.  
       For example, it is possible to create Headings with column formats like :

       ( ("headings1", "r"), "headings2", ("headings3", "c", ("sub1", "sub2")) )

        instead of creating headings and then formats and adding those formats 
        to TableFormatter. However this works only for those options recognized
        by parseFormat string. Format offers much more options to finetune.""" 
   if not formatString:
      return( None )
   directives = set(formatString)
   exp = "'hblcr' are the only valid format specifiers"
   assert directives.issubset( valid ), exp
   header = "h" in formatString
   border = "b" in formatString
   jset = set( formatString ).intersection( justifiers)
   assert len( jset)<=1,"Can only specify one of l, c, or r"
   if len( jset ) == 1:
      j = { "l": "left", "c": "center", "r": "right"}[jset.pop()]
   else:
      j = default
   return( Format( isHeading=header, border=border, justify=j ))

class Headings:
   """Front-end to make the common case of headings and subheadings
      a little friendlier.

      Common case of specifying headings for table should be easy.
      The following front-end seems to capture the common cases.
      Pass in a list of column heading entries.  A columnHeadingEntry is either a
      string, or a tuple of string followed by an optional formatString,
      followed by an optional tuple of subheadingEntries.
      The format string can contain one of "lcr" (for left center right), as
      well as "h" and/or "b" (for isHeading and border).
      Each subheadingEntry has the same format as a columnHeadingEntry,
      recursively

      For example, you can create headings and subheadings like :

      ( "heading1", "heading2", ( "heading3", ( "subheading1", "subheading2" ) ), 
        "heading4" )

      Then, subheading1 and subheading2 would be formatted on a line below 
      heading3 as subheadings. Their depth is calculated automatically
      and formatted.
      """
   def __init__( self, description ):
      self.formats = []
      self.colHeaders = [ [] ]
      self.headings = description
      self._mapHeadings( self.headings )

   # The only external method:
   def doApplyHeaders( self, table ):
      table.formatColumns( *self.formats )

      for row in reversed( self.colHeaders ):
         table.startRow( Format( isHeading=True,
                                 border=(row==self.colHeaders[0]) ))
         for cell in row:
            if isinstance( cell, str ):
               table.newCell( cell )
            else:
               table.newFormattedCell( cell[0],
                                       nCols=cell[1],
                                       format=cell[2] )

   def _mapHeadings( self, headings, entry=None ):
      totalCols = 0
      maxDepth = 1
      for subheading in headings:
         nCols, depth = self.process( subheading )
         totalCols += nCols
         maxDepth = max( maxDepth, depth )
         self.normalizeHeadingDepth( depth, maxDepth,
                                     # pylint: disable-next=singleton-comparison
                                     nCols, totalCols, entry==None )
      if entry:
         fmat = (( len( entry ) == 3 and # pylint: disable=consider-using-ternary
                   parseFormatString( entry[1], default="center" )) or
                 Format( justify="center"))
         self.colHeaders[ maxDepth ].append( ( entry[0], totalCols, fmat ))
         maxDepth += 1
      return( totalCols, maxDepth )

   def normalizeHeadingDepth( self,
                              thisDepth, totalDepth,
                              thisCols, totalCols,
                              isTop):
      if( thisDepth < totalDepth ):
         for d in range( thisDepth, totalDepth ):
            self.colHeaders[d] += [ "" ]*thisCols
      elif( not isTop ) and ( len(self.colHeaders) <= totalDepth ):
         assert len(self.colHeaders) == totalDepth
         startCol = len( self.colHeaders[0] ) - totalCols
         self.colHeaders.append( [ "" ]*startCol )

   def process( self, entry ): # an entry is either string or tuple
      "Peels off columns and returns ( nCols, subheadingDepth )"
      if isinstance( entry, str ):
         self.colHeaders[ 0 ].append( entry )
         self.formats.append( None )
         return( (1, 1) )
      else:
         # pylint: disable-next=consider-merging-isinstance
         assert( isinstance( entry, tuple ) or
                 isinstance( entry, list ))
         subheadings = len( entry ) > 1 and entry[-1]
         # pylint: disable-next=consider-merging-isinstance
         if( isinstance( subheadings, tuple ) or
             isinstance( subheadings, list )):
            return( self._mapHeadings( subheadings, entry=entry ))
         else:
            self.colHeaders[0].append( entry[ 0 ])
            self.formats.append( parseFormatString( len(entry) > 1 and
                                                    entry[1] ))
            return( (1, 1) )



def createTable( description, indent=0, tableWidth=None ):
   """Frontend to create a TableFormatter with headings
      For creating a TableFormatter with no headings
      use TableFormatter() directly
   """
   table = TableFormatter( indent, tableWidth )
   header = Headings( description )
   header.doApplyHeaders( table )
   return table
