#!/usr/bin/env python3
# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import Tracing
from TypeFuture import TacLazyType

__defaultTraceHandle__ = Tracing.Handle( 'TaskScheduler' )
t8 = __defaultTraceHandle__.trace8

TacTask = TacLazyType( 'Ark::Task' )
TacTimerTask = TacLazyType( 'Ark::TimerTask' )
TaskPriority = TacLazyType( 'Ark::TaskPriority' )
TaskPriorityEnum = TacLazyType( 'Ark::TaskPriority::PriorityEnum' )
TaskSchedulerRoot = TacLazyType( 'Ark::TaskSchedulerRoot' )

class TaskRunResult:
   '''
   Returned by a task run function to indicate:
   - shouldReschedule: whether the task needs to run again
   - workDone: number of units of work completed
   '''
   def __init__( self, shouldReschedule, workDone ):
      self.shouldReschedule = shouldReschedule
      self.workDone = workDone

class Task( Tac.Notifiee ):
   '''
   A very simple python interface for Ark::Task.
   This task can be scheduled and will call 'runFunction' until that function
   indicates there is no more work to do.

   ```
   runFunction( callable: shouldYield ) -> TaskRunResult
   ```
   runFunction should regularly call shouldYield and return if true. This is to
   ensure we don't overrun the time quantum for the task.
   runFunction should return an object of type TaskRunResult indicating if the task
   needs to be scheduled again (shouldReschedule) because it yielded before all
   work was complete and what amount of work has been completed in the run just
   finished for stat book keeping (workDone).

   If this simple class is not suficient for future requirements we should move on to
   using Ark::WalkSmBase, however that will require changes to Ark for a genericIf.
   '''
   notifierTypeName = 'Ark::Task'
   tacType = TacTask

   def __init__( self, name, runFunction, scheduler=None, priority=None ):
      self.name = name
      self.runFunction = runFunction
      scheduler = scheduler or TaskSchedulerRoot.findOrCreateScheduler()
      if priority is None:
         priority = TaskPriority( TaskPriorityEnum.normal )
      self.task = self.tacType( scheduler, priority, self.name )
      super().__init__( self.task )

   def schedule( self ):
      self.task.scheduleTask()

   def suspend( self ):
      self.task.taskSuspended = True

   def shouldYield( self ):
      return self.task.taskShouldYield()

   @property
   def taskForceYieldInTest( self ):
      return self.task.taskForceYieldInTest

   @taskForceYieldInTest.setter
   def taskForceYieldInTest( self, value ):
      self.task.taskForceYieldInTest = value

   @property
   def busy( self ):
      return self.task.taskBusy

   @Tac.handler( 'taskRun' )
   def handleRun( self ):
      runResult = self.runFunction( self.shouldYield )
      assert isinstance( runResult, TaskRunResult )
      self.task.taskWorkCountInc( runResult.workDone )
      if runResult.shouldReschedule:
         self.schedule()
      t8( 'Task', self.name, 'processed', runResult.workDone,
          'units, rescheduling:', runResult.shouldReschedule )

class TimerTask( Task ):
   notifierTypeName = 'Ark::TimerTask'
   tacType = TacTimerTask

   def currentExpirationTime( self ):
      return self.task.currentExpirationTime()

   def scheduleTaskDelay( self, delayIntoFuture ):
      self.task.scheduleTaskDelay( delayIntoFuture )

   def scheduleTaskLowestDelay( self, delayIntoFuture ):
      self.task.scheduleTaskLowestDelay( delayIntoFuture )

   def stopTimerTask( self ):
      self.task.stopTimerTask()
