# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""Utility to generate test values

Provide 2 classes easing the enabling of the contentcheck rule noRandomInTest:
   - Arbitrary: used in btest, provide the random API with deterministic values
   - RandomOutOfBtest: used for stest/btest hybrid. Behave as Arbitrary in a btest,
         as random.Random in an stest or ptest.

Expansion with other test value generators that can be widely used is welcomed
"""

from random import Random
import sys
import zlib
import os

class EmptySeedError( Exception ):
   def __init__(
         self, message="the seed must be a non-empty string" ):
      self.message = message
      super().__init__( message )

def _hash( data ):
   """The built-in hash is randomized when the -R option is used, so this a
   deterministic implementation"""
   return zlib.adler32( str( data ).encode( 'utf8' ) )


class Arbitrary( Random ):
   """Generate deterministic test values using Random's API

   Hardcoded arbitrary values is usually prefered in unit tests,
   but sometime we need a large number of different values that are simply
   impractical to hardcode. This class is here to answer that need.

   The prefered way to use it is to create an instance per file or per test. Do NOT
   import an Arbitrary from another file.

   If using this class to replace existing uses of random, make sure that we
   are not losing coverage: are the corner cases and boundary systematically tested?
   If it appears that randomness is needed for coverage, consider rewriting
   the test, or use RandomOutOfBtest instead and schedule the test as an stest.

   Constructor argument
   --------------------
   - seed: provide a unique string, which will be used as a seed
      The name of the file or the name of the test are both good choice
      Do NOT use the variable __file__ as it can change depending on the current
      working directory

   """
   def __init__( self, seed ):
      if not seed:
         raise EmptySeedError
      super().__init__( _hash( seed ) )


class RandomOutOfBtest( Random ):
   """Generate random or arbitrary values depending on the context

   This class is here to allow for hybrid test that are deterministic as a btest
   and randomized as an stest.

   In a btest context, behave as Arbitrary. Otherwise, behave as the stdlib Random.

   The prefered way to use it is to create an instance per file or per test. Do NOT
   import an Arbitrary from another file.

   In a test that will only be a btest, prefer Arbitrary so that we don't need to
   add NORANDOM=1 when running the test manually.

   If using this class to replace existing uses of random, make sure that the
   test is indeed scheduled as an stest to avoid coverage loss. It might also
   be worthwhile to rewrite the test to not count on randomness for coverage.

   The btest context is detected through environment variable. Either:
   - TEST_RUNNING_UNDER_QUBE, automatically set by Qube
   - NORANDOM=1, for manual test execution. It can also be set in the Makefile of
   packages not using Qube

   CAVEAT: make sure to set NORANDOM=1 in non-qube packages' Makefile or the test
   will always be random

   Constructor argument
   --------------------
   - seed: provide a unique string, which will be used as a seed
      The name of the file or the name of the test are both good choice
      Do NOT use the variable __file__ as it can change depending on the current
      working directory

   """
   _warning_showed = False
   def __init__( self, seed ):
      super().__init__()
      if not seed:
         raise EmptySeedError

      if ( os.environ.get( 'NORANDOM' ) == '1' or
            os.environ.get( 'TEST_RUNNING_UNDER_QUBE' ) == '1' ):
         self.seed( _hash( seed ) )
      else:
         if not self._warning_showed:
            print( "WARNING: A random seed is used "
                   "(ignore if running an stest or a ptest). \n"
                   "If you are running a btest manually, "
                   "set the following environment variable to use the same values "
                   "as a build run: 'NORANDOM=1'",
                   file=sys.stderr )
            # pylint: disable=protected-access
            self.__class__._warning_showed = True


__all__ = [ "Arbitrary", "RandomOutOfBtest" ]
