# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

_TacModule = None

class TacLazyType:
   '''
   TacLazyType is a proxy object that can be used at the module level.

   It is generally a good idea to avoid loading Tac Types into python at the
   module level during module init, especially in python plugins, etc.

   To avoid this slowdown, a TypeFuture.TacLazyType can be used instead.

   from TypeFuture import TacLazyType
   ArnetPrefix = TacLazyType( 'Arnet::Prefix' )
   print ArnetPrefix.attributes
   assert ArnetPrefix() == Tac.Value( 'Arnet::Prefix' )
   assert isinstance( Tac.Value( 'Arnet::Prefix' ), ArnetPrefix )

   returnValueConst: Only compatible with Tac.Value types.
   TypeError will be raised otherwise.

   when returnValueConst = True, returns Tac.ValueConst instance
   instead of Tac.Value when called.

   eg.
   IntfId = TacLazyType( 'Arnet::IntfId' )
   IntfIdConst = TacLazyType( 'Arnet::IntfId', returnValueConst=True )
   { IntfId( 'Ethernet1' ) } # TypeError because IntfId( 'Ethernet1' ) is unhashable
   { IntfIdConst( 'Ethernet1' ) } # No errors since it is a const instance
   '''

   def __init__( self, typeName, returnValueConst=False ):
      self._typeName = typeName
      # Do not access __type, use the self._type property only.
      self.__type = None
      self._returnValueConst = returnValueConst

   def __maybeResolve( self ):
      '''
      __maybeResolve() is used only in the implementation of the _type property
      and will set the state of the internal __type attribute used as a cache
      for the _type property.
      '''
      if self.__type is None:
         global _TacModule
         if _TacModule is None:
            import Tac as _imported # pylint: disable=import-outside-toplevel
            _TacModule = _imported
         self.__type = _TacModule.Type( self._typeName )
         if not self.__type.tacType.isValue and self._returnValueConst:
            # pylint: disable-next=consider-using-f-string
            raise TypeError( '{!r} is not a value type and cannot be used '
                             'with returnValueConst'.format( self._typeName ) )

   @property
   def _type( self ):
      self.__maybeResolve()
      return self.__type

   def __getattr__( self, name ):
      return getattr( self._type, name )

   def __call__( self, *args, **kwargs ):
      inst = self._type( *args, **kwargs ) # pylint: disable=not-callable
      if self._returnValueConst:
         return _TacModule.const( inst )
      else:
         return inst

   def __instancecheck__( self, instance ):
      return isinstance( instance, self._type )

   def __eq__( self, other ):
      return self._type == other

   def __hash__( self ):
      return hash( self._type )
