# Copyright (c) 2006-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=arguments-renamed
# pylint: disable=raise-missing-from
# pylint: disable=consider-using-f-string

import Arnet
import CliParser
import CmdExtension
import LazyMount
import PyWrappers.Curl as curl
import PyWrappers.Lftp as lftp
import SysMgrLib
import Tac
import Tracing
import Url

import os
import re
import socket

t0 = Tracing.trace0

class NetworkUrl( Url.Url ):
   def __init__( self, fs, url, rest, context ):
      import urllib.parse # pylint: disable=import-outside-toplevel
      Url.Url.__init__( self, fs, url )

      # Url loadPlugins does not provide a context for the entityManager.
      # Mounting is done when NetworkUrl object is instantiated.
      em = context.entityManager
      self.networkUrlConfig = LazyMount.mount( em, 'mgmt/networkUrl/config',
                                               'Mgmt::NetworkUrl::Config', 'r' )
      LazyMount.force( self.networkUrlConfig )
      self.ipStatus = LazyMount.mount( em, 'ip/status', 'Ip::Status', 'r' )
      LazyMount.force( self.ipStatus )
      self.ip6Status = LazyMount.mount( em, 'ip6/status', 'Ip6::Status', 'r' )
      LazyMount.force( self.ip6Status )
      self.mgmtSecConfig = LazyMount.mount( em, 'mgmt/security/config',
                                            'Mgmt::Security::Config', 'r' )
      LazyMount.force( self.mgmtSecConfig )

      self.cliSession = context.cliSession

      # The URL didn't begin with '<scheme>://'.  Be nice to the user and prepend
      # it for them, so that we can accept 'http:www.mysite.com/foo' or
      # 'http:/www.mysite.com/foo', for example.  Web browsers tend to do this.
      self.schemeSlashes = ''
      if rest.startswith( '//' ):
         self.schemeSlashes = '//'
         rest = rest[ 2: ]
      elif rest.startswith( '/' ):
         self.schemeSlashes = '/'
         rest = rest[ 1: ]
      rest = '//' + rest

      res = urllib.parse.urlsplit( rest )
      # The pathname gets modified in several places. Easier to store the original
      # for returning the URL as a string rather than trying to reform it.
      self.rawpathname = res.path
      self.pathname = urllib.parse.quote( self.rawpathname )
      self.username = res.username
      self.password = res.password
      self.hostname = '' if res.hostname is None else res.hostname
      self.urlhostname = self.hostname
      self.rawhostname = res.netloc.rpartition( '@' )[ 2 ]
      self.fragment = res.fragment

      # Can't just use res.port because it tries to parse it as an int
      # while this code expects it to just pass along invalid ports too.
      self.port = None
      hostinfo = self.rawhostname
      _, haveOpenBr, bracketed = hostinfo.partition( '[' )
      if haveOpenBr:
         m = re.match( r'[^\]]*][^:]*:(.*)', bracketed )
         if m:
            self.port = m.group( 1 )
      else:
         self.port = hostinfo.partition( ':' )[ 2 ]
      if not self.port:
         self.port = None
      # Note that self.port is a string, not an int.

      try:
         Arnet.IpAddress( self.hostname, addrFamily=socket.AF_INET6 )
         self.urlType = 'ipv6'
         self.urlhostname = f'[{self.hostname}]'
      except ValueError:
         try:
            Arnet.IpAddress( self.hostname )
            self.urlType = 'ipv4'
         except ValueError:
            self.urlType = 'hostname'

      if self.pathname:
         # The initial '/' (if present) is not strictly part of the pathname part of
         # the URL.  We need to remove it in order to allow URLs that are relative to
         # the FTP home directory.  For example, so that 'ftp://host/path' refers
         # to 'path' rather than '/path'.
         #
         # BUG519 It's not clear whether this behavior is (a) RFC-compliant, (b) the
         # same as other implementations, and/or (c) desirable.
         assert self.pathname[ 0 ] == '/'
         self.pathname = self.pathname[ 1: ]

      assert self.pathname is not None
      assert self.hostname is not None
      # Any of the other three attributes (username, password and port) may be None
      # if the field was not present.  Any of pathname, hostname, username or
      # password may be the empty string.
      self.vrfName = None

      # Set to True if you want to run the URL fetch as root user (use with caution)
      self.asRoot = False

   def _getStr( self, sanitized ):
      # Return a string as close to the input URL as possible as this
      # is used by the CLI save plugin. The URL needs to be constructed from
      # parts to obscure the password when sanitized.
      s = self.fs.scheme + self.schemeSlashes
      if self.username is not None:
         s += self.username
         if self.password is not None:
            s += ':' + ( '*' if sanitized else self.password )
         s += '@'
      s += self.rawhostname
      s += self.rawpathname
      if self.fragment:
         s += '#' + self.fragment
      return s

   def __str__( self ):
      return self._getStr( False )

   def aaaToken( self ):
      return self._getStr( True )

   def basename( self ):
      return os.path.basename( self.pathname )

   def isdir( self ):
      return False

   def getIpSrcAddr( self, protocol ):
      return SysMgrLib.getIp4SrcIntfAddr( self.networkUrlConfig,
                                          self.ipStatus,
                                          protocol,
                                          self.vrfName )

   def getIp6SrcAddr( self, protocol ):
      return SysMgrLib.getIp6SrcIntfAddr( self.networkUrlConfig,
                                          self.ip6Status,
                                          protocol,
                                          self.vrfName )

   def setUrlVrfName( self, vrfName ):
      self.vrfName = vrfName

   def setAsRoot( self, asRoot ):
      self.asRoot = asRoot

   def checkBlockedNetworkProtocols( self, protocol ):
      if ( protocol in self.mgmtSecConfig.blockedNetworkProtocols and
           self.mgmtSecConfig.blockedNetworkProtocols[ protocol ] ):
         self.cliSession.mode.addError( "%s client traffic is disabled"
                                         % protocol.upper() )
         raise CliParser.AlreadyHandledError()

   def _getFilterCmd( self, cmdArgv, filterCmd, local, addFdFile=True ):
      cmd = ' '.join( Tac.shellQuote( arg ) for arg in cmdArgv )
      if addFdFile:
         cmd += ' /dev/fd/100'
      return [ "RunPipe", "-o", local, "-d", "100", cmd, filterCmd ]

class LftpUrl( NetworkUrl ):

   def _runLftp( self, remote, local, op, bindCommand, getRoot=True,
                 filterCmd=None ):
      # See BUG7820 about 'getRoot'

      # Add debug attribute to gather information in case of failures
      commands = 'debug 1; '

      # Prevent lftp from helpfully moving itself into the background when it
      # receives a SIGHUP.
      commands += 'set cmd:move-background no; '

      # Set some sensible retry values (the defaults are to retry indefinitely).
      # When running commands from the CLI, we don't need long timeouts - the user
      # can retry the command if it fails.
      commands += 'set dns:fatal-timeout 60; '
      commands += 'set dns:max-retries 2; '
      commands += 'set net:timeout 300; '
      commands += 'set net:max-retries 2; '
      commands += 'set net:reconnect-interval-base 0; '

      # Make lftp terminate if the destination disk is full, rather than hanging
      # indefinitely while it waits for some space to magically appear.
      commands += 'set xfer:disk-full-fatal yes; '

      # Allow lftp to overwrite an existing file, which is necessary because we
      # _always_ ask it to write to a temporary file that we've already created.
      commands += 'set xfer:clobber yes; '

      # Set the default password for anonymous FTP.  This value is chosen because:
      # (a) it's a well-formed email address, which some FTP servers require;
      # (b) example.com is reserved in RFC2606;
      # (c) it doesn't mention "arastra" or "arista" anywhere,
      #     so there's no chance anyone will
      #     come after us for attempting to "hack" their system (see
      #     http://curl.haxx.se/mail/lib-2007-02/0092.html).
      commands += 'set ftp:anon-pass anonymous@example.com; '

      # Don't verify SSL certificates. This was the default in previous versions
      # of EOS. With the current ( 4.6.4 ) lftp upgrade I continue that trend.
      commands += 'set ssl:verify-certificate no; '

      commands += bindCommand

      lftpLocal = local
      if filterCmd:
         lftpLocal = "/dev/fd/100"

      # Plain HTTP get with no authentication. See BUG7820
      if not getRoot and self.username is None:
         assert op == "get"
         commands += 'get %s//%s' % ( self.fs.scheme, self.urlhostname )
         if self.port is not None:
            commands += ':%s' % self.port
         commands += '/%s -o %s' % ( remote, lftpLocal )
      else:
         commands += 'open'

         if self.username:
            commands += ' -u %s' % self.username
            if self.password:
               commands += ',%s' % self.password
            else:
               # lftp will prompt for a password.
               pass
         else:
            # lftp will do an anonymous login.
            pass

         commands += ' %s//%s' % ( self.fs.scheme, self.urlhostname )
         if self.port is not None:
            commands += ':%s' % self.port
         commands += '/ && '
         if op == 'get':
            commands += 'get %s -o %s' % ( remote, lftpLocal )
         else:
            commands += 'put %s -o %s' % ( lftpLocal, remote )

      cmdArgs = [ lftp.name(), '-c', commands ]
      if filterCmd:
         cmdArgs = self._getFilterCmd( cmdArgs, filterCmd, local,
                                       addFdFile=False )

      cliCmdExt = CmdExtension.getCmdExtender()
      try:
         cliCmdExt.runCmd( cmdArgs,
                           self.cliSession,
                           vrfName=self.vrfName,
                           stderr=Tac.CAPTURE )
      except Tac.SystemCommandError as e:
         raise OSError( 0, ( '\n' + str( e.output ) +
                                      '\nsee above for details\n' ) )

   def _lftpGet( self, remote, local, bindCommand, getRoot=True, filterCmd=None ):
      self._runLftp( remote, local, 'get', bindCommand,
                     getRoot=getRoot, filterCmd=filterCmd )

   def _lftpPut( self, local, remote, append, bindCommand,
                 getRoot=True, filterCmd=None ):
      if append:
         self._notSupported()
      self._runLftp( remote, local, 'put', bindCommand,
                     getRoot=getRoot, filterCmd=filterCmd )

   def _lftpBindCommand( self, protocol ):
      bindCommand = ''
      ipSrcAddr = self.getIpSrcAddr( protocol )
      if ipSrcAddr:
         bindCommand += 'set net:socket-bind-ipv4 %s; ' % ( ipSrcAddr )
      ip6SrcAddr = self.getIp6SrcAddr( protocol )
      if ip6SrcAddr:
         bindCommand += 'set net:socket-bind-ipv6 %s; ' % ( ip6SrcAddr )
      return bindCommand

class HttpUrl( LftpUrl ):
   # HTTP URIs are specified in RFC2616.  HTTPS URIs are specified in RFC2818.
   netAccount = 'http'

   def get( self, local ):
      self.checkBlockedNetworkProtocols( 'http' )
      self._lftpGet( self.pathname or '/', local,
                     self._lftpBindCommand( 'http' ), getRoot=False )

   def getWithFilter( self, local, filterCmd ):
      self.checkBlockedNetworkProtocols( 'http' )
      self._lftpGet( self.pathname or '/', local,
                     self._lftpBindCommand( 'http' ), getRoot=False,
                     filterCmd=filterCmd )

   def put( self, local, append=False ):
      self.checkBlockedNetworkProtocols( 'http' )
      self._lftpPut( local, self.pathname or '/', append,
                     self._lftpBindCommand( 'http' ) )

class FtpUrl( LftpUrl ):
   # RFC1738 appears to be the most recent specification of FTP URIs.
   netAccount = 'ftp'

   def get( self, local ):
      self.checkBlockedNetworkProtocols( 'ftp' )
      self._lftpGet( self.pathname or '.', local,
                     self._lftpBindCommand( 'ftp' ) )

   def getWithFilter( self, local, filterCmd ):
      self.checkBlockedNetworkProtocols( 'ftp' )
      self._lftpGet( self.pathname or '.', local,
                     self._lftpBindCommand( 'ftp' ), filterCmd=filterCmd )

   def put( self, local, append=False ):
      self.checkBlockedNetworkProtocols( 'ftp' )
      self._lftpPut( local, self.pathname or '.', append,
                     self._lftpBindCommand( 'ftp' ) )

class TftpUrl( NetworkUrl ):
   # See RFC3617 for the specification of TFTP URIs.  Note that the TFTP protocol
   # doesn't use usernames or passwords, so if a username/password are specified in
   # the URL we ignore them.

   def _curlBindCommandAndHostAddr( self ):
      bindCommand = []
      hostaddr = None
      # curl does not have separate bind option for IPv4 and IPv6 addresses.
      # If the URL contains a numerical address and not a hostname,
      # we need to make sure that the bind address is of the same type.

      ipSrcAddr = self.getIpSrcAddr( 'tftp' )
      ip6SrcAddr = self.getIp6SrcAddr( 'tftp' )
      if self.urlType == 'ipv4':
         if ipSrcAddr:
            bindCommand = [ '--interface', ipSrcAddr ]
      elif self.urlType == 'ipv6':
         if ip6SrcAddr:
            bindCommand = [ '--interface', ip6SrcAddr ]
      else:
         # If URL contains a host name, we do a lookup.
         try:
            addrList = socket.getaddrinfo( self.hostname, None, 0,
                                           socket.SOCK_DGRAM )
         except socket.gaierror:
            raise OSError( 0, "Could not resolve host" )

         # Pick the first address in the list. Bind to matching
         # address type if it is configured on the source interface.
         addr = addrList[ 0 ][ 4 ][ 0 ]
         if ':' in addr:
            if ip6SrcAddr:
               bindCommand = [ '--interface', ip6SrcAddr ]
            hostaddr = "[%s]" % addr
         else:
            if ipSrcAddr:
               bindCommand = [ '--interface', ipSrcAddr ]
            hostaddr = addr
      return bindCommand, hostaddr

   def _runCurlTftp( self, curlArgs ):
      argv = [ curl.name(), '-g' ] + curlArgs
      cliCmdExt = CmdExtension.getCmdExtender()
      try:
         cliCmdExt.runCmd( argv, self.cliSession, vrfName=self.vrfName )
      except Tac.SystemCommandError as e:
         raise OSError( 0, e.output.strip() or "Command error" )

   def _runCurlTftpWithFilter( self, curlArgs, filterCmd, destFile ):
      argv = [ "bash", "-c",
               "set -o pipefail; exec %s -g %s | %s > %s" %
               ( curl.name(),
                 ' '.join( Tac.shellQuote( arg ) for arg in curlArgs ),
                 filterCmd,
                 Tac.shellQuote( destFile ) ) ]
      cliCmdExt = CmdExtension.getCmdExtender()
      try:
         cliCmdExt.runCmd( argv, self.cliSession, vrfName=self.vrfName )
      except Tac.SystemCommandError as e:
         try:
            os.unlink( destFile )
         except OSError:
            pass
         raise OSError( 0, e.output.strip() or "Command error" )

   def get( self, local ):
      self.checkBlockedNetworkProtocols( 'tftp' )
      port = ':%s' % self.port if self.port is not None else ''
      bindArgs, urlhostname = self._curlBindCommandAndHostAddr()
      if urlhostname:
         self.urlhostname = urlhostname
      self._runCurlTftp( [ 'tftp://%s%s/%s' %
                           ( self.urlhostname, port, self.pathname ),
                           '-o', local ] + bindArgs )

   def getWithFilter( self, local, filterCmd ):
      self.checkBlockedNetworkProtocols( 'tftp' )
      port = ':%s' % self.port if self.port is not None else ''
      bindArgs, urlhostname = self._curlBindCommandAndHostAddr()
      if urlhostname:
         self.urlhostname = urlhostname
      # pkgdeps: rpm curl
      # requires curl with TFTP support. curl-minimal does not provide
      # enough functionality
      self._runCurlTftpWithFilter( [ 'tftp://%s%s/%s' %
                                     ( self.urlhostname, port, self.pathname ) ]
                                   + bindArgs,
                                   filterCmd, local )

   def put( self, local, append=False ):
      if self.pathname == '':
         raise OSError(
            0, "A destination filename is required to use TFTP" )
      if append:
         self._notSupported()
      self.checkBlockedNetworkProtocols( 'tftp' )
      port = ':%s' % self.port if self.port is not None else ''
      bindArgs, urlhostname = self._curlBindCommandAndHostAddr()
      if urlhostname:
         self.urlhostname = urlhostname
      self._runCurlTftp( [ 'tftp://%s%s/%s' %
                           ( self.urlhostname, port, self.pathname ),
                           '--upload-file', local ] + bindArgs )

class SshBaseUrl( NetworkUrl ):
   def _sshBindCommandAndHostAddr( self ):
      bindCommand = []
      hostaddr = None

      ipSrcAddr = self.getIpSrcAddr( 'ssh' )
      ip6SrcAddr = self.getIp6SrcAddr( 'ssh' )
      if self.urlType == 'ipv4':
         if ipSrcAddr:
            bindCommand = [ '-o', "BindAddress=%s" % ipSrcAddr ]
      elif self.urlType == 'ipv6':
         if ip6SrcAddr:
            bindCommand = [ '-o', "BindAddress=%s" % ip6SrcAddr ]
      else:
         # If URL contains a host name, we do a lookup.
         try:
            addrList = socket.getaddrinfo( self.hostname, None, 0,
                                           socket.SOCK_STREAM )
         except socket.gaierror:
            raise OSError( 0, "Could not resolve host" )

         # Pick the first address in the list. Bind to matching
         # address type if it is configured on the source interface.
         addr = addrList[ 0 ][ 4 ][ 0 ]
         if ':' in addr:
            if ip6SrcAddr:
               bindCommand = [ '-o', "BindAddress=%s" % ip6SrcAddr ]
            hostaddr = "[%s]" % addr
         else:
            if ipSrcAddr:
               bindCommand = [ '-o', "BindAddress=%s" % ipSrcAddr ]
            hostaddr = addr
      return bindCommand, hostaddr

   def _setupSshArgs(self):
      sshargs = []

      # Disable storing and checking host keys of servers we connect to.
      sshargs += [ '-o', 'StrictHostKeyChecking=no',
                   '-o', 'UserKnownHostsFile=/dev/null',
                   '-o', 'CheckHostIP=no' ]

      # We don't print any output from scp (except for the 'Password:' prompt, which
      # scp manages to write to the tty even though we have redirected stdout and
      # stderr) until after scp has exited.  Therefore, if we allowed multiple
      # password attempts, the user wouldn't see any of the 'Permission denied,
      # please try again' messages until the end, which would be weird.
      sshargs += [ '-o', 'NumberOfPasswordPrompts=1' ]

      bindArgs, urlhostname = self._sshBindCommandAndHostAddr()
      sshargs += bindArgs
      if urlhostname:
         self.urlhostname = urlhostname

      if self.port is not None:
         sshargs += [ '-o', 'Port=%s' % self.port ]

      return sshargs

class ScpUrl( SshBaseUrl ):
   # Note that SCP URIs were specified in draft-ietf-secsh-scp-sftp-ssh-uri-02, but
   # SCP was removed from subsequent revisions of this document (presumably because
   # SCP is not an IETF standard).  scp doesn't allow a password to be specified on
   # the command-line, so if a password is specified in the URL we ignore it.
   netAccount = 'scp'

   def _scpGetOrPut( self, local, op, filterCmd=None ):
      if not self.username:
         raise OSError( 0, "A username is required to use scp" )

      # OpenSSH >= 8.7 use SFTP protocol by default when you invoke scp.
      # It does not provide a way to limit the max "num_requests", and hardcodes
      # it to 64.
      # Use -O to fall back to legacy scp. All this is required because we want
      # to write to a pipe, and need to avoid scp seeking, which is possible
      # only when we don't have parallel requests for blocks of the file
      # pending.
      argv = [ 'scp', '-O' ]

      argv += self._setupSshArgs()

      remoteFile = '%s@%s:/%s' % ( self.username, self.urlhostname,
                                   self.pathname.lstrip( '/' ) )
      # BUG519 The remote file currently always begins with a '/', so we don't
      # support accessing files relative to the user's home directory.

      if op == 'put':
         argv += [ local, remoteFile ]
      else:
         assert op == 'get'
         argv.append( remoteFile )
         if filterCmd:
            argv = self._getFilterCmd( argv, filterCmd, local )
         else:
            argv.append( local )

      cliCmdExt = CmdExtension.getCmdExtender()
      if self.password:
         argv = [ "aScpPass", self.password ] + argv
      try:
         cliCmdExt.runCmd( argv, self.cliSession, stderr=Tac.CAPTURE,
                           vrfName=self.vrfName, asRoot=self.asRoot )
      except Tac.SystemCommandError as e:
         output = re.sub( r"Warning: Permanently added '.+' \([RD]SA\) to "
                          "the list of known hosts.",
                          "",
                          e.output ).strip()
         if re.match( r"Permission denied \(.+\)\.", output ):
            # Probably incorrect username or password.
            output = "Permission denied"
         elif output.startswith( 'bash: ' ):
            # An error from bash
            output = output[ len( 'bash: ' ): ]
         elif output.startswith( 'ssh: ' ):
            # An error such as 'Name or service not known'.
            output = output[ len( 'ssh: ' ): ]
         elif output.startswith( self.fs.scheme + ' ' ):
            # An error such as 'No such file or directory'.
            output = output[ len( self.fs.scheme ) + 1 : ]
         else:
            r = re.match( "([a-zA-Z]+: )(write error: )(.*)", output )
            if r:
               # An error from the filter
               output = r.groups()[2]
         raise OSError( 0, output or "Command error" )

   def get( self, local ):
      self._scpGetOrPut( local, 'get' )

   def getWithFilter( self, local, filterCmd ):
      self._scpGetOrPut( local, 'get', filterCmd=filterCmd )

   def put( self, local, append=False ):
      if append:
         self._notSupported()
      self._scpGetOrPut( local, 'put' )

class SftpUrl( SshBaseUrl ):
   # See draft-ietf-secsh-scp-sftp-ssh-uri for specification of sftp uri.
   netAccount = 'sftp'

   def __init__( self, fs, url, rest, context ):
      SshBaseUrl.__init__(self, fs, url, rest, context)

      # sftp doesn't allow a password to be specified on the command-line, so if a
      # password is specified in the URL we ignore it.
      self.password = None
      # draft-ietf-secsh-scp-sftp-ssh-uri : URI Semantics
      #   ...
      #   It is RECOMMENDED that path be interpreted as an absolute path from
      #   the root of the file system. An implementation MAY use the tilde
      #   ("~") character as the first path element in the path to denote a
      #   path relative to the user's home directory.
      #   ...
      if not self.pathname.startswith('/'):
         if self.pathname == '~':
            self.pathname = ''
         elif self.pathname.startswith('~/'):
            self.pathname = self.pathname[ len('~/'): ]
         else:
            self.pathname = '/' + self.pathname

   def _sftpGetOrPut( self, local, op, filterCmd=None ):
      if not self.username:
         raise OSError( 0, "A username is required to use sftp" )
      if not self.urlhostname:
         raise OSError( 0, "A hostname is required to use sftp" )

      argv = [ 'sftp' ] + self._setupSshArgs()

      # Escaping file names is necessary since sftp command prompt has special
      # interpretation for some characters. For eg '#' is for comment
      scmd = None
      if op == 'put':
         # upload has to go through interactive commands
         argv.append( '%s@%s' % ( self.username, self.urlhostname ) )
         scmd = '%s \'%s\' \'%s\'\n' % ( op, local, self.pathname )
      else:
         # download can be done directly from command line
         assert op == 'get'
         remoteFile = "%s@%s:%s" % ( self.username, self.urlhostname, self.pathname )

         # just run from command line directly
         if filterCmd:
            # sftp does lseek for parallel requests (default 64), so we limit it to
            # just one request at a time.
            argv += [ '-R', '1', '-B', '65536', remoteFile ]
            argv = self._getFilterCmd( argv, filterCmd, local )
         else:
            argv += [ remoteFile, local ]

      cliCmdExt = CmdExtension.getCmdExtender()
      rc = 0
      try:
         output = cliCmdExt.runCmd( argv, self.cliSession, stderr=Tac.CAPTURE,
                                    vrfName=self.vrfName, input=scmd )
      except Tac.SystemCommandError as e:
         rc = e.error
         output = e.output

      # sftp writes error messages to stderr (along with some info/warn messages)
      # Strip out the info/warn messages and then look for error messages
      output = re.sub( r"Connect(ing|ed).+\n", "", output ).strip()
      output = re.sub( r"Warning: Permanently added '.+' \(.+?\) to "
                       r"the list of known hosts.",
                       "",
                       output ).strip()
      output = re.sub( r"FIPS mode initialized", "", output ).strip()
      output = re.sub( r"rekeying in progress for .*", "", output ).strip()
      output = re.sub( r"set_newkeys: rekeying for .*", "", output ).strip()
      if output != '':
         if output.startswith( 'bash: ' ):
            # An error from bash
            output = output[ len( 'bash: ' ): ]
         elif output.startswith( 'ssh: ' ):
            # An error such as 'Name or service not known'.
            output = output[ len( 'ssh: ' ): ]
            raise OSError( 0, output or "Command error" )
         elif output.startswith( 'sftp: ' ):
            output = output[ len( 'sftp: ' ): ]
            raise OSError( 0, output or "Command error" )
         elif re.match( 'subsystem request failed', output ):
            output = 'ssh server is not suporting sftp\n' + output
            raise OSError( 0, output or "Command error" )
         else:
            sftpErrStrList = [
               "sftp process:",
               "(K|k)ill.+signal",
               "Connection reset by peer",
               "Permission denied",
               "Couldn't (open|stat) (local|remote) file",
               "No such file or directory",
               "File .+ not found",
               "skipping non-regular file",
               "Cannot download non-regular file:",
               "Invalid path",
               "stat .+:",
               "Couldn't get handle:",
               "Couldn't (read|send) packet:",
               "Couldn't.+:.+",
               ".+:.+",
            ]
            for s in sftpErrStrList:
               if re.match( s, output.splitlines()[ -1 ] ):
                  raise OSError( 0, output or "Command error" )
            # This usually happens if we fail to connect in the first place
            # (ex not having correct privileges).
            # Unfortunately, sftp loses the return code in interactive mode
            # when we start putting/getting things after a successful connection,
            # which is why we still need the above error message checks.
            if rc:
               raise OSError( 0, output or "Command error" )

            # Hopefully all error messages have been detected.
            # This must be some banner/greeting message from server
            print( output )

   def get( self, local ):
      self._sftpGetOrPut( local, 'get' )

   def getWithFilter( self, local, filterCmd ):
      self._sftpGetOrPut( local, 'get', filterCmd=filterCmd )

   def put( self, local, append=False ):
      if append:
         self._notSupported()
      self._sftpGetOrPut( local, 'put' )

class NetworkFilesystem( Url.Filesystem ):
   def __init__( self, scheme, urlClass ):
      Url.Filesystem.__init__( self, scheme, 'network', 'rw' )
      self.urlClass_ = urlClass

   def localFileSystem( self ):
      return False

   def parseUrl( self, url, rest, context ):
      return self.urlClass_( self, url, rest, context )

def Plugin( context=None ):
   Url.registerFilesystem( NetworkFilesystem( 'http:', HttpUrl ) )
   Url.registerFilesystem( NetworkFilesystem( 'https:', HttpUrl ) )
   Url.registerFilesystem( NetworkFilesystem( 'ftp:', FtpUrl ) )
   Url.registerFilesystem( NetworkFilesystem( 'tftp:', TftpUrl ) )
   Url.registerFilesystem( NetworkFilesystem( 'scp:', ScpUrl ) )
   Url.registerFilesystem( NetworkFilesystem( 'sftp:', SftpUrl ) )
