#!/usr/bin/env python3
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""This module implements the "session" URL scheme.
Valid pathnames are all of the form 
'/<session-name>-session-config[-<options>]'
"""

import errno
import io
import os
import re
import hashlib

import CliCommon
import CliSave
import CliSession as CS
import ConfigMount
import FileCliUtil
import SessionUrlUtil as SUU
from UrlPlugin import FlashUrl
from UrlPlugin import SystemUrl
import Url

class SessionUrl( SystemUrl.SystemUrl ):

   def __init__( self, fs, url, rawpathname, pathname, context, clean=False ):
      SystemUrl.SystemUrl.__init__( self, fs, 
                                    url, rawpathname, pathname, context )
      self.sessionName_ = ""
      self.cleanConfig_ = clean
      self.valid_ = self.validPathname()
      self.abortOnError = False
      self.skipConfigCheck = False
      # The expected md5sum of the file content, if verification required, else None.
      self.md5 = None
      if self.cleanConfig_:
         self.sessionName_ = "clean"
      else:
         m = re.match( r"/(\S+)-session-config\S*", self.pathname )
         if m:
            self.sessionName_ = m.group( 1 )

   def ignoreTrailingWhitespaceInDiff( self ):
      return True

   def validPathname( self ):
      if not self.pathname[0] == "/":
         return False
      if not '-session-config' in self.pathname:
         return False
      return True

   def listdir( self ):
      if not self.isdir():
         raise OSError( errno.ENOTDIR, os.strerror( errno.ENOTDIR ) )
      names = CS.sessionNames( self.context.entityManager )
      return ( [ "clean-session-config" ] + 
               # pylint: disable-next=consider-using-f-string
               [ '%s-session-config' % name for name in names ] )

   def _options( self ):
      '''Return all options if the syntax is valid, else None.'''
      if not self.valid_:
         return None
      basePathname = f'/{self.sessionName_}-session-config'
      if self.pathname == basePathname:
         return ( CliSave.ShowRunningConfigOptions(),
                  CliSave.ShowRunningConfigRenderOptions() )

      paths = set( self.pathname[ ( len( basePathname ) + 1 ): ].split( ':' ) )

      showAll = 'all' in paths
      showAllDetail = showAll and 'detail' in paths
      showNoSeqNum = 'noseqnum' in paths
      showSanitized = 'sanitized' in paths
      showJson = 'json' in paths

      return ( CliSave.ShowRunningConfigOptions( showAll,
                                                 showAllDetail,
                                                 showNoSeqNum ),
               CliSave.ShowRunningConfigRenderOptions( showSanitized,
                                                       showJson ) )

   def exists( self ):
      return self.isdir() or self.valid_

   def historyEntryName( self ):
      if self.valid_ and not self.isdir():
         return "running" # not sure if this is accurate
      return None

   def get( self, dstFn ):
      import Tac # pylint: disable=import-outside-toplevel

      optionsPair = self._options()
      if optionsPair is None:
         raise OSError( errno.ENOENT, os.strerror( errno.ENOENT ) )
      if self.isdir():
         raise OSError( errno.EISDIR, os.strerror( errno.EISDIR ) )

      options, renderOptions = optionsPair

      entityManager = self.context.entityManager

      # Store the config in a temporary bytes seq, in case we get an exception
      # or the session gets disconnected while rendering the config. Both binary
      # and text layer are required for python3 since some method will write direct
      # to binary buffer
      dstData = io.BytesIO()
      f = io.TextIOWrapper( dstData, write_through=True )

      if not entityManager.isLocalEm():
         sStatus = SUU.getSessionStatus( entityManager )
         SUU.doMountsForSaveSession( entityManager, sStatus, None )
         sleep = not Tac.activityManager.inExecTime.isCurrent
         Tac.waitFor( SUU.saveSessionMountsDone, sleep=sleep,
                      description="preparation for saveSessionConfig" )

      CliSave.saveSessionConfig(
            entityManager, f, self.sessionName_, cleanConfig=self.cleanConfig_,
            saveAll=options.saveAll,
            saveAllDetail=options.saveAllDetail,
            showNoSeqNum=options.showNoSeqNum,
            showSanitized=renderOptions.showSanitized,
            showJson=renderOptions.showJson )

      config = dstData.getvalue()
      dstData.close()
      with open( dstFn, 'wb' ) as dstFile:
         dstFile.write( config )

   def put( self, srcFn, append=False ):
      # Checks file content matches md5 digest, if one provided
      if self.md5:
         # pylint: disable-next=consider-using-f-string
         url = Url.parseUrl( 'file:%s' % srcFn, Url.Context( None, True ) )
         actualMd5 = FileCliUtil.chunkedHashCompute( url, hashlib.md5 )
         if actualMd5 != self.md5:
            raise CliCommon.MD5Error(
                    f'md5 mismatch; {actualMd5!r} != {self.md5!r}' )
      srcFile = open( srcFn ) # pylint: disable=consider-using-with
      try:
         if not self.exists():
            raise OSError( errno.EACCES, os.strerror( errno.EACCES ) )
         assert not self.isdir()
         assert self.validPathname()

         # avoid import loop
         import MainCli # pylint: disable=import-outside-toplevel
         import CliPlugin.SessionCli # pylint: disable=import-outside-toplevel

         disableGuards = not self.context.cliSession.guardsEnabledForLoadConfig()
         with ConfigMount.ConfigMountDisabler( disable=False ):
            MainCli.loadConfig( srcFile, self.context.cliSession,
                           initialModeClass=CliPlugin.SessionCli.ConfigSessionMode,
                           disableAaa=self.context.disableAaa,
                           disableGuards=disableGuards,
                           autoComplete=True,
                           abortOnError=self.abortOnError,
                           skipConfigCheck=self.skipConfigCheck )
      finally:
         srcFile.close()

class SessionFilesystem( FlashUrl.LocalFilesystem ):
   urlClass_ = SessionUrl

   def __init__( self, scheme ):
      FlashUrl.LocalFilesystem.__init__( self, scheme, 'session', 'rw', hidden=True )

   def supportsListing( self ):
      return True  

def Plugin( context=None ):
   Url.registerFilesystem( SessionFilesystem( 'session:' ) )
