# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import Url
from UrlPlugin.SslUrl import CliFileSystem

import LazyMount
import Tac
import SshCertLib

sshConstants_ = Tac.Type( "Mgmt::Ssh::Constants" )

class SshFileSystem( CliFileSystem ):
   def __init__( self, scheme, rawlocation, permission, mask ):
      CliFileSystem.__init__( self, scheme, rawlocation, fsType='ssh', 
                              permission=permission, mask=mask )

   def handleFileChange( self, context ):
      # configReq is mounted in CliPlugin/Ssh.py
      if context and context.entityManager:
         entityManager = context.entityManager
         root = entityManager.root()
         configReq = root.entity.get( "mgmt/ssh/configReq", None )
         if configReq is None:
            # Ideally configReq would be mounted in Plugin() however no context or
            # entity manager is provided there. Therefore, this workaround is
            # required until BUG689392 is addressed.
            configReq = LazyMount.mount( entityManager, "mgmt/ssh/configReq",
                                         "Mgmt::Ssh::ConfigReq", "w" )
         configReq.sshCertFileUpdateRequest = Tac.now()

class SshAuthPrincipalsCmdFilesystem( SshFileSystem ):
   def __init__( self ):
      SshFileSystem.__init__( self,
                              scheme='ssh-auth-principals-cmd:',
                              rawlocation=sshConstants_.authPrincipalsCmdDirPath(),
                              permission='rw', mask=0o002 )

   def validateFile( self, filename, durl=None, context=None ):
      # Can't check if the file is a valid authorized principal command so
      # leave it to ssh to determine.
      pass

class SshHostCertFilesystem( SshFileSystem ):
   def __init__( self ):
      SshFileSystem.__init__( self,
                              scheme='ssh-host-cert:', 
                              rawlocation=sshConstants_.hostCertsDirPath(),
                              permission='rw', mask=0o002 )

   def validateFile( self, filename, durl=None, context=None ):
      try:
         SshCertLib.validateHostCert( filename )
      except SshCertLib.SshHostCertError as e:
         raise OSError( 0, e ) # pylint: disable=raise-missing-from

class SshCaKeyFilesystem( SshFileSystem ):
   def __init__( self ):
      SshFileSystem.__init__( self, 
                              scheme='ssh-ca-key:', 
                              rawlocation=sshConstants_.caKeysDirPath(),
                              permission='rw', mask=0o002 )

   def validateFile( self, filename, durl=None, context=None ):
      try:
         SshCertLib.validateMultipleKeysFile( filename )
      except SshCertLib.SshKeyError as e:
         raise OSError( 0, e ) # pylint: disable=raise-missing-from

class SshHostKeyFilesystem( SshFileSystem ):
   def __init__( self ):
      SshFileSystem.__init__( self, 
                              scheme='ssh-key:', 
                              rawlocation=sshConstants_.hostKeysDirPath(),
                              permission='rw', mask=0o066 )

   def validateFile( self, filename, durl=None, context=None ):
      try:
         assert durl, "Destination URL must be available"

         # Verify the private key is valid
         SshCertLib.validateHostKey( filename )
      except SshCertLib.SshKeyError as e:
         raise OSError( 0, e ) # pylint: disable=raise-missing-from

class SshRevokeListFilesystem( SshFileSystem ):
   def __init__( self ):
      SshFileSystem.__init__( self, 
                              scheme='ssh-revoke-list:', 
                              rawlocation=sshConstants_.revokeListsDirPath(),
                              permission='rw', mask=0o002 )

   def validateFile( self, filename, durl=None, context=None ):
      try:
         SshCertLib.validateMultipleKeysFile( filename )
      except SshCertLib.SshKeyError as e:
         raise OSError( 0, e ) # pylint: disable=raise-missing-from

def Plugin( context=None ):
   Url.registerFilesystem( SshAuthPrincipalsCmdFilesystem() )
   Url.registerFilesystem( SshCaKeyFilesystem() )
   Url.registerFilesystem( SshHostCertFilesystem() )
   Url.registerFilesystem( SshRevokeListFilesystem() )
   Url.registerFilesystem( SshHostKeyFilesystem() )
