#!/usr/bin/env python3
# Copyright (c) 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=unnecessary-pass
# pylint: disable=consider-using-f-string
# pylint: disable=use-dict-literal

""" This file contains a class that can be used to do various VmTracer event 
handling on Vm add, delete, change and move. This will be used in breadth tests
for EventMgr for VmTracer feature and also can be used as action script 
for EVentMgr for VmTracer. 

This class provides a base class to handle VmTracer events in Event Handler. 
It provides some abstract functions that must be defined in derived classes 
to complete the functionality of various event handlers represented by these
functions.

"""

import os, subprocess, time
import abc
from datetime import datetime

#t0 = Tracing.trace0
validAttrs = [ 'Physical switchport', 'vNIC', 'vNIC MAC addr', 'vNIC Portgroup',
               'vNIC VLAN', 'vSwitch / vDS', 'Status', 'Host', 'Data Center' ]

class VmEventHandler:
   """
   Calculate the difference between two vm dictionaries as:
   (1) vms added
   (2) vms removed
   (3) vms same in both but changed attributes
   (4) vms same in both and unchanged attributes
   (5) vms that are powered on
   (6) vms that are powered off
   """
   def __init__( self, oldVmFile=None, newVmFile=None, useCli=True ):
      if not oldVmFile:
         self.oldVmFile = "/tmp/VmTracer_Snapshot.txt"
      else:
         self.oldVmFile = oldVmFile
      if not newVmFile:
         self.newVmFile = "/tmp/VmTracer_Vms.txt"
      else:
         self.newVmFile = newVmFile
      self.oldVms = dict()
      self.newVms = dict()
      self.setNew = self.setOld = self.intersect = None
      self.newVmsRaw = self.oldVmsRaw = None
      self.useCli = useCli

   def vmsAdded( self ):
      if not self.setNew:
         return None
      return self.setNew - self.intersect 
   
   def vmsDeleted( self ):
      if not self.setOld:
         return None
      return self.setOld - self.intersect 
   
   def vmsChanged( self ):
      if not self.intersect:
         return None
      return { o for o in self.intersect if self.oldVms[ o ] != self.newVms[ o ] }
   
   def vmsUnchanged( self ):
      if not self.intersect:
         return None
      return { o for o in self.intersect if self.oldVms[ o ] == self.newVms[ o ] }

   # Builds a dictionary for Vms from the raw dump in vmDump (output of show CLI)
   # and puts it in vmDict
   def buildVmDict( self, vmDict, vmDump ):
      if not vmDump: 
         return
      vm = None
      for line in vmDump.split( '\n' ):
         if "VM Name" in line:
            name = ' '.join( line.split()[ 2: ] )
            vmDict[ name ] = dict()
            vm = vmDict[ name ] 
            continue
         if not line.strip():
            vm = None
            continue
         attr = line.split( ": " )
         attrName = attr[ 0 ].strip()
         # Skip some non-relevant lines. 
         if attrName not in validAttrs:
            continue
         value = ' '.join( attr[ 1: ] ).strip()
         if attrName: 
            vm[ attrName ] = value
         
   # Builds two dictionaries - one for existing VMs (gotten via CLI or passed in 
   # file) and from the snapshot file that was passed to the constructor. These 
   # two dictionaries are then used for rest of the operations
   def getVmsFromVmTracer( self ):
      cliCmd = [ '/usr/bin/Cli', '-c', 'show vmtracer vm detail' ]
      # if old snapshot file is there get Vms from there. This file is nothing
      # but a previous dump from CLI 'sh vmtracer vm detail' 
      if not os.path.exists( self.oldVmFile ):
         f = open( self.oldVmFile, 'w' ) # pylint: disable=consider-using-with
         f.write('')
         f.close()
      f = open( self.oldVmFile ) # pylint: disable=consider-using-with
      self.oldVmsRaw = f.read()
      f.close()

       # if new Vms are in a file get from there else run the CLi to get the Vms
      if self.useCli:
         try:
            self.newVmsRaw = subprocess.check_output( cliCmd )
         except subprocess.CalledProcessError as e:
            print( "Error: Couldn't get new Vms from Cli. Error: %d" % e.returncode )
            print( "Error: %s" % e.output )
            return
      elif self.newVmFile and os.path.exists( self.newVmFile ):
         f = open( self.newVmFile ) # pylint: disable=consider-using-with
         self.newVmsRaw = f.read()
         f.close()
      self.buildVmDict( self.oldVms, self.oldVmsRaw )
      self.buildVmDict( self.newVms, self.newVmsRaw )
            
      self.setNew, self.setOld = set( self.newVms.keys() ), \
          set( self.oldVms.keys() )
      self.intersect = self.setNew.intersection( self.setOld )

   # Print Vm details (all its attributes) from the given dictionary for vms in 
   # vmlist
   def printVms( self, vmlist, vmDict, prefix='' ):
      for vm in vmlist:
         data =  prefix + " : VM Name " + "%s : %f : %s" \
             % ( vm, time.time(), datetime.now() )
         print( data )
         attrs = vmDict[ vm ]
         for attr in attrs:
            print( attr + " : " + attrs[ attr ], end=' ' )
            print()
         print()

   # returns a dicitonary of all the Vms that are newly added - ones that are in 
   # current VMs but not in the old snapshot
   def getAddedVms( self ):
      addedVms = self.vmsAdded()
      if not addedVms:
         return None
      vms = dict()
      for vmName in addedVms:
         vms[ vmName ] = self.newVms[ vmName ]
      return vms if len(vms) else None

   # returns a dictionary of VMs that were deleted
   def getDeletedVms( self ):
      deletedVms = self.vmsDeleted()
      if not deletedVms:
         return None
      vms = dict()
      for vmName in deletedVms:
         vms[ vmName ] = self.oldVms[ vmName ]
      return vms if len(vms) else None

   # returns a dictionary of Vms whose attributes were modified from the previous
   # snapshot along with the old and new values for the attributes that were 
   # changed
   def getChangedVms( self ):
      changedVms = self.vmsChanged()
      if not changedVms:
         return None
      vms = dict()
      for vmName in changedVms:
         vms[ vmName ] = dict()
         changedAttrs = vms[ vmName ]
         oldVmAttrs = self.oldVms[ vmName ]
         newVmAttrs = self.newVms[ vmName ]
         for attr in oldVmAttrs:
            if oldVmAttrs[ attr ] != newVmAttrs[ attr ]:
               changedAttrs[ attr ] = oldVmAttrs[ attr ] + "=>" + newVmAttrs[ attr ]
      return vms if len(vms) else None

   # returns a dictionary of Vms that are moved with Vm names and from and to 
   # hosts. None if no Vms moved
   def getMovedVms( self ):
      changedVms = self.vmsChanged()
      if not changedVms:
         return None
      vms = dict()
      hostAttr = 'Host'
      for vmName in changedVms:
         oldVmAttrs = self.oldVms[ vmName ]
         newVmAttrs = self.newVms[ vmName ]
         if oldVmAttrs[ hostAttr ] != newVmAttrs[ hostAttr ]:
            vms[ vmName ] = dict()
            changedAttrs = vms[ vmName ]
            changedAttrs[ hostAttr ] = oldVmAttrs[ hostAttr ] + "=>" \
                + newVmAttrs[ hostAttr ]
      return vms if len(vms) else None

   # returns a dictionary of Vms that are powered on from last time
   # None if no Vms were powered on
   def getPowerChangedVms( self, newStatus ):
      newStatus = newStatus.capitalize()
      status = { 'Up':'Down', 'Down':'Up' }
      oldStatus = status[ newStatus ]

      changedVms = self.vmsChanged()
      if not changedVms:
         return None
      vms = dict()
      attr = 'Status'
      for vmName in changedVms:
         oldVmAttrs = self.oldVms[ vmName ]
         newVmAttrs = self.newVms[ vmName ]
         if oldVmAttrs[ attr ] != newVmAttrs[ attr ]:
            oldPower = oldVmAttrs[ attr ].split('/')[ 0 ].capitalize()
            newPower = newVmAttrs[ attr ].split('/')[ 0 ].capitalize()
            
            # If power status changed as expected
            if oldPower  == oldStatus and newPower == newStatus:
               vms[ vmName ] = dict()
               changedAttrs = vms[ vmName ]
               changedAttrs[ attr ] = oldVmAttrs[ attr ] + "=>" \
                                        + newVmAttrs[ attr ]
      return vms if len(vms) else None

   # returns a dictionary of Vms that are not changed from previous snapshot
   def getUnchangedVms( self ):
      unchangedVms = self.vmsUnchanged()
      if not unchangedVms:
         return None
      vms = dict()
      for vmName in unchangedVms:
         vms[ vmName ] = self.oldVms[ vmName ]
      return vms if len(vms) else None
      
   def printAddedVms( self, addedVms ):
      if addedVms:
         print( "-------ADD-----" )
         self.printVms( list( addedVms ), addedVms, "VM Added" )
             
   def printDeletedVms( self, deletedVms ):
      if deletedVms:
         print( "-------DELETE----" )
         self.printVms( list( deletedVms ), deletedVms, "VM Deleted" )
      
   def printChangedVms( self, changedVms ):
      if changedVms:
         print( "-------CHANGED----" )
         self.printVms( list( changedVms ), changedVms, "VM Changed" )
      
   def printMovedVms( self, movedVms ):
      if movedVms:
         print( "-------MOVED----" )
         self.printVms( list( movedVms ), movedVms, "VM Moved" )
      
   def printPoweredOnVms( self, poweredVms ):
      if poweredVms:
         print( "-------POWEREDON----" )
         self.printVms( list( poweredVms ), poweredVms, "VM Powered On" )
      
   def printPoweredOffVms( self, poweredVms ):
      if poweredVms:
         print( "-------POWEREDOFF----" )
         self.printVms( list( poweredVms ), poweredVms, "VM Powered Off" )
      
   def printUnchangedVms( self ):
      unchangedVms = self.vmsUnchanged()
      if unchangedVms:
         print( "-------UNCHANGED----" )
         self.printVms( unchangedVms, self.oldVms )

   # Save snapshot of currently existing Vms. This will be used later to diff with 
   # new collection of Vms when action script is run
   def saveSnapshot( self ):
      os.remove( self.oldVmFile )
      f = open( self.oldVmFile, 'w' ) # pylint: disable=consider-using-with
      for vm in self.newVms: # pylint: disable=consider-using-dict-items
         vmName = "VM Name " + vm + "\n"
         f.write( vmName )
         attrs = self.newVms[ vm ]
         for attr in attrs:
            attribute = attr + " : " + attrs[attr] + "\n"
            f.write( attribute )
      f.close()

   # Handle the Vms that got added. For now just print them but this can be changed 
   # to do other things like vlan modification, ACL, etc. 
   @abc.abstractmethod
   def handleAddedVms( self ):
      """ This is an abstract method and must be implemented in the derived class. 
      The derived functions can do just what is below if they want to only print
      addedVms = self.getAddedVms()
      self.printAddedVms( addedVms )
      """
      pass

   @abc.abstractmethod
   def handleDeletedVms( self ):
      """ This is an abstract method and must be implemented in the derived class. 
      The derived functions can do just what is below if they want to only print
      deletedVms = self.getAddedVms()
      self.printDeletedVms( deletedVms )
      """
      pass


   @abc.abstractmethod
   def handleChangedVms( self ):
      """ This is an abstract method and must be implemented in the derived class. 
      The derived functions can do just what is below if they want to only print
      changedVms = self.getChangedVms()
      self.printChangedVms( changedVms )
      """
      pass

   @abc.abstractmethod
   def handleMovedVms( self ):
      """ This is an abstract method and must be implemented in the derived class. 
      The derived functions can do just what is below if they want to only print
      movedVms = self.getMovedVms()
      self.printMovedVms( movedVms )
      """
      pass

   @abc.abstractmethod
   def handlePoweredOnVms( self ):
      """ This is an abstract method and must be implemented in the derived class. 
      The derived functions can do just what is below if they want to only print
      poweredVms = self.getPowerChangedVms( 'Up' )
      self.printPoweredOnVms( poweredVms )
      """
      pass

   @abc.abstractmethod
   def handlePoweredOffVms( self ):
      """ This is an abstract method and must be implemented in the derived class. 
      The derived functions can do just what is below if they want to only print
      poweredVms = self.getPowerChangedVms( 'Down' )
      self.printPoweredOffVms( poweredVms )
      """
      pass

   # This routine does it all. It will get all the Vms that have changed
   # and take actions based on whether Vms are added, deleted or changed
   def handleVms( self ):
      # First current VMs
      self.getVmsFromVmTracer()
      self.handleAddedVms()
      self.handleDeletedVms()
      self.handlePoweredOnVms()
      self.handlePoweredOffVms()
      self.handleMovedVms()
      self.handleChangedVms()
      # Save Vms in a snapshot file
      self.saveSnapshot()
      
