#!/usr/bin/env python3
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import re
import Tac
import CliMatcher
import MultiRangeRule
from CliParser import Completion
from CliParserCommon import MatchResult, noMatch, ParseError, safeInt

invalidVni = Tac.Type( 'Vxlan::VniExtOrNone' ).invalidVni

class VniDottedDoParse( MultiRangeRule.DoParseNoMerge ):
   '''
   This do parse class is intended for a range of VNIs as opposed to
   ASNs like the original DoParse. To accomplish this, we implemented our
   own matchNumber and convertToNumber method with our own regex.

   Use this class as a parameter in MultiRangeMatcher to match on
   multiple VNIs. numberFormat should be set to NumberFormat.DOTTED if this class
   is being used.
   '''

   def __init__( self, mode, buff, noSingletons, explicitIntfTypes,
                 allowedSubIntfTypes=None,
                 genericRange=False,
                 numberFormat=MultiRangeRule.NumberFormat.DEFAULT,
                 maxRanges=0,
                 noHoleRange=False, dollarCompHelpText=None, matchIfWildcard=False ):
      super().__init__( mode, buff, noSingletons,
                        explicitIntfTypes, allowedSubIntfTypes,
                        genericRange, numberFormat, maxRanges,
                        noHoleRange, dollarCompHelpText, matchIfWildcard=False )


   numberDottedRe_ = re.compile( r"(\d+\.){0,3}\d*" )

   def matchNumber( self ):
      '''Specifies the criteria/regex to match for a number'''
      return self._need( self.numberDottedRe_ )

   def convertToNumber( self, soFar ):
      '''Specifies the criteria/regex to convert the matched string
         to an integer value'''
      if '.' not in soFar:
         soFar = safeInt( soFar )
      else:
         soFarSpl = soFar.split( '.' )
         numSplit = len( soFarSpl )
         if numSplit < 3: # pylint: disable=no-else-raise
            # pylint: disable-next=consider-using-f-string
            raise ParseError( "%r invalid format" % soFar )
         else:
            val = 0
            for s in soFarSpl:
               if s == '':
                  # pylint: disable-next=consider-using-f-string
                  raise ParseError( "%r invalid format" % soFar )
               tmp = safeInt( s )
               val = val << 8 | tmp
            soFar = val
      return soFar

# Better a free function than a @static method of VniFormat class
def vniToString( vniNum, dottedNotation ):
   # blind conversion from string or int
   vniNum = int( vniNum )
   if vniNum == invalidVni:
      return 'invalid'
   elif dottedNotation:
      if vniNum & 0xFF000000 == 0:
         # pylint: disable-next=consider-using-f-string
         return "{}.{}.{}".format( ( vniNum >> 16 & 0xFF ),
                                   ( vniNum >> 8 & 0xFF ),
                                   ( vniNum & 0xFF ) )
      else:
         # pylint: disable-next=consider-using-f-string
         return "{}.{}.{}.{}".format( ( vniNum >> 24 & 0xFF ),
                                      ( vniNum >> 16 & 0xFF ),
                                      ( vniNum >> 8 & 0xFF ),
                                      ( vniNum & 0xFF ) )
   else:
      return str( vniNum )
      
# Convert Vni ranges to a string representation. vniRanges is a list of
# VniRange() objects
def vniRangesToString( vniRanges, dottedNotation ):
   sList = []
   for vniRange in vniRanges:
      if vniRange.minVni != vniRange.maxVni:
         vniString = ( vniToString( vniRange.minVni, dottedNotation ) +
                       ( "--" if dottedNotation else "-" ) +
                       vniToString( vniRange.maxVni, dottedNotation ) )
      else:
         vniString = vniToString( vniRange.minVni, dottedNotation ) 
      sList.append( vniString )
   return ", ".join( sList )
   
# Convert Vni ranges to list of Vnis. vniRanges is a list of VniRange() objects
def vniRangesToVnis( vniRanges ):
   vnis = []
   for vniRange in vniRanges:
      vnis += list( range( vniRange.minVni, vniRange.maxVni + 1 ) )
   return vnis
   
class VniFormat:
   vni24DottedRe_ = re.compile( r'(\d+)\.(\d+)\.(\d+)$' )
   vni32DottedRe_ = re.compile( r'(\d+)\.(\d+)\.(\d+)\.(\d+)$' )

   def __init__( self, vniNum, dottedNotation=False ):
      self.dottedNotation = dottedNotation
      if isinstance( vniNum, int ):
         self.vniNum = vniNum
      elif isinstance( vniNum, str ):
         vniNum = vniNum.strip()
         if vniNum.isdigit():
            self.vniNum = int( vniNum )
         else:
            # Match 24-bit dotted notation
            m = VniFormat.vni24DottedRe_.match( vniNum )
            if m:
               self.vniNum = ( int( m.group( 1 ) ) << 16 ) | \
                  ( int( m.group( 2 ) ) << 8 ) | \
                  int( m.group( 3 ) )
            else:
               # Match 32-bit dotted notation
               m = VniFormat.vni32DottedRe_.match( vniNum )
               if m:
                  self.vniNum = ( int( m.group( 1 ) ) << 24 ) | \
                     ( int( m.group( 2 ) ) << 16 ) | \
                     ( int( m.group( 3 ) ) << 8 ) | \
                     int( m.group( 4 ) )
               elif vniNum == 'invalid':
                  self.vniNum = invalidVni
               else:
                  # pylint: disable-next=consider-using-f-string
                  raise ValueError( '%s is not a valid VNI' % vniNum )
      else:
         self.vniNum = invalidVni

   def __str__( self ):
      return vniToString( self.vniNum, self.dottedNotation )

   def __lt__( self, other ):
      return self.vniNum < other.vniNum

   def toNum( self ):
      return self.vniNum

   def isValid( self ):
      # invalidVni is outside the range from 0 - maxVni
      return 0 < self.vniNum < invalidVni

class VniMatcher( CliMatcher.Matcher ):
   '''Match VNI in dotted format or 24-bit/32-bit number'''

   # Regex for VNI completion
   vniCompletionRe_ = re.compile( r'(\d+)(\.(\d+)?){0,3}?$' )

   def __init__( self, helpdesc, vxlanCntrlCfgBox, **kwargs ):
      super().__init__( helpdesc=helpdesc, **kwargs )
      self.completionDotted_ = Completion( 'A.B.C.D', helpdesc, False )
      self.completion_ = Completion( '<1-4294967294>', helpdesc, False )
      self.vxlanCntrlCfgBox = vxlanCntrlCfgBox

   def match( self, mode, context, token ):
      try:
         vni = VniFormat( token )
      except ValueError:
         return noMatch
      # Validate VNI range
      if not vni.isValid():
         return noMatch
      result = str( vni ) 
      return MatchResult( result, result )

   def completions( self, mode, context, token ):
      # Ignore the completion if vxlanCntrlCfgBox is not populated yet
      if not self.vxlanCntrlCfgBox:
         return []

      # Display completion help for dotted notation when configured
      if self.vxlanCntrlCfgBox[ 0 ].vniInDottedNotation:
         completion = self.completionDotted_
      else:
         completion = self.completion_

      if not token:
         return [ completion ]

      m = VniMatcher.vniCompletionRe_.match( token )
      if m is None:
         return []
      # If matched string can construct VNI, validate that VNI is valid before 
      # returning completion
      try:
         vni = VniFormat( m.group( 0 ) )
         if not vni.isValid():
            return []
      except ValueError:
         pass
      return [ completion ]

   def __str__( self ):
      return '<Vni>'
