#!/usr/bin/env python3
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tracing
import Tac
import os
import time

wsDefaultTimeout = 600

traceHandle = Tracing.Handle( 'TelegrafMgrSuperServerPlugin' )
t0 = traceHandle.trace0

def pidCheck( daemonName, pidFile ):
   try:
      with open( pidFile ) as f:
         pid = f.read().strip()
         pgrepPid = Tac.run( [ "pgrep", daemonName ], stdout=Tac.CAPTURE,
                             ignoreReturnCode=True ).strip()
         if pid == pgrepPid:
            return True
   except OSError:
      pass
   return False

def start( serviceName, daemonName, startCmds, pidFile, config ):
   if pidCheck( daemonName, pidFile ):
      return [ "echo", "{} process is already running".format(
         serviceName ) ]

   t0( f"Starting {serviceName}..." )
   for cmd, stdout, stderr, timeout, asDaemon, pidWaitTimeout in startCmds:
      t0( cmd )
      out = Tac.run( cmd, stdout=stdout, stderr=stderr, timeout=timeout,
                     asDaemon=asDaemon )
      t0( out )

      for _ in range( pidWaitTimeout ):
         if os.path.exists( pidFile ):
            break
         time.sleep( 1 )

   if pidCheck( daemonName, pidFile ):
      return [ "echo", f"{serviceName} process was started" ]
   else:
      return [ "echo", "{} process was unable to start".format(
         serviceName ) ]

def stop( serviceName, daemonName, pidFile ):
   try:
      with open( pidFile ) as f:
         pid = f.read().strip()
         pgrepPid = Tac.run( [ "pgrep", daemonName ], stdout=Tac.CAPTURE,
                             ignoreReturnCode=True ).strip()
         if pid != pgrepPid:
            return [ "echo", "Faulty PID tracking {} "
                     "process already stopped".format( serviceName ) ]
   except OSError:
      return [ "echo", "{} process already stopped".format(
         serviceName ) ]

   t0( f"Stopping {serviceName}..." )

   with open( os.devnull, 'w' ) as devnull:
      Tac.run( [ "kill", "-9", f"{pid}" ],
               stdout=devnull, stderr=devnull, asRoot=True )
      Tac.run( [ "rm", "-f", pidFile ],
               stdout=devnull, stderr=devnull, asRoot=True )
   return [ "echo", f"{serviceName} process was stopped" ]

def restart( serviceName, daemonName, startCmds, pidFile, config ):
   msgCmd = stop( serviceName, daemonName, pidFile )
   Tac.run( msgCmd, stdout=Tac.CAPTURE, stderr=Tac.CAPTURE,
            timeout=wsDefaultTimeout )
   msgCmd = start( serviceName, daemonName, startCmds, pidFile, config )
   return msgCmd

def status( daemonName, pidFile ):
   if pidCheck( daemonName, pidFile ):
      return [ "echo", f"{daemonName} process is running" ]
   else:
      return [ "echo", f"{daemonName} process is not running" ]
