# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from TypeFuture import TacLazyType

# Column widths:
# 26 -> the length of the longest error key + 1,
# ex. len( "Module power down timeout" ) + 1
errorColumnWidth = 26
# 25 -> the length of the longest error state + 1,
# ex. len( "program memory corrupted" ) + 1
stateColumnWidth = 25
# 6 -> length of "Count" + 1
countColumnWidth = 6
# 13 -> length of longest timestamp + 1,
# ex. len( "10:10:07 ago" ) + 1
lastErrorColumnWidth = 13
columnWidths = [ errorColumnWidth, stateColumnWidth,
                countColumnWidth, lastErrorColumnWidth ]
columnFormat = " ".join( f'{{:{c}s}}' for c in columnWidths )

faultStateKey = "Controller fault state"
smbusKey = "Smbus failures"
cmisConfigKey = "Config state"
moduleFaultKey = "Module fault"
dpFirmwareKey = "Datapath firmware error"
moduleFirmwareKey = "Module firmware error"
vdmFreezeKey = "VDM freeze timeout"
vdmUnfreezeKey = "VDM unfreeze timeout"
modulePwrUpKey = "Module power up timeout"
modulePwrDwnKey = "Module power down timeout"
dpDeinitKey = "Datapath deinit timeout"
mixedApplicationsKey = "Mixed application state"
errorKeys = [ faultStateKey, smbusKey,
              cmisConfigKey, mixedApplicationsKey, moduleFaultKey,
              dpFirmwareKey, moduleFirmwareKey, vdmFreezeKey, vdmUnfreezeKey,
              modulePwrUpKey, modulePwrDwnKey, dpDeinitKey ]

CmisConfigurationStatus = TacLazyType( "Xcvr::CmisConfigurationStatus" )
MixedApplicationStatus = TacLazyType( "Xcvr::MixedApplicationStatus" )
SmbusErrorLoc = TacLazyType( "Xcvr::SmbusErrorLoc" )
CmisModuleFaultCause = TacLazyType( "Xcvr::CmisModuleFaultCause" )
FaultyStateReason = TacLazyType( "Xcvr::FaultyStateReason" )

def lowerCamelCase( errorKey ):
   # Convert an errorKey in errorKeys to lower camel case so the dict keys
   # in the cli model are json compliant.
   modelKey = errorKey.title().replace( " ", "" )
   modelKey = ''.join( [ modelKey[ 0 ].lower(), modelKey[ 1 : ] ] )
   return modelKey

def getCmisConfigStatus( state ):
   if state == CmisConfigurationStatus.configAccepted:
      return "accepted"
   elif state == CmisConfigurationStatus.configRejected:
      return "rejected"
   else:
      assert state == CmisConfigurationStatus.configNoStatus
      return "no status"

def getMixedApplicationStatus( state ):
   if state == MixedApplicationStatus.mixedAppsAccepted:
      return "accepted"
   elif state == MixedApplicationStatus.mixedAppsRejected:
      return "rejected"
   else:
      assert state == MixedApplicationStatus.mixedAppsNone
      return "none"

def getSmbusErrorReason( state ):
   if state == SmbusErrorLoc.smbusErrNone:
      return "ok"
   elif state == SmbusErrorLoc.smbusErrInitialization:
      return "initialization"
   elif state == SmbusErrorLoc.smbusErrEepromRead:
      return "eeprom read"
   elif state == SmbusErrorLoc.smbusErrFaulty:
      return "faulty"
   elif state == SmbusErrorLoc.smbusErrConfig:
      return "configuration"
   elif state == SmbusErrorLoc.smbusErrMonitor:
      return "monitor"
   elif state == SmbusErrorLoc.smbusErrTestPattern:
      return "test pattern"
   else:
      assert state == SmbusErrorLoc.smbusErrUnspecified
      return "unspecified"

def getModuleFaultCause( state ):
   if state == CmisModuleFaultCause.moduleNoFaultCause:
      return "ok"
   elif state == CmisModuleFaultCause.moduleFaultCauseTecRunaway:
      return "TEC runaway"
   elif state == CmisModuleFaultCause.moduleFaultCauseDataMemCorrupted:
      return "data memory corrupted"
   elif state == CmisModuleFaultCause.moduleFaultCauseProgramMemCorrupted:
      return "program memory corrupted"
   elif state == CmisModuleFaultCause.moduleFaultCauseReserved:
      return "reserved"
   elif state == CmisModuleFaultCause.moduleFaultCauseCustom:
      return "custom"
   else:
      assert state == CmisModuleFaultCause.moduleFaultCauseUnknown
      return "unknown"

def getFaultyStateReason( state ):
   if state == FaultyStateReason.faultCauseInvalidChecksum:
      return "invalid eeprom checksum"
   elif state == FaultyStateReason.faultCauseSmbusError:
      return "smbus error"
   elif state == FaultyStateReason.faultCauseUnsupportedPower:
      return "unsupported power"
   elif state == FaultyStateReason.faultCauseUnsupportedXcvr:
      return "unsupported or counterfeit transceiver"
   else:
      assert state == FaultyStateReason.faultCauseNone
      return "none"
