#!/usr/bin/env python3
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import collections

import Tac

Speeds = Tac.Type( "Inventory::XcvrLaneMgr::PortSpeed" )

class XcvrLaneMgrFdlHelper:
   """The port configuration descriptions can get complicated.  This
   class helps in both generating and reading back in a table of
   interface configurations. The 'genxxx' functions can generate the
   typical table entries for SFP, QSFP, and OSFP interfaces. The
   'configure' method then parses the table and fills in the inventory
   Tac model. We could have just directly populated the FDL but I
   wanted to keep the intermediate table format to allow unusual
   configurations to be described.

   This is the format expected by the laneMap parameter:
   [
   ( ethPortId,
   [ ( configId1,
   ( [speed1, speed2,...], [subordinateEthPortIds, ...],
   [ ( laneId, xcvrSlotId, xcvrLane, used ), ... ] ) )
   ( configId2,
   ( [speed1, speed2,...], [subordinateEthPortIds, ...],
   [ ( laneId, xcvrSlotId, xcvrLane, used ), ... ] ) ),
   ] ),
   ( next ethPort entry ), ...
   ]

   There is also a simpler table that basically encodes the parameters to the
   genxxx functions. See 'parseSimplePortTable' method.
   """

   def __init__( self, ethPortDir, xcvrSlotDir, portLaneConfigDir, laneMap=None ):
      self.ethPortDir = ethPortDir
      self.xcvrSlotDir = xcvrSlotDir
      self.portLaneConfigDir = portLaneConfigDir
      if laneMap:
         self.laneMap = laneMap
      else:
         self.laneMap = []

   @staticmethod
   def compressEthMapping( ethMapping ):
      """Takes an ethMapping of the form:

{
   relativeHostLane : [
      ( port speed0, [ xcvrLane0, ... ], [ subordinate xcvr host lanes, ... ] ),
      ( port speed1, ... ),
      ...
   ],
   relativeHostLane : [ ... ]
   ...
}

      and compresses it so that port speeds that have the same xcvrLane and
      subordinate ethPortId information are grouped.

      Return a format that is the same as ethMapping but with the port speed
      attribute as a list.
      """
      compressedEthMapping = {}
      for hostLane, speedInfoList in ethMapping.items():
         cfgs = []
         commonSpeedData = collections.defaultdict( list )
         for speedInfo in speedInfoList:
            portSpeed = speedInfo[ 0 ]
            portSpeedData = speedInfo[ 1 : ]
            commonSpeedData[ portSpeedData ].append( portSpeed )
         # sort this so that changes are easier to spot. If a change
         # is made the order will be consistent so it is easier
         # to find in a diff of the inventory models
         sortedCommonSpeedData = sorted( commonSpeedData.items(),
                                         key=lambda x: x[ 0 ] )
         for speedData, portSpeeds in sortedCommonSpeedData:
            cfgs.append( ( portSpeeds, ) + speedData )
         compressedEthMapping[ hostLane ] = cfgs
      return compressedEthMapping

   def populateBasicEthMapping( self, ethPortIdMap, xcvrSlotId, ethMapping ):
      """Populates self.laneMap given an ethMapping of the form

{
   relativeHostLane : [
      ( port speed0, [ xcvrLane0, ... ], [ subordinate xcvr host lanes, ... ] ),
      ( port speed1, ... ),
      ...
   ],
   relativeHostLane : [ ... ]
   ...
}

      It is assumed that only one xcvrSlotId is involved, and ethernet port IDs are
      specified in ethPortIdMap.
      """
      compessedEthMapping = XcvrLaneMgrFdlHelper.compressEthMapping( ethMapping )
      for relativeHostLane, speedInfoList in compessedEthMapping.items():

         # if there is no ethPortId for the xcvr host lane then do not populate any
         # capabilities for that lane
         if relativeHostLane not in ethPortIdMap:
            continue
         ethPortId = ethPortIdMap[ relativeHostLane ]

         cfgs = []
         cfgId = 0
         for portSpeeds, xcvrLanes, subHostLanes in speedInfoList:
            ethPortIdList = [ ethPortIdMap[ x ] for x in subHostLanes
                              if x in ethPortIdMap ]
            cfgs.append( ( cfgId, portSpeeds,
                           ethPortIdList,
                           ( ( intfLane, xcvrSlotId, xcvrLane, True )
                             for intfLane, xcvrLane in enumerate( xcvrLanes ) ) ) )
            cfgId += 1
         entry = ( ethPortId, cfgs )
         self.laneMap.append( entry )

   def genRj45SlotMap( self, ethPortIdMap, xcvrSlotId ):
      rj45EthMapping = {
         0: ( ( Speeds.portSpeed100Mbps, ( 0, ), () ),
               ( Speeds.portSpeed10Mbps, ( 0, ), () ),
               ( Speeds.portSpeed1Gbps, ( 0, ), () ),
               ( Speeds.portSpeed2p5Gbps, ( 0, ), () ),
               ( Speeds.portSpeed5Gbps, ( 0, ), () ),
               ( Speeds.portSpeed10Gbps, ( 0, ), () ), ),
      }
      self.populateBasicEthMapping( ethPortIdMap, xcvrSlotId, rj45EthMapping )

   # TODO: See BUG535096; This should be removed in favor of the sfp10 values
   def genSfpSlotMap( self, ethPortIdMap, xcvrSlotId ):
      """Generates a SFP one lane configuration"""
      sfpEthMapping = {
         0: ( ( Speeds.portSpeed10Gbps, ( 0, ), () ), ),
      }
      self.populateBasicEthMapping( ethPortIdMap, xcvrSlotId, sfpEthMapping )

   def genSfp1SlotMap( self, ethPortIdMap, xcvrSlotId ):
      """Generates a SFP one lane configuration"""
      sfpEthMapping = {
         0: ( ( Speeds.portSpeed100Mbps, ( 0, ), () ),
               ( Speeds.portSpeed1Gbps, ( 0, ), () ), ),
      }
      self.populateBasicEthMapping( ethPortIdMap, xcvrSlotId, sfpEthMapping )

   def genSfp10SlotMap( self, ethPortIdMap, xcvrSlotId ):
      """Generates a SFP one lane configuration"""
      sfpEthMapping = {
         0: ( ( Speeds.portSpeed100Mbps, ( 0, ), () ),
               ( Speeds.portSpeed1Gbps, ( 0, ), () ),
               ( Speeds.portSpeed10Gbps, ( 0, ), () ), ),
      }
      self.populateBasicEthMapping( ethPortIdMap, xcvrSlotId, sfpEthMapping )

   def genSfp25SlotMap( self, ethPortIdMap, xcvrSlotId ):
      """Generates a SFP one lane configuration"""
      sfpEthMapping = {
         0: ( ( Speeds.portSpeed1Gbps, ( 0, ), () ),
               ( Speeds.portSpeed10Gbps, ( 0, ), () ),
               ( Speeds.portSpeed25Gbps, ( 0, ), () ), ),
      }
      self.populateBasicEthMapping( ethPortIdMap, xcvrSlotId, sfpEthMapping )

   def genSfp50SlotMap( self, ethPortIdMap, xcvrSlotId ):
      """Generates a SFP one lane configuration"""
      sfpEthMapping = {
         0: ( ( Speeds.portSpeed10Gbps, ( 0, ), () ),
               ( Speeds.portSpeed25Gbps, ( 0, ), () ),
               ( Speeds.portSpeed50Gbps1Lane, ( 0, ), () ), ),
      }
      self.populateBasicEthMapping( ethPortIdMap, xcvrSlotId, sfpEthMapping )

   def genDsfpSlotMap( self, ethPortIdMap, xcvrSlotId ):
      """Generates a DSFP two lane configuration"""
      dsfpEthMapping = {
         0: ( ( Speeds.portSpeed10Gbps, ( 0, ), () ),
               ( Speeds.portSpeed25Gbps, ( 0, ), () ),
               ( Speeds.portSpeed50Gbps1Lane, ( 0, ), () ),
               ( Speeds.portSpeed50Gbps, ( 0, 1 ), ( 1, ), ),
               ( Speeds.portSpeed100Gbps2Lane, ( 0, 1 ), ( 1, ) ), ),
         1: ( ( Speeds.portSpeed10Gbps, ( 1, ), () ),
               ( Speeds.portSpeed25Gbps, ( 1, ), () ),
               ( Speeds.portSpeed50Gbps1Lane, ( 1, ), () ), ),
      }
      self.populateBasicEthMapping( ethPortIdMap, xcvrSlotId, dsfpEthMapping )

   def genSfpDdSlotMap( self, ethPortIdMap, xcvrSlotId ):
      """Generates a SFP-DD two lane configuration"""
      sfpDdEthMapping = {
         0: ( ( Speeds.portSpeed10Gbps, ( 0, ), () ),
               ( Speeds.portSpeed25Gbps, ( 0, ), () ),
               ( Speeds.portSpeed50Gbps1Lane, ( 0, ), () ),
               ( Speeds.portSpeed50Gbps, ( 0, 1 ), ( 1, ), ),
               ( Speeds.portSpeed100Gbps2Lane, ( 0, 1 ), ( 1, ) ), ),
         1: ( ( Speeds.portSpeed10Gbps, ( 1, ), () ),
               ( Speeds.portSpeed25Gbps, ( 1, ), () ),
               ( Speeds.portSpeed50Gbps1Lane, ( 1, ), () ), ),
      }
      self.populateBasicEthMapping( ethPortIdMap, xcvrSlotId, sfpDdEthMapping )

   def genQsfp40SlotMap( self, ethPortIdMap, xcvrSlotId, startLane=0 ):
      """Generate a 4 lane configuration like we would want for a QSFP slot. QSFP28
      also support 100G and potentially 4x25G and 2x50G as well as 40G and 4x10G, but
      may not be always true, for example optic 100G xcvrs.
      """
      qsfpEthMapping = {
         0: ( ( Speeds.portSpeed100Mbps, ( 0, ), () ),
               ( Speeds.portSpeed1Gbps, ( 0, ), () ),
               ( Speeds.portSpeed10Gbps, ( 0, ), () ),
               ( Speeds.portSpeed25Gbps, ( 0, ), () ),
               ( Speeds.portSpeed50Gbps, ( 0, 1 ), ( 1, ) ),
               ( Speeds.portSpeed40Gbps, ( 0, 1, 2, 3 ), ( 1, 2, 3 ) ), ),
         1: ( ( Speeds.portSpeed100Mbps, ( 1, ), () ),
               ( Speeds.portSpeed1Gbps, ( 1, ), () ),
               ( Speeds.portSpeed10Gbps, ( 1, ), () ),
               ( Speeds.portSpeed25Gbps, ( 1, ), () ), ),
         2: ( ( Speeds.portSpeed100Mbps, ( 2, ), () ),
               ( Speeds.portSpeed1Gbps, ( 2, ), () ),
               ( Speeds.portSpeed10Gbps, ( 2, ), () ),
               ( Speeds.portSpeed25Gbps, ( 2, ), () ),
               ( Speeds.portSpeed50Gbps, ( 2, 3 ), ( 3, ) ), ),
         3: ( ( Speeds.portSpeed100Mbps, ( 3, ), () ),
               ( Speeds.portSpeed1Gbps, ( 3, ), () ),
               ( Speeds.portSpeed10Gbps, ( 3, ), () ),
               ( Speeds.portSpeed25Gbps, ( 3, ), () ), ),
      }
      self.populateBasicEthMapping( ethPortIdMap, xcvrSlotId, qsfpEthMapping )

   def genQsfp100SlotMap( self, ethPortIdMap, xcvrSlotId ):
      """Generate a 4 lane configuration like we would want for a QSFP100 slot.
      A QSFP28 slot can support 100G and 4x25G and 2x50G as well as 40G and 4x10G.
      """

      qsfp100EthMapping = {
         0: ( ( Speeds.portSpeed1Gbps, ( 0, ), () ),
               ( Speeds.portSpeed10Gbps, ( 0, ), () ),
               ( Speeds.portSpeed25Gbps, ( 0, ), () ),
               ( Speeds.portSpeed40Gbps, ( 0, 1, 2, 3 ), ( 1, 2, 3 ) ),
               ( Speeds.portSpeed50Gbps, ( 0, 1 ), ( 1, ) ),
               ( Speeds.portSpeed100Gbps, ( 0, 1, 2, 3 ), ( 1, 2, 3 ) ), ),
         1: ( ( Speeds.portSpeed1Gbps, ( 1, ), () ),
               ( Speeds.portSpeed10Gbps, ( 1, ), () ),
               ( Speeds.portSpeed25Gbps, ( 1, ), () ), ),
         2: ( ( Speeds.portSpeed1Gbps, ( 2, ), () ),
               ( Speeds.portSpeed10Gbps, ( 2, ), () ),
               ( Speeds.portSpeed25Gbps, ( 2, ), () ),
               ( Speeds.portSpeed50Gbps, ( 2, 3 ), ( 3, ) ) ),
         3: ( ( Speeds.portSpeed1Gbps, ( 3, ), () ),
               ( Speeds.portSpeed10Gbps, ( 3, ), () ),
               ( Speeds.portSpeed25Gbps, ( 3, ), () ), )
      }

      self.populateBasicEthMapping( ethPortIdMap, xcvrSlotId, qsfp100EthMapping )

   # TODO remove this and use genQsfp100SlotMap
   def genQsfp100SlotMapSingleLane( self, ethPortIdMap, xcvrSlotId ):
      """Generate a 4 lane configuration like we would want for a QSFP100 slot.
      A QSFP28 slot can support 100G and 4x25G and 2x50G as well as 40G and 4x10G.
      """

      qsfp100EthMapping = {
         0: ( ( Speeds.portSpeed40Gbps, ( 0, 1, 2, 3 ), ( 1, 2, 3 ) ),
               ( Speeds.portSpeed100Gbps, ( 0, 1, 2, 3 ), ( 1, 2, 3 ) ), ),
      }

      self.populateBasicEthMapping( ethPortIdMap, xcvrSlotId, qsfp100EthMapping )

   # TODO remove this and use genQsfp100SlotMap
   def genQsfp100SecondarySlotMap( self, ethPortIdBase, slotId ):
      """Generate a QSFP100 port that has a conversion from PAM4 to QSFP100 somewhere
      on the NIF, that is slaved to another QSFP100 port.

      This slaved port only supports 50G and 100G, and only has 2 Ethernet
      interfaces.
      """
      pam4Qsfp100EthMapping = {
         0: ( ( Speeds.portSpeed40Gbps, ( 0, 1, 2, 3 ), ( 1, 2, 3 ) ),
               ( Speeds.portSpeed50Gbps, ( 0, 1 ), ( 1, ) ),
               ( Speeds.portSpeed100Gbps, ( 0, 1, 2, 3 ), ( 1, 2, 3 ) ), ),
         2: ( ( Speeds.portSpeed50Gbps, ( 2, 3 ), ( 3, ) ), ),
      }
      self.populateBasicEthMapping( ethPortIdBase, slotId, pam4Qsfp100EthMapping )

   def genQsfp200SlotMap( self, ethPortIdMap, xcvrSlotId ):
      """Generate a 4 lane configuration like we would want for a QSFP100 slot.
      A QSFP28 slot can support 100G and 4x25G and 2x50G as well as 40G and 4x10G.
      """

      qsfp200EthMapping = {
         0: ( ( Speeds.portSpeed10Gbps, ( 0, ), () ),
               ( Speeds.portSpeed25Gbps, ( 0, ), () ),
               ( Speeds.portSpeed50Gbps1Lane, ( 0, ), () ),
               ( Speeds.portSpeed50Gbps, ( 0, 1 ), ( 1, ) ),
               ( Speeds.portSpeed100Gbps2Lane, ( 0, 1 ), ( 1, ) ),
               ( Speeds.portSpeed40Gbps, ( 0, 1, 2, 3 ), ( 1, 2, 3 ) ),
               ( Speeds.portSpeed100Gbps, ( 0, 1, 2, 3 ), ( 1, 2, 3 ) ),
               ( Speeds.portSpeed200Gbps4Lane, ( 0, 1, 2, 3 ), ( 1, 2, 3 ) ), ),
         1: ( ( Speeds.portSpeed10Gbps, ( 1, ), () ),
               ( Speeds.portSpeed25Gbps, ( 1, ), () ),
               ( Speeds.portSpeed50Gbps1Lane, ( 1, ), () ), ),
         2: ( ( Speeds.portSpeed10Gbps, ( 2, ), () ),
               ( Speeds.portSpeed25Gbps, ( 2, ), () ),
               ( Speeds.portSpeed50Gbps1Lane, ( 2, ), () ),
               ( Speeds.portSpeed50Gbps, ( 2, 3 ), ( 3, ) ),
               ( Speeds.portSpeed100Gbps2Lane, ( 2, 3 ), ( 3, ) ), ),
         3: ( ( Speeds.portSpeed10Gbps, ( 3, ), () ),
               ( Speeds.portSpeed25Gbps, ( 3, ), () ),
               ( Speeds.portSpeed50Gbps1Lane, ( 3, ), () ), )
      }
      self.populateBasicEthMapping( ethPortIdMap, xcvrSlotId, qsfp200EthMapping )

   # TODO remove this and use genQsfp200SlotMap
   def genQsfp200SlotMapSingleLane( self, ethPortIdMap, xcvrSlotId ):
      """Generate a 4 lane configuration like we would want for a QSFP100 slot.
      A QSFP28 slot can support 100G and 4x25G and 2x50G as well as 40G and 4x10G.
      """

      qsfp200EthMapping = {
         0: ( ( Speeds.portSpeed40Gbps, ( 0, 1, 2, 3 ), ( 1, 2, 3 ) ),
               ( Speeds.portSpeed100Gbps, ( 0, 1, 2, 3 ), ( 1, 2, 3 ) ),
               ( Speeds.portSpeed200Gbps4Lane, ( 0, 1, 2, 3 ), ( 1, 2, 3 ) ), ),
      }
      self.populateBasicEthMapping( ethPortIdMap, xcvrSlotId, qsfp200EthMapping )

   def genCfp2_8SlotMap( self, ethPortIdMap, xcvrSlotId ):
      cfp2EthMapping = {
         0: ( ( Speeds.portSpeed10Gbps, ( 0, ), () ),
               ( Speeds.portSpeed25Gbps, ( 0, ), () ),
               ( Speeds.portSpeed40Gbps, ( 0, 1, 2, 3 ), ( 1, 2, 3 ) ),
               ( Speeds.portSpeed50Gbps1Lane, ( 0, ), () ),
               ( Speeds.portSpeed50Gbps, ( 0, 1 ), ( 1, ) ),
               ( Speeds.portSpeed100Gbps2Lane, ( 0, 1 ), ( 1, ) ),
               ( Speeds.portSpeed100Gbps, ( 0, 1, 2, 3 ), ( 1, 2, 3 ) ),
               ( Speeds.portSpeed200Gbps4Lane, ( 0, 1, 2, 3 ), ( 1, 2, 3 ) ),
               ( Speeds.portSpeed400Gbps8Lane, ( 0, 1, 2, 3, 4, 5, 6, 7 ),
                 ( 1, 2, 3, 4, 5, 6, 7 ) ) ),
         1: ( ( Speeds.portSpeed10Gbps, ( 1, ), () ),
               ( Speeds.portSpeed25Gbps, ( 1, ), () ),
               ( Speeds.portSpeed50Gbps1Lane, ( 1, ), () ), ),
         2: ( ( Speeds.portSpeed10Gbps, ( 2, ), () ),
               ( Speeds.portSpeed25Gbps, ( 2, ), () ),
               ( Speeds.portSpeed50Gbps1Lane, ( 2, ), () ),
               ( Speeds.portSpeed50Gbps, ( 2, 3 ), ( 3, ) ),
               ( Speeds.portSpeed100Gbps2Lane, ( 2, 3 ), ( 3, ) ), ),
         3: ( ( Speeds.portSpeed10Gbps, ( 3, ), () ),
               ( Speeds.portSpeed25Gbps, ( 3, ), () ),
               ( Speeds.portSpeed50Gbps1Lane, ( 3, ), () ), ),
         4: ( ( Speeds.portSpeed10Gbps, ( 4, ), () ),
               ( Speeds.portSpeed25Gbps, ( 4, ), () ),
               ( Speeds.portSpeed40Gbps, ( 4, 5, 6, 7 ), ( 5, 6, 7 ) ),
               ( Speeds.portSpeed50Gbps1Lane, ( 4, ), () ),
               ( Speeds.portSpeed50Gbps, ( 4, 5 ), ( 5, ) ),
               ( Speeds.portSpeed100Gbps2Lane, ( 4, 5 ), ( 5, ) ),
               ( Speeds.portSpeed100Gbps, ( 4, 5, 6, 7 ), ( 5, 6, 7 ) ),
               ( Speeds.portSpeed200Gbps4Lane, ( 4, 5, 6, 7 ), ( 5, 6, 7 ) ), ),
         5: ( ( Speeds.portSpeed10Gbps, ( 5, ), () ),
               ( Speeds.portSpeed25Gbps, ( 5, ), () ),
               ( Speeds.portSpeed50Gbps1Lane, ( 5, ), () ), ),
         6: ( ( Speeds.portSpeed10Gbps, ( 6, ), () ),
               ( Speeds.portSpeed25Gbps, ( 6, ), () ),
               ( Speeds.portSpeed50Gbps1Lane, ( 6, ), () ),
               ( Speeds.portSpeed50Gbps, ( 6, 7 ), ( 7, ) ),
               ( Speeds.portSpeed100Gbps2Lane, ( 6, 7 ), ( 7, ) ), ),
         7: ( ( Speeds.portSpeed10Gbps, ( 7, ), () ),
               ( Speeds.portSpeed25Gbps, ( 7, ), () ),
               ( Speeds.portSpeed50Gbps1Lane, ( 7, ), () ), ),
      }

      self.populateBasicEthMapping( ethPortIdMap, xcvrSlotId, cfp2EthMapping )

   # Once arbitrary capabilities are supported, we should be able to use
   # genCfp2_8SlotMap instead of this
   def genCfp2_4SlotMap( self, ethPortIdMap, xcvrSlotId ):
      cfp2EthMapping = {
         0: ( ( Speeds.portSpeed10Gbps, ( 0, ), () ),
               ( Speeds.portSpeed25Gbps, ( 0, ), () ),
               ( Speeds.portSpeed50Gbps1Lane, ( 0, ), () ),
               ( Speeds.portSpeed50Gbps, ( 0, 1 ), ( 1, ) ),
               ( Speeds.portSpeed100Gbps2Lane, ( 0, 1 ), ( 1, ) ),
               ( Speeds.portSpeed40Gbps, ( 0, 1, 2, 3 ), ( 1, 2, 3 ) ),
               ( Speeds.portSpeed100Gbps, ( 0, 1, 2, 3 ), ( 1, 2, 3 ) ),
               ( Speeds.portSpeed200Gbps4Lane, ( 0, 1, 2, 3 ), ( 1, 2, 3 ) ), ),
         1: ( ( Speeds.portSpeed10Gbps, ( 1, ), () ),
               ( Speeds.portSpeed25Gbps, ( 1, ), () ),
               ( Speeds.portSpeed50Gbps1Lane, ( 1, ), () ), ),
         2: ( ( Speeds.portSpeed10Gbps, ( 2, ), () ),
               ( Speeds.portSpeed25Gbps, ( 2, ), () ),
               ( Speeds.portSpeed50Gbps1Lane, ( 2, ), () ),
               ( Speeds.portSpeed50Gbps, ( 2, 3 ), ( 3, ) ),
               ( Speeds.portSpeed100Gbps2Lane, ( 2, 3 ), ( 3, ) ), ),
         3: ( ( Speeds.portSpeed10Gbps, ( 3, ), () ),
               ( Speeds.portSpeed25Gbps, ( 3, ), () ),
               ( Speeds.portSpeed50Gbps1Lane, ( 3, ), () ), )
      }

      self.populateBasicEthMapping( ethPortIdMap, xcvrSlotId, cfp2EthMapping )

   def genOsfpSlotMap( self, ethPortIdMap, xcvrSlotId ):
      """Generate a 8 lane configuration like we would want for a OSFP or QSFP-DD
      slot.
      """

      osfpEthMapping = {
         0: ( ( Speeds.portSpeed10Gbps, ( 0, ), () ),
               ( Speeds.portSpeed25Gbps, ( 0, ), () ),
               ( Speeds.portSpeed40Gbps, ( 0, 1, 2, 3 ), ( 1, 2, 3 ) ),
               ( Speeds.portSpeed50Gbps1Lane, ( 0, ), () ),
               ( Speeds.portSpeed50Gbps, ( 0, 1 ), ( 1, ) ),
               ( Speeds.portSpeed100Gbps2Lane, ( 0, 1 ), ( 1, ) ),
               ( Speeds.portSpeed100Gbps, ( 0, 1, 2, 3 ), ( 1, 2, 3 ) ),
               ( Speeds.portSpeed200Gbps4Lane, ( 0, 1, 2, 3 ), ( 1, 2, 3 ) ),
               ( Speeds.portSpeed400Gbps8Lane, ( 0, 1, 2, 3, 4, 5, 6, 7 ),
                 ( 1, 2, 3, 4, 5, 6, 7 ) ) ),
         1: ( ( Speeds.portSpeed10Gbps, ( 1, ), () ),
               ( Speeds.portSpeed25Gbps, ( 1, ), () ),
               ( Speeds.portSpeed50Gbps1Lane, ( 1, ), () ), ),
         2: ( ( Speeds.portSpeed10Gbps, ( 2, ), () ),
               ( Speeds.portSpeed25Gbps, ( 2, ), () ),
               ( Speeds.portSpeed50Gbps1Lane, ( 2, ), () ),
               ( Speeds.portSpeed50Gbps, ( 2, 3 ), ( 3, ) ),
               ( Speeds.portSpeed100Gbps2Lane, ( 2, 3 ), ( 3, ) ), ),
         3: ( ( Speeds.portSpeed10Gbps, ( 3, ), () ),
               ( Speeds.portSpeed25Gbps, ( 3, ), () ),
               ( Speeds.portSpeed50Gbps1Lane, ( 3, ), () ), ),
         4: ( ( Speeds.portSpeed10Gbps, ( 4, ), () ),
               ( Speeds.portSpeed25Gbps, ( 4, ), () ),
               ( Speeds.portSpeed40Gbps, ( 4, 5, 6, 7 ), ( 5, 6, 7 ) ),
               ( Speeds.portSpeed50Gbps1Lane, ( 4, ), () ),
               ( Speeds.portSpeed50Gbps, ( 4, 5 ), ( 5, ) ),
               ( Speeds.portSpeed100Gbps2Lane, ( 4, 5 ), ( 5, ) ),
               ( Speeds.portSpeed100Gbps, ( 4, 5, 6, 7 ), ( 5, 6, 7 ) ),
               ( Speeds.portSpeed200Gbps4Lane, ( 4, 5, 6, 7 ), ( 5, 6, 7 ) ), ),
         5: ( ( Speeds.portSpeed10Gbps, ( 5, ), () ),
               ( Speeds.portSpeed25Gbps, ( 5, ), () ),
               ( Speeds.portSpeed50Gbps1Lane, ( 5, ), () ), ),
         6: ( ( Speeds.portSpeed10Gbps, ( 6, ), () ),
               ( Speeds.portSpeed25Gbps, ( 6, ), () ),
               ( Speeds.portSpeed50Gbps1Lane, ( 6, ), () ),
               ( Speeds.portSpeed50Gbps, ( 6, 7 ), ( 7, ) ),
               ( Speeds.portSpeed100Gbps2Lane, ( 6, 7 ), ( 7, ) ), ),
         7: ( ( Speeds.portSpeed10Gbps, ( 7, ), () ),
               ( Speeds.portSpeed25Gbps, ( 7, ), () ),
               ( Speeds.portSpeed50Gbps1Lane, ( 7, ), () ), ),
      }

      self.populateBasicEthMapping( ethPortIdMap, xcvrSlotId, osfpEthMapping )

   # TODO remove this and use osfp slot map once we can handle arbitrary capabilities
   def genOsfp4LaneSlotMap( self, ethPortIdMap, xcvrSlotId ):
      """Generate a 8 lane configuration like we would want for a OSFP or QSFP-DD
      slot.
      """

      osfpEthMapping = {
         0: ( ( Speeds.portSpeed100Gbps2Lane, ( 0, 1 ), ( 1, ) ),
               ( Speeds.portSpeed100Gbps, ( 0, 1, 2, 3 ), ( 1, 2, 3 ) ),
               ( Speeds.portSpeed200Gbps4Lane, ( 0, 1, 2, 3 ), ( 1, 2, 3 ) ),
               ( Speeds.portSpeed400Gbps8Lane, ( 0, 1, 2, 3, 4, 5, 6, 7 ),
                 ( 1, 2, 3, 4, 5, 6, 7 ) ) ),
         2: ( ( Speeds.portSpeed100Gbps2Lane, ( 2, 3 ), ( 3, ) ), ),
         4: ( ( Speeds.portSpeed100Gbps2Lane, ( 4, 5 ), ( 5, ) ),
               ( Speeds.portSpeed100Gbps, ( 4, 5, 6, 7 ), ( 5, 6, 7 ) ),
               ( Speeds.portSpeed200Gbps4Lane, ( 4, 5, 6, 7 ), ( 5, 6, 7 ) ), ),
         6: ( ( Speeds.portSpeed100Gbps2Lane, ( 6, 7 ), ( 7, ) ), ),
      }

      self.populateBasicEthMapping( ethPortIdMap, xcvrSlotId, osfpEthMapping )

   # TODO remove this and use osfp slot map once we can handle arbitrary capabilities
   def genOsfp6LaneSlotMap( self, ethPortIdMap, xcvrSlotId ):
      """Generate a configuration like we would want for a OSFP or QSFP-DD
      slot.  Includes support for 4x10g, 4x25g, 40g.  Not supporting 8xN (for
      example 8x25) as that would eat up to many ports.
      """

      osfpEthMapping = {
         0: ( ( Speeds.portSpeed10Gbps, ( 0, ), () ),
               ( Speeds.portSpeed25Gbps, ( 0, ), () ),
               ( Speeds.portSpeed40Gbps, ( 0, 1, 2, 3 ), ( 1, 2, 3 ) ),
               ( Speeds.portSpeed50Gbps1Lane, ( 0, ), () ),
               ( Speeds.portSpeed50Gbps, ( 0, 1 ), ( 1, ) ),
               ( Speeds.portSpeed100Gbps2Lane, ( 0, 1 ), ( 1, ) ),
               ( Speeds.portSpeed100Gbps, ( 0, 1, 2, 3 ), ( 1, 2, 3 ) ),
               ( Speeds.portSpeed200Gbps4Lane, ( 0, 1, 2, 3 ), ( 1, 2, 3 ) ),
               ( Speeds.portSpeed400Gbps8Lane, ( 0, 1, 2, 3, 4, 5, 6, 7 ),
                 ( 1, 2, 3, 4, 5, 6, 7 ) ) ),
         1: ( ( Speeds.portSpeed10Gbps, ( 1, ), () ),
               ( Speeds.portSpeed25Gbps, ( 1, ), () ),
               ( Speeds.portSpeed50Gbps1Lane, ( 1, ), () ), ),
         2: ( ( Speeds.portSpeed10Gbps, ( 2, ), () ),
               ( Speeds.portSpeed25Gbps, ( 2, ), () ),
               ( Speeds.portSpeed50Gbps1Lane, ( 2, ), () ),
               ( Speeds.portSpeed50Gbps, ( 2, 3 ), ( 3, ) ),
               ( Speeds.portSpeed100Gbps2Lane, ( 2, 3 ), ( 3, ) ), ),
         3: ( ( Speeds.portSpeed10Gbps, ( 3, ), () ),
               ( Speeds.portSpeed25Gbps, ( 3, ), () ),
               ( Speeds.portSpeed50Gbps1Lane, ( 3, ), () ), ),
         4: ( ( Speeds.portSpeed40Gbps, ( 4, 5, 6, 7 ), ( 5, 6, 7 ) ),
               ( Speeds.portSpeed50Gbps, ( 4, 5 ), ( 5, ) ),
               ( Speeds.portSpeed100Gbps2Lane, ( 4, 5 ), ( 5, ) ),
               ( Speeds.portSpeed100Gbps, ( 4, 5, 6, 7 ), ( 5, 6, 7 ) ),
               ( Speeds.portSpeed200Gbps4Lane, ( 4, 5, 6, 7 ), ( 5, 6, 7 ) ), ),
         6: ( ( Speeds.portSpeed50Gbps, ( 6, 7 ), ( 7, ) ),
               ( Speeds.portSpeed100Gbps2Lane, ( 6, 7 ), ( 7, ) ), ),
      }

      self.populateBasicEthMapping( ethPortIdMap, xcvrSlotId, osfpEthMapping )

   # BUG567064 : remove qsfpDdTh4 slotMap when alternative is available
   def genOsfpTh4SlotMap( self, ethPortIdMap, xcvrSlotId ):
      """Generate a 8 lane configuration like we would want for a OSFP or QSFP-DD
      slot.
      """

      osfpEthMapping = {
         0: ( ( Speeds.portSpeed10Gbps, ( 0, ), () ),
               ( Speeds.portSpeed25Gbps, ( 0, ), () ),
               ( Speeds.portSpeed40Gbps, ( 0, 1, 2, 3 ), ( 1, 2, 3 ) ),
               ( Speeds.portSpeed50Gbps1Lane, ( 0, ), () ),
               ( Speeds.portSpeed50Gbps, ( 0, 1 ), ( 1, ) ),
               ( Speeds.portSpeed100Gbps2Lane, ( 0, 1 ), ( 1, ) ),
               ( Speeds.portSpeed100Gbps, ( 0, 1, 2, 3 ), ( 1, 2, 3 ) ),
               ( Speeds.portSpeed200Gbps4Lane, ( 0, 1, 2, 3 ), ( 1, 2, 3 ) ),
               ( Speeds.portSpeed400Gbps8Lane, ( 0, 1, 2, 3, 4, 5, 6, 7 ),
                 ( 1, 2, 3, 4, 5, 6, 7 ) ) ),
         1: ( ( Speeds.portSpeed10Gbps, ( 1, ), () ),
               ( Speeds.portSpeed25Gbps, ( 1, ), () ),
               ( Speeds.portSpeed50Gbps1Lane, ( 1, ), () ), ),
         2: ( ( Speeds.portSpeed10Gbps, ( 2, ), () ),
               ( Speeds.portSpeed25Gbps, ( 2, ), () ),
               ( Speeds.portSpeed50Gbps1Lane, ( 2, ), () ),
               ( Speeds.portSpeed50Gbps, ( 2, 3 ), ( 3, ) ),
               ( Speeds.portSpeed100Gbps2Lane, ( 2, 3 ), ( 3, ) ), ),
         3: ( ( Speeds.portSpeed10Gbps, ( 3, ), () ),
               ( Speeds.portSpeed25Gbps, ( 3, ), () ),
               ( Speeds.portSpeed50Gbps1Lane, ( 3, ), () ), ),
         4: ( ( Speeds.portSpeed40Gbps, ( 4, 5, 6, 7 ), ( 5, 6, 7 ) ),
               ( Speeds.portSpeed50Gbps1Lane, ( 4, ), () ),
               ( Speeds.portSpeed50Gbps, ( 4, 5 ), ( 5, ) ),
               ( Speeds.portSpeed100Gbps2Lane, ( 4, 5 ), ( 5, ) ),
               ( Speeds.portSpeed100Gbps, ( 4, 5, 6, 7 ), ( 5, 6, 7 ) ),
               ( Speeds.portSpeed200Gbps4Lane, ( 4, 5, 6, 7 ), ( 5, 6, 7 ) ), ),
         6: ( ( Speeds.portSpeed50Gbps1Lane, ( 6, ), () ),
               ( Speeds.portSpeed50Gbps, ( 6, 7 ), ( 7, ) ),
               ( Speeds.portSpeed100Gbps2Lane, ( 6, 7 ), ( 7, ) ), ),
      }

      self.populateBasicEthMapping( ethPortIdMap, xcvrSlotId, osfpEthMapping )

   def genOsfpSlotMapSingleLane( self, ethPortIdMap, xcvrSlotId ):
      """Generate a 8 lane configuration like we would want for a OSFP or QSFP-DD
      slot.
      """

      osfpEthMapping = {
         0: ( ( Speeds.portSpeed40Gbps, ( 0, 1, 2, 3 ), ( 1, 2, 3 ) ),
               ( Speeds.portSpeed100Gbps, ( 0, 1, 2, 3 ), ( 1, 2, 3 ) ),
               ( Speeds.portSpeed200Gbps4Lane, ( 0, 1, 2, 3 ), ( 1, 2, 3 ) ),
               ( Speeds.portSpeed400Gbps8Lane, ( 0, 1, 2, 3, 4, 5, 6, 7 ),
                 ( 1, 2, 3, 4, 5, 6, 7 ) ) ),
      }

      self.populateBasicEthMapping( ethPortIdMap, xcvrSlotId, osfpEthMapping )

   def genOsfp800SlotMap( self, ethPortIdMap, xcvrSlotId ):
      """Generate a 8 lane configuration like we would want for a OSFP or QSFP-DD
      800G slot.
      """
      osfpEthMapping = {
         0: ( ( Speeds.portSpeed10Gbps, ( 0, ), () ),
               ( Speeds.portSpeed25Gbps, ( 0, ), () ),
               ( Speeds.portSpeed40Gbps, ( 0, 1, 2, 3 ), ( 1, 2, 3 ) ),
               ( Speeds.portSpeed50Gbps1Lane, ( 0, ), () ),
               ( Speeds.portSpeed50Gbps, ( 0, 1 ), ( 1, ) ),
               ( Speeds.portSpeed100Gbps1Lane, ( 0, ), () ),
               ( Speeds.portSpeed100Gbps2Lane, ( 0, 1 ), ( 1, ) ),
               ( Speeds.portSpeed100Gbps, ( 0, 1, 2, 3 ), ( 1, 2, 3 ) ),
               ( Speeds.portSpeed200Gbps2Lane, ( 0, 1 ), ( 1, ) ),
               ( Speeds.portSpeed200Gbps4Lane, ( 0, 1, 2, 3 ), ( 1, 2, 3 ) ),
               ( Speeds.portSpeed400Gbps4Lane, ( 0, 1, 2, 3 ), ( 1, 2, 3 ) ),
               ( Speeds.portSpeed400Gbps8Lane, ( 0, 1, 2, 3, 4, 5, 6, 7 ),
                 ( 1, 2, 3, 4, 5, 6, 7 ) ),
               ( Speeds.portSpeed800Gbps8Lane, ( 0, 1, 2, 3, 4, 5, 6, 7 ),
                 ( 1, 2, 3, 4, 5, 6, 7 ) ), ),
         1: ( ( Speeds.portSpeed10Gbps, ( 1, ), () ),
               ( Speeds.portSpeed25Gbps, ( 1, ), () ),
               ( Speeds.portSpeed50Gbps1Lane, ( 1, ), () ),
               ( Speeds.portSpeed100Gbps1Lane, ( 1, ), () ), ),
         2: ( ( Speeds.portSpeed10Gbps, ( 2, ), () ),
               ( Speeds.portSpeed25Gbps, ( 2, ), () ),
               ( Speeds.portSpeed50Gbps1Lane, ( 2, ), () ),
               ( Speeds.portSpeed50Gbps, ( 2, 3 ), ( 3, ) ),
               ( Speeds.portSpeed100Gbps2Lane, ( 2, 3 ), ( 3, ) ),
               ( Speeds.portSpeed100Gbps1Lane, ( 2, ), () ),
               ( Speeds.portSpeed200Gbps2Lane, ( 2, 3 ), ( 3, ) ), ),
         3: ( ( Speeds.portSpeed10Gbps, ( 3, ), () ),
               ( Speeds.portSpeed25Gbps, ( 3, ), () ),
               ( Speeds.portSpeed50Gbps1Lane, ( 3, ), () ),
               ( Speeds.portSpeed100Gbps1Lane, ( 3, ), () ), ),
         4: ( ( Speeds.portSpeed10Gbps, ( 4, ), () ),
               ( Speeds.portSpeed25Gbps, ( 4, ), () ),
               ( Speeds.portSpeed40Gbps, ( 4, 5, 6, 7 ), ( 5, 6, 7 ) ),
               ( Speeds.portSpeed50Gbps1Lane, ( 4, ), () ),
               ( Speeds.portSpeed50Gbps, ( 4, 5 ), ( 5, ) ),
               ( Speeds.portSpeed100Gbps1Lane, ( 4, ), () ),
               ( Speeds.portSpeed100Gbps2Lane, ( 4, 5 ), ( 5, ) ),
               ( Speeds.portSpeed100Gbps, ( 4, 5, 6, 7 ), ( 5, 6, 7 ) ),
               ( Speeds.portSpeed200Gbps2Lane, ( 4, 5 ), ( 5, ) ),
               ( Speeds.portSpeed200Gbps4Lane, ( 4, 5, 6, 7 ), ( 5, 6, 7 ) ),
               ( Speeds.portSpeed400Gbps4Lane, ( 4, 5, 6, 7 ), ( 5, 6, 7 ) ), ),
         5: ( ( Speeds.portSpeed10Gbps, ( 5, ), () ),
               ( Speeds.portSpeed25Gbps, ( 5, ), () ),
               ( Speeds.portSpeed50Gbps1Lane, ( 5, ), () ),
               ( Speeds.portSpeed100Gbps1Lane, ( 5, ), () ), ),
         6: ( ( Speeds.portSpeed10Gbps, ( 6, ), () ),
               ( Speeds.portSpeed25Gbps, ( 6, ), () ),
               ( Speeds.portSpeed50Gbps1Lane, ( 6, ), () ),
               ( Speeds.portSpeed50Gbps, ( 6, 7 ), ( 7, ) ),
               ( Speeds.portSpeed100Gbps1Lane, ( 6, ), () ),
               ( Speeds.portSpeed100Gbps2Lane, ( 6, 7 ), ( 7, ) ),
               ( Speeds.portSpeed200Gbps2Lane, ( 6, 7 ), ( 7, ) ), ),
         7: ( ( Speeds.portSpeed10Gbps, ( 7, ), () ),
               ( Speeds.portSpeed25Gbps, ( 7, ), () ),
               ( Speeds.portSpeed50Gbps1Lane, ( 7, ), () ),
               ( Speeds.portSpeed100Gbps1Lane, ( 7, ), () ), ),
      }
      self.populateBasicEthMapping( ethPortIdMap, xcvrSlotId, osfpEthMapping )

   def parseSimplePortTable( self, portTable ):
      """
      Parses a mimimalistic table which can be used to
      configure most ports in a system using this helper
      class.
      [ ( type, { kwargs } ), ... ]
      # where type is 'sfp', 'qsfp', etc ...
      """
      for portType, ethPortIdMap, xcvrSlotId in portTable:
         if portType == 'qsfp40':
            self.genQsfp40SlotMap( ethPortIdMap, xcvrSlotId )
         elif portType == 'qsfp100':
            self.genQsfp100SlotMap( ethPortIdMap, xcvrSlotId )
         elif portType == 'qsfp100Single':
            self.genQsfp100SlotMapSingleLane( ethPortIdMap, xcvrSlotId )
         elif portType == 'qsfp100Secondary':
            self.genQsfp100SecondarySlotMap( ethPortIdMap, xcvrSlotId )
         elif portType == 'qsfp200':
            self.genQsfp200SlotMap( ethPortIdMap, xcvrSlotId )
         elif portType == 'qsfp200Single':
            self.genQsfp200SlotMapSingleLane( ethPortIdMap, xcvrSlotId )
         elif portType == 'sfp50':
            self.genSfp50SlotMap( ethPortIdMap, xcvrSlotId )
         elif portType == 'sfp25':
            self.genSfp25SlotMap( ethPortIdMap, xcvrSlotId )
         elif portType == 'sfp10':
            self.genSfp10SlotMap( ethPortIdMap, xcvrSlotId )
         elif portType == 'sfp1':
            self.genSfp1SlotMap( ethPortIdMap, xcvrSlotId )
         elif portType == 'sfp':
            self.genSfpSlotMap( ethPortIdMap, xcvrSlotId )
         elif portType == 'dsfp':
            self.genDsfpSlotMap( ethPortIdMap, xcvrSlotId )
         elif portType in ( 'osfp', 'qsfpDd', ):
            self.genOsfpSlotMap( ethPortIdMap, xcvrSlotId )
         elif portType in ( 'osfp4Lane', 'qsfpDd4Lane', ):
            self.genOsfp4LaneSlotMap( ethPortIdMap, xcvrSlotId )
         elif portType in ( 'osfp6Lane', 'qsfpDd6Lane', ):
            self.genOsfp6LaneSlotMap( ethPortIdMap, xcvrSlotId )
         elif portType == 'osfpTh4':
            self.genOsfpTh4SlotMap( ethPortIdMap, xcvrSlotId )
         elif portType in ( 'osfpSingle', 'qsfpDdSingle' ):
            self.genOsfpSlotMapSingleLane( ethPortIdMap, xcvrSlotId )
         elif portType in ( 'osfp800', 'qsfpDd800' ):
            self.genOsfp800SlotMap( ethPortIdMap, xcvrSlotId )
         elif portType == 'RJ45':
            self.genRj45SlotMap( ethPortIdMap, xcvrSlotId )
         elif portType == 'cfp2-8':
            self.genCfp2_8SlotMap( ethPortIdMap, xcvrSlotId )
         elif portType == 'cfp2-4':
            self.genCfp2_4SlotMap( ethPortIdMap, xcvrSlotId )
         elif portType == 'sfpDd':
            self.genSfpDdSlotMap( ethPortIdMap, xcvrSlotId )
         else:
            assert 0

   def configure( self ):
      """
      Parses the contents of self.laneMap and populates the FDL.
      """
      for ethPortId, configs in self.laneMap:
         for config in configs:
            configId, speeds, subordinatePorts, laneCfgs = config
            portLaneConfig = self.portLaneConfigDir.newPortLaneConfig( ethPortId )
            portLaneConfig.port = self.ethPortDir.port[ ethPortId ]
            portSpeedConfig = portLaneConfig.newPortSpeedConfig( configId )

            for speed in speeds:
               portSpeedConfig.supportedSpeed[ speed ] = True

            for subPortId in subordinatePorts:
               portSpeedConfig.subordinatePort[ subPortId ] = \
                   self.ethPortDir.port[ subPortId ]

            for ( ethLane, xcvrSlotId, xcvrLane, used ) in laneCfgs:
               slot = self.xcvrSlotDir.xcvrSlot[ xcvrSlotId ]
               portSpeedConfig.newPortXcvrConfig( ethLane, slot, xcvrLane, used )
