#!/usr/bin/env python3
# Copyright (c) 2008-2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

from __future__ import absolute_import, division, print_function
from typing import Union, Optional

import math
import re
import Arnet
import Cell
import Logging
import Plugins
import csv
from TypeFuture import TacLazyType
from collections import namedtuple

MediaType = TacLazyType( "Xcvr::MediaType" )
XcvrApi = TacLazyType( "Xcvr::XcvrApi" )
XcvrType = TacLazyType( "Xcvr::XcvrType" )
XcvrNewStatus = TacLazyType( "Xcvr::XcvrNewStatus" )
MgmtIntfId = TacLazyType( "Arnet::MgmtIntfId" )
EthIntfId = TacLazyType( "Arnet::EthIntfId" )
FabricIntfId = TacLazyType( "Arnet::FabricIntfId" )

ConfigToModelMapping = {
   'host': {
      'NoLoopback': 'none',
      'Input': 'system',
      'Output': 'network',
   },
   'media': {
      'NoLoopback': 'none',
      'Input': 'network',
      'Output': 'system',
   }
}

TRANSCEIVER_POWER_CYCLE = Logging.LogHandle(
   "TRANSCEIVER_POWER_CYCLE",
   severity=Logging.logInfo,
   fmt="Transceiver slots %s were power cycled.",
   explanation="The transceivers were power cycled due to user configuration.",
   recommendedAction=Logging.NO_ACTION_REQUIRED
)

def selectCollection( loopbackCfg, isHost: Union[ bool, str ] ):
   isHost = isHost in ( True, 'host' )
   return loopbackCfg.hostLoopback if isHost else loopbackCfg.mediaLoopback

def internalLaneIdToCli( lane ):
   return lane + 1

def populateAttenuationMap( attnTable ):
   Plugins.loadPlugins( 'XcvrPlugin', context=attnTable )

def allStatusDirType():
   return 'Xcvr::AllStatusDir'

def getXcvrSlotName( intfName ):
   """
   Gets the name of the transceiver slot given an Ethernet interface name.

   Important note: do NOT use this method in ptests, use the XcvrPtestLib
   implementation instead

   Parameters
   ----------
   intfName : str
   """
   # Assume this is a fixed-system if cellType is not 'supervisor'.
   # Breadth tests will have a cellType of 'generic' by default which
   # we will assume to be fixed-system.
   isModular = Cell.cellType() == 'supervisor'
   fixedSystemWithCardSlots = not isModular and Cell.cardSlotsSupported()
   return getXcvrSlotNameHelper( intfName, isModular,
                                 fixedSystemWithCardSlots=fixedSystemWithCardSlots )

def getXcvrSlotNameHelper( intfName, isModular, fixedSystemWithCardSlots=False ):
   """
   Helper function to get the name of a transceiver slot given an
   Ethernet interface name.

   Parameters
   ----------
   intfName : str
      Name of an Ethernet interface
   isModular : bool
      True if the system is a modular system, False otherwise in which
      case we assume fixed-system.
   fixedSystemWithCardSlots : bool (optional)
      Defaults to False.  If True, then it indicates a special scenario
      where our fixed-system has card slots (like NIM cards on Caravan platforms).
   """
   # Cannot be both a modular and fixed-system.  Assert on user input error.
   if isModular:
      assert not fixedSystemWithCardSlots

   fwdSlashIdx = [ i for i, ltr in enumerate( intfName ) if ltr == '/' ]
   numFwdSlash = len( fwdSlashIdx )
   trueFixedSystem = not isModular and not fixedSystemWithCardSlots
   if( ( trueFixedSystem and numFwdSlash == 1 ) or
       ( ( isModular or fixedSystemWithCardSlots ) and numFwdSlash == 2 ) ):
      # Chop-off everything including and after the last forward slash
      return intfName[ 0 : fwdSlashIdx[ -1 ] ]
   return intfName

def getLinecardName( intfName ):
   m = re.match( "Ethernet([0-9]+)/([0-9]+)", intfName )
   assert m, 'Could not parse interface: %s' % intfName
   return 'Linecard%s' % m.group( 1 )

def isManagementIntf( intfName: str ):
   return re.match( r"Management\d+/\d+", intfName ) is not None

def isPrimaryIntf( intfName ):
   """
   Returns True if the passed in interface name is a primary interface.
   A primary interface is characterized as the interface associated with lane 1
   if its a multi-lane interface.  In other words, the interface ending in "/1".
   If the interface is not multi-lane, for example an SFP port has a single
   lane and single interface, then this is also considered a primary interface.
   Returns False if the passed in interface name is not a primary interface.

   Note: This function depends on Cell/cellType/cellStatus being setup
   correctly, so this function should not be used in product tests.

   Parameters
   ----------
   intfName : str

   Returns
   -------
   bool
   """
   isModular = Cell.cellType() == 'supervisor'
   fixedSystemWithCardSlots = not isModular and Cell.cardSlotsSupported()
   return isPrimaryIntfHelper( intfName, isModular,
         fixedSystemWithCardSlots=fixedSystemWithCardSlots )

def primaryIntfSuffix() -> str:
   '''
   Currently, returns /1.
   May not be always correct if a module has multiple lasers
   '''
   return '/1'

def isPrimaryIntfHelper( intfName, isModular, fixedSystemWithCardSlots=False ):
   """
   Parameters
   ----------
   intfName : str
   isModular: bool
   fixedSystemWithCardSlots : bool (optional)
      Defaults to False.  If True, then it indicates a special scenario
      where our fixed-system has card slots (like NIM cards on Caravan platforms).

   Returns
   -------
   True if intfName is a primary interface.  Returns False otherwise.
   """
   hasPrimaryIntfSuffix = intfName[ -2 : ] == primaryIntfSuffix()
   if isModular or fixedSystemWithCardSlots:
      return intfName.count( '/' ) != 2 or hasPrimaryIntfSuffix
   else:
      return '/' not in intfName or hasPrimaryIntfSuffix

def isCfp2DcoSlotWithNonDcoInserted( status ):
   """
   This function returns True if this is a DCO slot with a non-DCO
   transceiver inserted.

   gyorgym Note:  A little context as to why this function exists.  For some
   reason CFP2-DCO slots do not provide the 'correct' host-lane to interface-name
   mapping in xcvrConfig->intfName.  To work around this on the CLI, we have this
   function to help us identify this special case.

   Parameters
   ----------
   status : Xcvr::XcvrNewStatus
      The status associated with the slot.

   Returns
   -------
   bool
   """
   return ( status.xcvrType == XcvrType.cfp2 and
            status.cfp2SlotCapabilities.cfp2DcoSupported and
            status.mediaTypeString != "100G-DWDM-DCO" )

def getNumIntfs( status ):
   """
   Parameters
   ----------
   status : Xcvr::XcvrNewStatus

   Returns
   -------
   int
      The number of interfaces associated with the transceiver slot
   """
   if isCfp2DcoSlotWithNonDcoInserted( status ):
      # Each interface of cfp2-dco capable port represents 100G host interface.
      # If non-coherent 100G CFP2 transceiver is inserted, set number of intfs to 1
      # to pretend like the /2 interface does not exist.
      return 1
   return len( status.xcvrConfig.intfName )

def getIntfNameByHostLane( laneId, status, maxChannels ):
   """
   Gets the interface name associated with the passed in host electrical lane ID.

   Parameters
   ----------
   laneId : int
      Represents the host electrical lane which is indexed starting at 0.

   status : Xcvr::XcvrNewStatus
      A object derived from XcvrNewStatus.  This should be a leaf XcvrNewStatus,
      meaning the XcvrNewStatus associated with the inserted transceiver.

   maxChannels : int
      Number of host electrical lanes belonging to the inserted transceiver

   Returns
   -------
   interface name : str
      Returns a string representing the full interface name, for example
      "Ethernet3/1/8".
   """
   assert laneId >= 0
   numIntfs = getNumIntfs( status )
   intfNameMap = status.xcvrConfig.intfName

   if isCfp2DcoSlotWithNonDcoInserted( status ):
      # This is a really weird special case.  When a non-DCO is inserted inside
      # CFP2-DCO slot, we pretend like there is only a /1 interface.  That's why
      # we spoof the 'intfNameMap' here.
      assert numIntfs == 1
      intfNameMap = {}
      intfNameMap[ 0 ] = status.xcvrConfig.intfName[ 0 ]

   if maxChannels > numIntfs:
      assert 0 in intfNameMap
      tmpHostLaneId = laneId
      while tmpHostLaneId not in intfNameMap:
         # While the collection does not have a key for this tmpHostLaneId keep
         # searching for a lower host lane which has an interface name mapping.
         # Lane 0 is always guaranteed to be in the xcvrConfig->intfName collection,
         # so this while loop is guaranteed to exit.
         tmpHostLaneId -= 1

      intfName = intfNameMap[ tmpHostLaneId ]
   else:
      # gyorgym
      # This 'if-else' probably doesn't need to exist, but I'm to afraid
      # to change this '.get()' behavior right now.  We can change it later.
      intfName = intfNameMap.get( laneId )

   return intfName

def getAndFilterPrintableIntfNameByLane( laneId, status, maxChannels, intfNames ):
   """
   This function uses 'laneId' to get an interface name, filters the interface name
   against the passed in 'intfNames' collection, returning an empty sting if the
   interface name does not pass the filter, otherwise returning a printable
   interface name if the filter is passed.

   Yeah...this function does a little bit too much.

   Parameters
   ----------
   laneId : int
      The host electrical lane ID which start indexing at 0

   status : Xcvr::XcvrNewStatus
      Derived XcvrNewStatus object.  This should be a leaf XcvrNewStatus, meaning
      the XcvrNewStatus associated with the inserted transceiver.

   maxChannels : int
      The number of host electrical lanes belonging to the inserted transceiver

   intfNames : list( str )
      List of interface names to filter against.

   Returns
   -------
   printable interface name : str
      Returns a printable interface name (the name we print on the CLI
      for this host electrical lane).
   """
   intfName = getIntfNameByHostLane( laneId, status, maxChannels )

   if intfName not in intfNames:
      # XXX_LWR: we have to be careful that the names for
      #          interfaces that correspond to multi-lane xcvrs
      #          are handled properly by the interface range
      #          stuff.
      return ""

   numIntfs = getNumIntfs( status )
   if maxChannels > numIntfs:
      # Always append lane to the printable interface name, even if the
      # interface does not specify one (e.g. 40G-only interface).
      name = "%s/%d" % ( getXcvrSlotName( intfName ), laneId + 1 )
   else:
      name = intfName
   return name

def getLaneId( intfName ):
   if MgmtIntfId.isMgmtIntfId( intfName ):
      return 0
   elif FabricIntfId.isFabricIntfId( intfName ):
      return FabricIntfId.port( intfName )
   return EthIntfId.lane( intfName )

def getLineSideChannels( status ):
   if status.mediaTypeString == "100G-DWDM-DCO":
      # In CFP2-DCO transceiver, there's only one network interface,
      # while maxChannels is currently representing host interfaces.
      # We only want to show information for interface /1
      # BUG370088: the current status.capabilities.lineSideChannels for
      #            CFP2-DCO is set to 2 which is wrong. CFP2-DCO only has
      #            1 optical channel
      lineSideChannels = 1
   elif status.capabilities.muxponderModeCapable and status.muxponderMode:
      # Picking up the active application on the first data path lane
      activeApp = status.applicationStatus.activeApplication.get( 0 )
      apSel = activeApp.apSel if activeApp else None
      maxHostLanes = status.capabilities.maxChannels
      appAdv = status.eepromContents.application.get( apSel ) if apSel else None
      # If we found the active application in the set of advertised applications,
      # we'll use the number of host lanes it reports, otherwise, we'll fall back to
      # the maximum number of host lanes
      numHostLanes = appAdv.hostLaneCount if appAdv else maxHostLanes
      # ZR/ZRP only have 1 media channel, which doesn't really work with most of out
      # CLI code where we assume that each interface has a media channel associated
      # with it. So we'll tell CLI that we have some other number of channels, based
      # on the active application: there will be an interface for every group of host
      # lanes associated with the application.
      lineSideChannels = maxHostLanes // numHostLanes
   else:
      lineSideChannels = status.capabilities.lineSideChannels
   return lineSideChannels

def getPrimaryIntfName( primaryIntfNames, inactiveIntfName ):
   """
   This function is used to get the primary interface that an inactive interface
   is subsumed to when it is known an inactive interface is being passed in.

   This works on the assumption that the primary interface can be determined
   by finding the nearest primary lane that is smaller than the current
   interface's lane. (e.g. et1/1 is the primary interface for et1/2,
   whereas et1/3 will never be et1/2's primary interface)
   """
   if primaryIntfNames and inactiveIntfName:
      curIntfLaneNum = inactiveIntfName.split( '/' )
      intfNamesMaxToMin = Arnet.sortIntf( primaryIntfNames )
      # Handle Agile which are SFP and don't have sub intf lanes
      if len( curIntfLaneNum ) == 1 or len( intfNamesMaxToMin ) == 1:
         return intfNamesMaxToMin[ 0 ]
      intfNamesMaxToMin.reverse()
      for intf in intfNamesMaxToMin:
         if int( intf.split( '/' )[ -1 ] ) > int( curIntfLaneNum[ -1 ] ):
            continue
         return intf
   # Could not find a primary intf
   return None

# Helper functions. Return true if this is the specified form-factor adapter
def isQsfpToSfpSwizzler( status ):
   return status and status.xcvrType == XcvrType.qsfpPlus and status.sfpStatus

def isOsfpToQsfpSwizzler( status ):
   return status and status.xcvrType == XcvrType.osfp and status.qsfpStatus

def isQsfpDdToQsfpSwizzler( status ):
   return status and status.xcvrType == XcvrType.qsfpDd and status.qsfpStatus

def isQsfpCmisToQsfpAdapter( status ):
   return status and status.xcvrType == XcvrType.qsfpCmis and status.qsfpStatus

def isCmisToSfpAdapter( status ):
   return status and isCmisType( status.xcvrType ) and status.sfpStatus

def getXcvrStatus( status ):
   # Check if this is a form-factor adapter
   if isQsfpToSfpSwizzler( status ) or isCmisToSfpAdapter( status ):
      return status.sfpStatus
   if isOsfpToQsfpSwizzler( status ):
      if isQsfpToSfpSwizzler( status.qsfpStatus ):
         return status.qsfpStatus.sfpStatus
      return status.qsfpStatus
   if isQsfpDdToQsfpSwizzler( status ):
      if isQsfpToSfpSwizzler( status.qsfpStatus ):
         return status.qsfpStatus.sfpStatus
      return status.qsfpStatus
   if isQsfpCmisToQsfpAdapter( status ):
      if isQsfpToSfpSwizzler( status.qsfpStatus ):
         return status.qsfpStatus.sfpStatus
      return status.qsfpStatus
   return status

def getXcvrConfig( status, config ):
   # Check if this is a form-factor adapter
   if isQsfpToSfpSwizzler( status ):
      return status.sfpConfig
   if isOsfpToQsfpSwizzler( status ):
      if isQsfpToSfpSwizzler( status.qsfpStatus ):
         return status.qsfpStatus.sfpConfig
      return status.qsfpConfig
   if isQsfpDdToQsfpSwizzler( status ):
      if isQsfpToSfpSwizzler( status.qsfpStatus ):
         return status.qsfpStatus.sfpConfig
      return status.qsfpConfig
   if isQsfpCmisToQsfpAdapter( status ):
      if isQsfpToSfpSwizzler( status.qsfpStatus ):
         return status.qsfpStatus.sfpConfig
      return status.qsfpConfig
   return config

# Check if the mediaType is valid in this slot
def validMediaType( mediaType, xcvrStatus ):
   if not xcvrStatus:
      return False
   return xcvrStatus.isSupportedMediaType( mediaType )

def isCmisType( xcvrType ):
   """
   Determines if the passed in XcvrType is a CMIS type.

   Parameters
   ----------
   xcvrType : Xcvr::XcvrType

   Returns
   -------
   bool
      Returns true if the passed in xcvrType is a CMIS type, false otherwise.
   """
   return XcvrApi.isCmisType( xcvrType )

def isCmisTransceiver( xcvrStatus ):
   """
   Determines if the passed in transceiver is a CMIS transceiver.

   NOTE: We use XcvrStatus.typeInUse here, so this function uses the type of the
   inserted transceiver--not the type of the slot.

   Parameters
   ----------
   xcvrStatus : Xcvr::XcvrNewStatus derived object

   Returns
   -------
   bool
      Returns true if the inserted transceiver is a CMIS transceiver, false
   otherwise.
   """
   return isCmisType( xcvrStatus.typeInUse )

def isSingleLaneTransceiver( xcvrStatus ):
   return xcvrStatus.typeInUse == XcvrType.sfpPlus

def isCmisTypeStr( xcvrTypeStr ):
   """
   Determines whether the passed in transceiver type is classified as a CMIS
   transceiver type.

   NOTE:  This function exists because of the unfortunate inconsistency in our
   code base.  Some packages/files use strings to represent xcvrType and some
   use the Xcvr::XcvrType enum values.  Those packages/files that use strings
   often use 'qsfp'/'sfp'.  The Xcvr::XcvrType enum uses 'qsfpPlus'/'sfpPlus'.
   Because of these differences, we need this function too.

   Parameters
   ----------
   str : a string representing transceiver type

   Returns
   -------
   bool
      Returns true if the xcvr type string represents a CMIS transceiver type,
   otherwise returns false.
   """
   return xcvrTypeStr in [ "osfp", "qsfpDd", "qsfpCmis", "dsfp", "sfpDd" ]

def isCmisCoherentMediaType( mediaType ):
   """
   Parameters
   ----------
   mediaType : a string representing Xcvr media type

   Returns true if transceiver complies with C-CMIS and CMIS4.0 specifications
   """
   return mediaType in [ "400GBASE-ZR", "400GBASE-ZRP", "800GBASE-ZR" ]

def validateDomThresholdOverrideData( headers, domThresholds ):
   """
   Validates the DOM threshold override data retrieved from a CSV file.

   Parameters:
   - headers (list[str]): Contains ordered list of headers from CSV file.
   - domThresholds (list[dict]): Contains a list of DOM threshold dicts. Each
                                 entry corresponds to a row of the CSV file.
   Returns:
   - list[namedtuple]: A list of ErrInfo namedtuples. Contains error messages, row,
                       and column indices if validation fails; If validation is
                       succesful, returns an empty list.

   """
   ErrInfo = namedtuple( 'ErrInfo', ( 'errMsg row col' ) )
   errors = []
   validHeaders = {
                     "vendorName", "vendorPn", "vendorSn",
                     "txPowerHighWarn", "txPowerLowWarn", "txPowerHighAlarm",
                     "txPowerLowAlarm", "rxPowerHighWarn", "rxPowerLowWarn",
                     "rxPowerHighAlarm", "rxPowerLowAlarm", "tempHighWarn",
                     "tempLowWarn", "tempHighAlarm", "tempLowAlarm",
                     "voltageHighWarn", "voltageLowWarn", "voltageHighAlarm",
                     "voltageLowAlarm", "txBiasHighWarn", "txBiasLowWarn",
                     "txBiasHighAlarm", "txBiasLowAlarm"
                     }
   compulsoryHeaders = [ "vendorName", "vendorPn", "vendorSn" ]

   # Check that the headers: vendorName, vendorPn, vendorSn are all present.
   if not all( keyHeader in headers for keyHeader in compulsoryHeaders ):
      errors.append( ErrInfo( "The headers vendorName, vendorPn, vendorSn are"
                              " required", 1, None ) )
      return errors

   # Check that the given headers are all within the set of valid headers.
   for i, header in enumerate( headers ):
      if header not in validHeaders:
         errors.append( ErrInfo( f"Header {header} is invalid", 1, i + 1 ) )
   if errors:
      return errors

   # Check that there are no duplicate headers.
   seenHeaders = set()
   for i, currHeader in enumerate( headers ):
      if currHeader in seenHeaders:
         errors.append( ErrInfo( f"Duplicate header {currHeader}", 1, i + 1 ) )
      seenHeaders.add( currHeader )

   if errors:
      return errors

   # Check that at least one threshold header is specified.
   if len( headers ) == len( compulsoryHeaders ):
      errors.append( ErrInfo( "At least one threshold header must be specified",
                              1, None ) )
      return errors

   def isOpt( x ):
      return x == "" or x is None

   for i, override in enumerate( domThresholds ):
      # Rows are displayed using 1-indexing and the first row is the header row
      # so we start incrementing from row 2.
      rowNum = i + 2

      # Check that at least one value of vendorName and vendorPn is non-empty.
      if isOpt( override[ "vendorName" ] ) and isOpt( override[ "vendorPn" ] ):
         errors.append( ErrInfo( "At least one value of vendorName and vendorPn"
                                 " must be non-empty", rowNum, None ) )

      for keyHeader in compulsoryHeaders:
         if override[ keyHeader ] == None: # pylint: disable=singleton-comparison
            # This occurs when rows/cols mismatch. Unfilled entries are
            # interpreted as None.
            continue
         colNum = headers.index( keyHeader ) + 1
         maxColCharWidth = 16
         # Check that no compulsory header value exceeds the maximum char length.
         if len( override[ keyHeader ] ) > maxColCharWidth:
            errors.append( ErrInfo( f"Values for {keyHeader} must not exceed 16"
                                       " characters in length", rowNum, colNum ) )
         # Check that all compulsory header values consist of ascii strings.
         if not override[ keyHeader ].isascii():
            errors.append( ErrInfo( "Vendor values must be ascii strings",
                                    rowNum, colNum ) )

      # Check that all threshold values are either an integer, a double, or the
      # empty string.
      for key, value in override.items():
         # We only want to check threshold headers with a value.
         if key not in compulsoryHeaders and key in headers:
            if value in ( None, "" ):
               continue

            try:
               _ = float( value )
            except ValueError:
               colNum = headers.index( key ) + 1
               errors.append( ErrInfo( "Threshold values must be an integer, "
                                       "double, or the empty string",
                                       rowNum, colNum ) )

   if errors:
      return errors

   # Check that there are no duplicate (vendorName, vendorPn, vendorSn) entries.
   keys = set()
   for i, override in enumerate( domThresholds ):
      rowNum = i + 2
      vendorName = "{:<16}".format( override[ "vendorName" ] )
      vendorPn = "{:<16}".format( override[ "vendorPn" ] )
      vendorSn = "{:<16}".format( override[ "vendorSn" ] )
      newKey = ( vendorName, vendorPn, vendorSn )
      if newKey in keys:
         errors.append( ErrInfo( f"vendorName:{vendorName}, vendorPn:{vendorPn}, "
                                 f"vendorSn:{vendorSn} already exists",
                                    rowNum, None ) )
      keys.add( newKey )

   return errors

def buildDomThresholdOverrideKey( domThreshold ):
   """
   Helper function which formats a key corresponding to the dom override
   threshold data.

   Parameters
   ----------
   domThreshold : dict
     Dictionary containing the data for a DOM threshold override. It should have:
     * 'vendorName' (str): Vendor name
     * 'vendorPn' (str): Vendor part number
     * 'vendorSn' (str): Vendor serial number
     * ... (Threshold values are unused within this function)
     *

   Returns
   -------
   str : unique key corresponding to the data; should be used by the caller to
         instantiate an entry in overrideDataCollection.
   """
   vendorName = domThreshold[ 'vendorName' ]
   vendorPn = domThreshold[ 'vendorPn' ]
   vendorSn = domThreshold[ 'vendorSn' ]
   # Create padded key for collection.
   return f"{vendorName:<16} {vendorPn:<16} {vendorSn:<16}"

def domThresholdsCsvParser( xgc, domThresholdDataSrc ):
   """
   Function that handles the parsing of the thresholds CSV file
   and loading of values into collection in xgc .

   Returns:
   - list[namedtuple]: A list of ErrInfo namedtuples. Contains error messages, row,
                       and column indices if parsing fails; If parsing is
                       succesful, returns an empty list.
   """
   def commit( xgc, domThresholds ):
      xgc.domThresholdOverrideData.clear()

      for dt in domThresholds:
         dtKey = buildDomThresholdOverrideKey( dt )
         # Use key to create new object in the collection
         dtData = xgc.newDomThresholdOverrideData( dtKey )
         # Load values into the object
         populateDomThresholdOverrideData( dtData, dt )

   with open( domThresholdDataSrc, 'r', encoding='utf-8' ) as csvFile:
      domThresholdReader = csv.DictReader( csvFile, delimiter=',' )
      headers = domThresholdReader.fieldnames
      domThresholds = list( domThresholdReader )

   errors = validateDomThresholdOverrideData( headers, domThresholds )
   if errors:
      return errors

   commit( xgc, domThresholds )
   # We can assume that the updates have been committed and that no errors have
   # occurred so return an empty list to indicate no errors.
   return []

def populateDomThresholdOverrideData( obj, data ):
   '''
   Helper function to populate information from "data" to "obj". Types are
   specified below. Use case is wanting to populate "obj" with data from a row
   of a CSV describing DOM threhold override values.

   Parameters
   ----------
   obj : Xcvr::DomThresholdOverride
   data : dict, typically data from a row of a CSV
   '''
   # vendorName and vendorPn are padded for accurate matching. vendorSn is used
   # in partial matches, so okay to not pad.
   obj.vendorName = "{:<16}".format( data[ "vendorName" ] )
   obj.vendorPn = "{:<16}".format( data[ "vendorPn" ] )
   obj.vendorSn = data[ "vendorSn" ]

   parameters = {
      "txPowerHighWarn", "txPowerLowWarn", "txPowerHighAlarm",
      "txPowerLowAlarm", "rxPowerHighWarn", "rxPowerLowWarn",
      "rxPowerHighAlarm", "rxPowerLowAlarm", "tempHighWarn",
      "tempLowWarn", "tempHighAlarm", "tempLowAlarm",
      "voltageHighWarn", "voltageLowWarn", "voltageHighAlarm",
      "voltageLowAlarm", "txBiasHighWarn", "txBiasLowWarn",
      "txBiasHighAlarm", "txBiasLowAlarm"
   }

   for key in parameters:
      # Verify the key exists, contains an actual value and isn't the empty string
      # pylint: disable-next=singleton-comparison
      if key in data and data[ key ] != None and data[ key ] != "":
         setattr( obj, key, float( data[ key ] ) )

def decodeDomICTxBias( u16 ):
   """
   Utility function to convert unsigned 16-bit value representing bias current to
   equivalent measurement in Amps.

   Parameters
   ----------
   s16 : int, e.g. value programmed in EEPROM for an ampere field

   Returns
   -------
   float : presentation of voltage in units of ampere
   """
   return u16 * 2e-3

def decodeDomVoltage( u16 ):
   """
   Utility function to convert unsigned 16-bit value representing temperature to
   equivalent measurement in Voltage.

   Parameters
   ----------
   s16 : int, e.g. value programmed in EEPROM for a voltage field

   Returns
   -------
   float : presentation of voltage in units of voltage
   """
   return u16 * 1e-4

def decodeDomICTemperature( s16 ):
   """
   Utility function to convert signed 16-bit value representing temperature to
   equivalent measurement in degrees Celsius.

   Parameters
   ----------
   s16 : int, e.g. value programmed in EEPROM for a temperature field

   Returns
   -------
   float : presentation of temperature in units of degrees Celsius
   """
   return XcvrApi.decodeDomICTemperature( s16 )

def decodeDomICPower( u16 ):
   """
   Utility function to convert 16-bit value representing laser power to equivalent
   dBm measurement. Equivalent functionally to Xcvr::decodeDomICPower.

   Parameters
   ----------
   u16 : int, e.g. value programmed in EEPROM for one of the laser power fields

   Returns
   -------
   float : representation of power in units of dBm
   """
   if u16 == 0:
      return float( "-inf" )
   return 10 * math.log10( u16 * 1e-7 ) + 30

def checkNumericType( val, expectedType ):
   """
   Utility function to ensure that the value provided is a legal instance of the
   type provided.

   Parameters
   ----------
   val: The value to check
   expectedType: The type that the value is expected to be an instance of.  Legal
                 type values are 'float', 'uint8', 'uint16', 'uint32', 'uint64',
                 'int8', 'int16', int32'

   Returns
   -------
   Tuple of ( result, reason) where

   result:
   True if the value conforms to the limits of the expected type
   False if it does not

   reason:
   If the result is false, a string explaining why it failed
   """
   if 'float' in expectedType:
      return ( isinstance( val, ( float, int ) ),
         "float value %s out of bounds" % val )
   elif 'uint8' in expectedType:
      return ( 0 <= val < 256, "uint8 value %s out of bounds" % val )
   elif 'uint16' in expectedType:
      return( 0 <= val < ( 1 << 16 ), "uint16 value %s out of bounds" % val )
   elif 'uint32' in expectedType:
      return( 0 <= val < ( 1 << 32 ), "uint32 value %s out of bounds" % val )
   elif 'int8' in expectedType:
      return( ( -128 ) <= val < 128, "int8 value % s out of bounds" % val )
   elif 'int16' in expectedType:
      return( ( -32768 ) <= val < 32768, "int16 value % s out of bounds" % val )
   elif 'int32' in expectedType:
      return( ( -1 << 31 ) <= val < ( 1 << 31 ),
         "int32 value % s out of bounds" % val )
   elif 'uint64' in expectedType:
      return( 0 <= val < ( 1 << 64 ),
         "uint64 value % s out of bounds" % val )
   else:
      return( 0, "unknown type %s" % expectedType )

def getSlotId( name ):
   """
   This function returns the slotId of a given xcvrName or slotName. For example, on
   a fixed system, a name might be "slot5" or "Ethernet5". They will both return "5".

   For a modular system, a slotName might be "slot5/2". This will return "5/2"

   This function works by removing the leading non-digit characters from "name"

   Parameters
   ----------
   name : str
      can be xcvrName or slotName (ex. "slot5" or "ethernet5" )

   Returns
   -------
   slotId : str | None
      returns None if the name is not formatted correctly
   """
   fixedMatch = re.match( r"\D+(\d+)$", name )
   modularMatch = re.match( r"\D+(\d+/\d+)$", name )

   slotId = None
   if fixedMatch:
      slotId = fixedMatch.group( 1 )
   elif modularMatch:
      slotId = modularMatch.group( 1 )
   return slotId

def slotHardwareControllerStatus( hardwareStatusDir, slotType: str, slotName: str ):
   """
   This function returns the controller status of the requested slot.
   Returns None if there is no controller available for the requested slot type
   Parameters
   ------------
   hardwareStatusDir: Hardware::XcvrController::AllStatusDir
      Aggregator SM containing all transceiver hardware status entities
   Returns
   -------
   """
   # XcvrType.rj45 has no corresponding controller so we return None
   slotTypeToController = {
      XcvrType.cfp2: hardwareStatusDir.cfp2ControllerStatus,
      XcvrType.osfp: hardwareStatusDir.osfpControllerStatus,
      XcvrType.qsfpDd: hardwareStatusDir.osfpControllerStatus,
      XcvrType.qsfpPlus: hardwareStatusDir.qsfpPlusControllerStatus,
      XcvrType.qsfpCmis: hardwareStatusDir.osfpControllerStatus,
      XcvrType.dsfp: hardwareStatusDir.sfpPlusControllerStatus,
      XcvrType.sfpPlus: hardwareStatusDir.sfpPlusControllerStatus,
      XcvrType.sfpDd: hardwareStatusDir.sfpDdControllerStatus,
      XcvrType.rj45: { slotName: None },
      XcvrType.unknown: { slotName: None },
   }
   return slotTypeToController[ slotType ].get( slotName )

def negInfToNone( value: Optional[ float ] ) -> Optional[ float ]:
   return value if value != float( "-inf" ) else None

def noneToNegInf( value: Optional[ float ] ) -> float:
   return value if value is not None else float( "-inf" )
