# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
#
# This file holds a bunch of module globals related to transceiver media types.

from TypeFuture import TacLazyType

XcvrMediaType = TacLazyType( 'Xcvr::MediaType' )

def convertMediaTypeCliData( cliData ):
   '''
   Parameters
   ----------
   cliData : list
      List of form [ mediaType, cliToken, helpString ]
      mediaType : Xcvr::MediaType
      cliToken : str
      helpString : str

   Returns
   -------
   Two dictionaries of the form { mediaType : cliToken } and
   { cliToken : helpString }
   '''
   mediaTypeEnumToCliDict = {}
   tokensToHelpDict = {}
   for mediaType, cliToken, helpString in cliData:
      mediaTypeEnumToCliDict[ mediaType ] = cliToken
      tokensToHelpDict[ cliToken ] = helpString
   return mediaTypeEnumToCliDict, tokensToHelpDict

sfpMediaTypeCliData = [
   ( XcvrMediaType.xcvr10GBaseAr, '10gbase-ar',
     'Override the media type to be 10GBASE-AR' ),
   ( XcvrMediaType.xcvr1000BaseLx, '1000base-lx',
     'Override the media type to be 1000BASE-LX' ),
   ( XcvrMediaType.xcvr100BaseLx10, '100base-lx10',
     'Override the media type to be 100BASE-LX10' ),
   ( XcvrMediaType.xcvr100BaseFx, '100base-fx',
     'Override the media type to be 100BASE-Fx' ),
   ( XcvrMediaType.xcvr10GBaseDwdmZt, '10gbase-dwdm',
     'Override the media type to be 10GBASE-DWDM' ),
   ( XcvrMediaType.xcvr1000BaseBx10D, '1000base-bx10-d',
     'Override the media type to be 1000BASE-BX10-D' ),
   ( XcvrMediaType.xcvr25GBaseCrN, '25gbase-crn',
     'Override the media type to be 25GBASE-CR CA-N' ),
   ( XcvrMediaType.xcvr25GBaseArN, '25gbase-arn',
     'Override the media type to be 25GBASE-AR' ),
   ( XcvrMediaType.xcvr10GBaseEr, '10gbase-er',
     'Override the media type to be 10GBASE-ER' ),
   ( XcvrMediaType.xcvr25GBaseCr, '25gbase-cr',
     'Override the media type to be 25GBASE-CR CA-L' ),
   ( XcvrMediaType.xcvr1000BaseSx, '1000base-sx',
     'Override the media type to be 1000BASE-SX' ),
   ( XcvrMediaType.xcvr10GBaseDwdmZr, '10gbase-dwdm-zr',
     'Override the media type to be 10GBASE-DWDM-ZR' ),
   ( XcvrMediaType.xcvr25GBaseCrS, '25gbase-crs',
     'Override the media type to be 25GBASE-CR CA-S' ),
   ( XcvrMediaType.xcvr10GBaseZr, '10gbase-zr',
     'Override the media type to be 10GBASE-ZR' ),
   ( XcvrMediaType.xcvr1000BaseBx10U, '1000base-bx10-u',
     'Override the media type to be 1000BASE-BX10-U' ),
   ( XcvrMediaType.xcvr10GBaseDwdmEr, '10gbase-dwdm-er',
     'Override the media type to be 10GBASE-DWDM-ER' ),
   ( XcvrMediaType.xcvr10GBaseErbdU, '10gbase-erbd-u',
     'Override the media type to be 10GBASE-ERBD-U' ),
   ( XcvrMediaType.xcvr10GBaseSrLite, '10gbase-srl',
     'Override the media type to be 10GBASE-SRL' ),
   ( XcvrMediaType.xcvr25GBaseLr, '25gbase-lr',
     'Override the media type to be 25GBASE-LR' ),
   ( XcvrMediaType.xcvr25GBaseLrE, '25gbase-lr-e',
     'Override the media type to be 25GBASE-LR (no FEC)' ),
   ( XcvrMediaType.xcvr10GBaseT, '10gbase-t',
     'Override the media type to be 10GBASE-T' ),
   ( XcvrMediaType.xcvr1000BaseT, '1000base-t',
     'Override the media type to be 1000BASE-T' ),
   ( XcvrMediaType.xcvr10GBaseErbdD, '10gbase-erbd-d',
     'Override the media type to be 10GBASE-ERBD-D' ),
   ( XcvrMediaType.xcvr10GBaseSr, '10gbase-sr',
     'Override the media type to be 10GBASE-SR' ),
   ( XcvrMediaType.xcvr10GBaseErlbdD, '10gbase-erlbd-d',
     'Override the media type to be 10GBASE-ERLBD-D' ),
   ( XcvrMediaType.xcvr10GBaseLr, '10gbase-lr',
     'Override the media type to be 10GBASE-LR' ),
   ( XcvrMediaType.xcvr25GBaseAr, '25gbase-ar',
     'Override the media type to be 25GBASE-AR' ),
   ( XcvrMediaType.xcvr10GBaseCr, '10gbase-cr',
     'Override the media type to be 10GBASE-CR' ),
   ( XcvrMediaType.xcvr25GBaseLrLite, '25gbase-lrl',
     'Override the media type to be 25GBASE-LRL' ),
   ( XcvrMediaType.xcvr10GBaseErlbdU, '10gbase-erlbd-u',
     'Override the media type to be 10GBASE-ERLBD-U' ),
   ( XcvrMediaType.xcvr25GBaseSr, '25gbase-sr',
     'Override the media type to be 25GBASE-SR' ),
   ( XcvrMediaType.xcvr25GBaseSrE, '25gbase-sr-e',
     'Override the media type to be 25GBASE-SR (no FEC)' ),
   ( XcvrMediaType.xcvr25GBaseXsr, '25gbase-xsr',
     'Override the media type to be 25GBASE-XSR' ),
   ( XcvrMediaType.xcvr25GBaseEr, '25gbase-er',
     'Override the media type to be 25GBASE-ER' ),
   ( XcvrMediaType.xcvr10GBaseLrLite, '10gbase-lrl',
     'Override the media type to be 10GBASE-LRL' ),
   ( XcvrMediaType.xcvr50GBaseCr, '50gbase-cr',
     'Override the media type to be 50GBASE-CR' ),
   ( XcvrMediaType.xcvr50GBaseSr, '50gbase-sr',
     'Override the media type to be 50GBASE-SR' ),
   ( XcvrMediaType.xcvr50GBaseLr, '50gbase-lr',
     'Override the media type to be 50GBASE-LR' ),
]

# The dictionary mediaTypeEnumToClis and mediaTypeCliToEnums are used in
# CliSave and related tests and ptests in XcvrCtrl.
sfpMediaTypeEnumToClis, sfpTokensToHelp = \
   convertMediaTypeCliData( sfpMediaTypeCliData )

qsfpMediaTypeCliData = [
   ( XcvrMediaType.xcvr40GBasePlr4, '40gbase-plr4',
     'Override the media type to be 40GBASE-PLR4' ),
   ( XcvrMediaType.xcvr50GBaseCr2N, '50gbase-cr2n',
     'Override the media type to be 50GBASE-CR2 CA-N' ),
   ( XcvrMediaType.xcvr50GBaseCr2L, '50gbase-cr2',
     'Override the media type to be 50GBASE-CR2 CA-L' ),
   ( XcvrMediaType.xcvr100GBaseLr, '100gbase-lr',
     'Override the media type to be 100GBASE-LR, CAUI-4 (no FEC)' ),
   ( XcvrMediaType.xcvr100GBaseCr4, '100gbase-cr4',
     'Override the media type to be 100GBASE-CR4 CA-L' ),
   ( XcvrMediaType.xcvr40GBaseEr4, '40gbase-er4',
     'Override the media type to be 40GBASE-ER4' ),
   ( XcvrMediaType.xcvr100GBaseXsr4, '100gbase-xsr4',
     'Override the media type to be 100GBASE-XSR4' ),
   ( XcvrMediaType.xcvr100GBaseCwdm4, '100gbase-cwdm4',
     'Override the media type to be 100GBASE-CWDM4' ),
   ( XcvrMediaType.xcvr200GBaseCr4, '200gbase-cr4',
     'Override the media type to be 200GBASE-CR4' ),
   ( XcvrMediaType.xcvr100GBaseCr2, '100gbase-cr2',
     'Override the media type to be 100GBASE-CR2' ),
   ( XcvrMediaType.xcvr100GEDwdm2, '100ge-dwdm2',
     'Override the media type to be 100GE-DWDM2' ),
   ( XcvrMediaType.xcvr100GBaseCrActive4, '100gbase-cra4',
     'Override the media type to be 100GBASE-CRA4' ),
   ( XcvrMediaType.xcvr100GBaseEr4, '100gbase-er4',
     'Override the media type to be 100GBASE-ER4' ),
   ( XcvrMediaType.xcvr40GBaseSrbd, '40gbase-srbd',
     'Override the media type to be 40GBASE-SRBD' ),
   ( XcvrMediaType.xcvr100GBaseAr4, '100gbase-ar4',
     'Override the media type to be 100GBASE-AR4 CA-L' ),
   ( XcvrMediaType.xcvr100GBaseLrl4, '100gbase-lrl4',
     'Override the media type to be 100GBASE-LRL4' ),
   ( XcvrMediaType.xcvr100GBaseAr4N, '100gbase-ar4n',
     'Override the media type to be 100GBASE-AR4 CA-N' ),
   ( XcvrMediaType.xcvr100GBasePsm4, '100gbase-psm4',
     'Override the media type to be 100GBASE-PSM4' ),
   ( XcvrMediaType.xcvr100GBasePlrl4, '100gbase-plrl4',
     'Override the media type to be 100GBASE-PLRL4' ),
   ( XcvrMediaType.xcvr100GBasePlr4, '100gbase-plr4',
     'Override the media type to be 100GBASE-PLR4' ),
   ( XcvrMediaType.xcvr40GBaseUniv, '40gbase-univ',
     'Override the media type to be 40GBASE-UNIV' ),
   ( XcvrMediaType.xcvr40GBaseCr4, '40gbase-cr4',
     'Override the media type to be 40GBASE-CR4' ),
   ( XcvrMediaType.xcvr40GBasePlrl4, '40gbase-plrl4',
     'Override the media type to be 40GBASE-PLRL4' ),
   ( XcvrMediaType.xcvr40GBaseLr4, '40gbase-lr4',
     'Override the media type to be 40GBASE-LR4' ),
   ( XcvrMediaType.xcvr40GBaseLrl4, '40gbase-lrl4',
     'Override the media type to be 40GBASE-LRL4' ),
   ( XcvrMediaType.xcvr100GBaseSr4, '100gbase-sr4',
     'Override the media type to be 100GBASE-SR4' ),
   ( XcvrMediaType.xcvr100GBaseSr4E, '100gbase-sr4-e',
     'Override the media type to be 100GBASE-SR4 (no FEC)' ),
   ( XcvrMediaType.xcvr100GBaseCr4S, '100gbase-cr4s',
     'Override the media type to be 100GBASE-CR4 CA-S' ),
   ( XcvrMediaType.xcvr100GBaseSwdm4, '100gbase-swdm4',
     'Override the media type to be 100GBASE-SWDM4' ),
   ( XcvrMediaType.xcvr200GBaseFr4, '200gbase-fr4',
     'Override the media type to be 200GBASE-FR4' ),
   ( XcvrMediaType.xcvr200GBaseAr4, '200gbase-ar4',
     'Override the media type to be 200GBASE-AR4' ),
   ( XcvrMediaType.xcvr50GBaseCr2S, '50gbase-cr2s',
     'Override the media type to be 50GBASE-CR2 CA-S' ),
   ( XcvrMediaType.xcvr40GBaseSr4, '40gbase-sr4',
     'Override the media type to be 40GBASE-SR4' ),
   ( XcvrMediaType.xcvr100GBaseErl4, '100gbase-erl4',
     'Override the media type to be 100GBASE-ERL4' ),
   ( XcvrMediaType.xcvr100GBaseFr, '100gbase-fr',
     'Override the media type to be 100GBASE-FR, CAUI-4 (no FEC)' ),
   ( XcvrMediaType.xcvr100GBaseEr, '100gbase-er',
     'Override the media type to be 100GBASE-ER, CAUI-4 (no FEC)' ),
   ( XcvrMediaType.xcvr100GBaseXcwdm4, '100gbase-xcwdm4',
     'Override the media type to be 100GBASE-XCWDM4' ),
   ( XcvrMediaType.xcvr40GBaseXsr4, '40gbase-xsr4',
     'Override the media type to be 40GBASE-XSR4' ),
   ( XcvrMediaType.xcvr100GBaseDr, '100gbase-dr',
     'Override the media type to be 100GBASE-DR' ),
   ( XcvrMediaType.xcvr100GBaseCr4N, '100gbase-cr4n',
     'Override the media type to be 100GBASE-CR4 CA-N' ),
   ( XcvrMediaType.xcvr40GBasePsm4, '40gbase-psm4',
     'Override the media type to be 40GBASE-PSM4' ),
   ( XcvrMediaType.xcvr100GBaseSrbd, '100gbase-srbd',
     'Override the media type to be 100GBASE-SRBD' ),
   ( XcvrMediaType.xcvr40GBaseAr4, '40gbase-ar4',
     'Override the media type to be 40GBASE-AR4' ),
   ( XcvrMediaType.xcvr200GBaseSr4, '200gbase-sr4',
     'Override the media type to be 200GBASE-SR4' ),
   ( XcvrMediaType.xcvr100GBaseLr4, '100gbase-lr4',
     'Override the media type to be 100GBASE-LR4' ),
   ( XcvrMediaType.xcvr100GBaseZr4, '100gbase-zr4',
     'Override the media type to be 100GBASE-ZR4' ),
   # The 100gbase-fr-fec and 100gbase-lr-fec media overrides help protect against
   # QSFP-100G-FR/LR modules misidentifying themselves as not requiring host side
   # RS-FEC when they actually do require it for operation
   ( XcvrMediaType.xcvr100GBaseFrFec, '100gbase-fr-fec',
     'Override the media type to be 100GBASE-FR, CAUI-4 (RS-FEC)' ),
   ( XcvrMediaType.xcvr100GBaseLrFec, '100gbase-lr-fec',
     'Override the media type to be 100GBASE-LR, CAUI-4 (RS-FEC)' ),
   ( XcvrMediaType.xcvr100GBaseErFec, '100gbase-er-fec',
     'Override the media type to be 100GBASE-ER, CAUI-4 (RS-FEC)' ),
   ( XcvrMediaType.xcvr200GOptical, '200g-optical',
     'Override the media type to be Optical200G' ),
   ( XcvrMediaType.xcvr100GBaseSr1_2, '100gbase-sr1.2',
     'Override the media type to be 100GBASE-SR1.2' ),
]
qsfpMediaTypeEnumToClis, qsfpTokensToHelp = \
   convertMediaTypeCliData( qsfpMediaTypeCliData )

cfp2MediaTypeCliData = [
   ( XcvrMediaType.xcvr100GBaseEr4, '100gbase-er4',
     'Override the media type to be 100GBASE-ER4' ),
   ( XcvrMediaType.xcvr100GDwdmCoherentE, '100g-dwdm-e',
     'Override the media type to be 100G-DWDM-E' ),
   ( XcvrMediaType.xcvr100GBaseSr10, '100gbase-sr10',
     'Override the media type to be 100GBASE-SR10' ),
   ( XcvrMediaType.xcvr100GBaseXsr10, '100gbase-xsr10',
     'Override the media type to be 100GBASE-XSR10' ),
   ( XcvrMediaType.xcvr100GBaseLr4, '100gbase-lr4',
     'Override the media type to be 100GBASE-LR4' ),
   ( XcvrMediaType.xcvr100GDwdmCoherent, '100g-dwdm',
     'Override the media type to be 100G-DWDM' ),
]
cfp2MediaTypeEnumToClis, cfp2TokensToHelp = \
   convertMediaTypeCliData( cfp2MediaTypeCliData )

osfpMediaTypeCliData = [
   ( XcvrMediaType.xcvr8x25GBaseLr, '8x25gbase-lr',
     'Override the media type to be 8x25GBASE-LR' ),
   ( XcvrMediaType.xcvr200GBaseCr8, '200gbase-cr8',
     'Override the media type to be 200GBASE-CR8 CA-L' ),
   ( XcvrMediaType.xcvr200GBase2Lr4, '200gbase-2lr4',
     'Override the media type to be 200GBASE-2LR4' ),
   ( XcvrMediaType.xcvr400GBaseCr8N, '400gbase-cr8n',
     'Override the media type to be 400GBASE-CR8 CA-N' ),
   ( XcvrMediaType.xcvr400GBaseCrActive8, '400gbase-cra8',
     'Override the media type to be 400GBASE-CRA8' ),
   ( XcvrMediaType.xcvr200GBaseCr8S, '200gbase-cr8s',
     'Override the media type to be 200GBASE-CR8 CA-S' ),
   ( XcvrMediaType.xcvr400GBaseSr8, '400gbase-sr8',
     'Override the media type to be 400GBASE-SR8' ),
   ( XcvrMediaType.xcvr400GOptical, '400g-optical',
     'Override the media type to be Optical400G' ),
   ( XcvrMediaType.xcvr400GBaseAr8, '400gbase-ar8',
     'Override the media type to be 400GBASE-AR8' ),
   ( XcvrMediaType.xcvr200GBaseCr8N, '200gbase-cr8n',
     'Override the media type to be 200GBASE-CR8 CA-N' ),
   ( XcvrMediaType.xcvr400GBaseSr4_2, '400gbase-sr4.2',
     'Override the media type to be 400GBASE-SR4.2' ),
   ( XcvrMediaType.xcvr400GBaseXdr4, '400gbase-xdr4',
     'Override the media type to be 400GBASE-XDR4' ),
   ( XcvrMediaType.xcvr400GBaseLr8, '400gbase-lr8',
     'Override the media type to be 400GBASE-LR8' ),
   ( XcvrMediaType.xcvr400GBase2Fr4, '400gbase-2fr4',
     'Override the media type to be 400GBASE-2FR4' ),
   ( XcvrMediaType.xcvr400GBaseCr8, '400gbase-cr8',
     'Override the media type to be 400GBASE-CR8 CA-L' ),
   ( XcvrMediaType.xcvr400GBaseDr4, '400gbase-dr4',
     'Override the media type to be 400GBASE-DR4' ),
   ( XcvrMediaType.xcvr400GBaseSr4, '400gbase-sr4',
     'Override the media type to be 400GBASE-SR4' ),
   ( XcvrMediaType.xcvr400GBaseFr4, '400gbase-fr4',
     'Override the media type to be 400GBASE-FR4' ),
   ( XcvrMediaType.xcvr400GBaseCr8S, '400gbase-cr8s',
     'Override the media type to be 400GBASE-CR8 CA-S' ),
   ( XcvrMediaType.xcvr400GBaseLr4, '400gbase-lr4',
     'Override the media type to be 400GBASE-LR4' ),
   ( XcvrMediaType.xcvr400GBasePlr4, '400gbase-plr4',
     'Override the media type to be 400GBASE-PLR4' ),
   ( XcvrMediaType.xcvr400GBaseZr, '400gbase-zr',
     'Override the media type to be 400GBASE-ZR' ),
   ( XcvrMediaType.xcvr400GBaseZrP, '400gbase-zrp',
     'Override the media type to be 400GBASE-ZRP' ),
   ( XcvrMediaType.xcvr800GBaseCr8, '800gbase-cr8',
     'Override the media type to be 800GBASE-CR8 CA-L' ),
   ( XcvrMediaType.xcvr800GBaseCr8N, '800gbase-cr8n',
     'Override the media type to be 800GBASE-CR8 CA-N' ),
   ( XcvrMediaType.xcvr800GBaseCr8S, '800gbase-cr8s',
     'Override the media type to be 800GBASE-CR8 CA-S' ),
   ( XcvrMediaType.xcvr800GBase2Fr4, '800gbase-2fr4',
     'Override the media type to be 800GBASE-2FR4' ),
   ( XcvrMediaType.xcvr800GBaseDr8, '800gbase-dr8',
     'Override the media type to be 800GBASE-DR8' ),
   ( XcvrMediaType.xcvr800GBaseXdr8, '800gbase-xdr8',
     'Override the media type to be 800GBASE-XDR8' ),
   ( XcvrMediaType.xcvr800GBase2lr4, '800gbase-2lr4',
     'Override the media type to be 800GBASE-2LR4' ),
   ( XcvrMediaType.xcvr800GBase2Dr4, '800gbase-2dr4',
     'Override the media type to be 800GBASE-2DR4' ),
   ( XcvrMediaType.xcvr800GBase2Xdr4, '800gbase-2xdr4',
     'Override the media type to be 800GBASE-2XDR4' ),
   ( XcvrMediaType.xcvr800GBase2Plr4, '800gbase-2plr4',
     'Override the media type to be 800GBASE-2PLR4' ),
   ( XcvrMediaType.xcvr800GBaseSr8, '800gbase-sr8',
     'Override the media type to be 800GBASE-SR8' ),
   ( XcvrMediaType.xcvr800GBase2Sr4, '800gbase-2sr4',
     'Override the media type to be 800GBASE-2SR4' ),
   ( XcvrMediaType.xcvr800GBaseAr8, '800gbase-ar8',
     'Override the media type to be 800GBASE-AR8' ),
   ( XcvrMediaType.xcvr800GBaseCrActive8, '800gbase-cra8',
     'Override the media type to be 800GBASE-CRA8' ),
   ( XcvrMediaType.xcvr800GOptical, '800g-optical',
     'Override the media type to be Optical800G' ),
]
osfpMediaTypeEnumToClis, osfpTokensToHelp = \
   convertMediaTypeCliData( osfpMediaTypeCliData )

qsfpDdMediaTypeCliData = [
   ( XcvrMediaType.xcvr8x25GBaseSr, '8x25gbase-sr',
     'Override the media type to be 8x25GBASE-SR' ),
   ( XcvrMediaType.xcvr200GBaseAr8, '200gbase-ar8',
     'Override the media type to be 200GBASE-AR8' ),
   ( XcvrMediaType.xcvr200GBaseCr8, '200gbase-cr8',
     'Override the media type to be 200GBASE-CR8 CA-L' ),
   ( XcvrMediaType.xcvr200GBase2Lr4, '200gbase-2lr4',
     'Override the media type to be 200GBASE-2LR4' ),
   ( XcvrMediaType.xcvr400GBaseCr8N, '400gbase-cr8n',
     'Override the media type to be 400GBASE-CR8 CA-N' ),
   ( XcvrMediaType.xcvr400GBaseCrActive8, '400gbase-cra8',
     'Override the media type to be 400GBASE-CRA8' ),
   ( XcvrMediaType.xcvr200GBaseCr8S, '200gbase-cr8s',
     'Override the media type to be 200GBASE-CR8 CA-S' ),
   ( XcvrMediaType.xcvr400GBaseSr8, '400gbase-sr8',
     'Override the media type to be 400GBASE-SR8' ),
   ( XcvrMediaType.xcvr400GOptical, '400g-optical',
     'Override the media type to be Optical400G' ),
   ( XcvrMediaType.xcvr400GBaseAr8, '400gbase-ar8',
     'Override the media type to be 400GBASE-AR8' ),
   ( XcvrMediaType.xcvr200GBaseCr8N, '200gbase-cr8n',
     'Override the media type to be 200GBASE-CR8 CA-N' ),
   ( XcvrMediaType.xcvr400GBaseXdr4, '400gbase-xdr4',
     'Override the media type to be 400GBASE-XDR4' ),
   ( XcvrMediaType.xcvr400GBaseLr8, '400gbase-lr8',
     'Override the media type to be 400GBASE-LR8' ),
   ( XcvrMediaType.xcvr400GBase2Fr4, '400gbase-2fr4',
     'Override the media type to be 400GBASE-2FR4' ),
   ( XcvrMediaType.xcvr400GBaseCr8, '400gbase-cr8',
     'Override the media type to be 400GBASE-CR8 CA-L' ),
   ( XcvrMediaType.xcvr400GBaseDr4, '400gbase-dr4',
     'Override the media type to be 400GBASE-DR4' ),
   ( XcvrMediaType.xcvr400GBaseSr4, '400gbase-sr4',
     'Override the media type to be 400GBASE-SR4' ),
   ( XcvrMediaType.xcvr400GBaseFr4, '400gbase-fr4',
     'Override the media type to be 400GBASE-FR4' ),
   ( XcvrMediaType.xcvr400GBaseCr8S, '400gbase-cr8s',
     'Override the media type to be 400GBASE-CR8 CA-S' ),
   ( XcvrMediaType.xcvr400GBaseLr4, '400gbase-lr4',
     'Override the media type to be 400GBASE-LR4' ),
   ( XcvrMediaType.xcvr400GBasePlr4, '400gbase-plr4',
     'Override the media type to be 400GBASE-PLR4' ),
   ( XcvrMediaType.xcvr400GBaseZr, '400gbase-zr',
     'Override the media type to be 400GBASE-ZR' ),
   ( XcvrMediaType.xcvr400GBaseZrP, '400gbase-zrp',
     'Override the media type to be 400GBASE-ZRP' ),
   ( XcvrMediaType.xcvr400GBaseSr4_2, '400gbase-sr4.2',
     'Override the media type to be 400GBASE-SR4.2' ),
   ( XcvrMediaType.xcvr800GBaseCr8, '800gbase-cr8',
     'Override the media type to be 800GBASE-CR8 CA-L' ),
   ( XcvrMediaType.xcvr800GBaseCr8N, '800gbase-cr8n',
     'Override the media type to be 800GBASE-CR8 CA-N' ),
   ( XcvrMediaType.xcvr800GBaseCr8S, '800gbase-cr8s',
     'Override the media type to be 800GBASE-CR8 CA-S' ),
   ( XcvrMediaType.xcvr800GBase2Fr4, '800gbase-2fr4',
     'Override the media type to be 800GBASE-2FR4' ),
   ( XcvrMediaType.xcvr800GBaseDr8, '800gbase-dr8',
     'Override the media type to be 800GBASE-DR8' ),
   ( XcvrMediaType.xcvr800GBaseXdr8, '800gbase-xdr8',
     'Override the media type to be 800GBASE-XDR8' ),
   ( XcvrMediaType.xcvr800GBase2lr4, '800gbase-2lr4',
     'Override the media type to be 800GBASE-2LR4' ),
   ( XcvrMediaType.xcvr800GBasePlr8, '800gbase-plr8',
     'Override the media type to be 800GBASE-PLR8' ),
   ( XcvrMediaType.xcvr800GBaseSr8, '800gbase-sr8',
     'Override the media type to be 800GBASE-SR8' ),
   ( XcvrMediaType.xcvr800GBase2Sr4, '800gbase-2sr4',
     'Override the media type to be 800GBASE-2SR4' ),
   ( XcvrMediaType.xcvr800GBaseAr8, '800gbase-ar8',
     'Override the media type to be 800GBASE-AR8' ),
   ( XcvrMediaType.xcvr800GBase2Dr4, '800gbase-2dr4',
     'Override the media type to be 800GBASE-2DR4' ),
   ( XcvrMediaType.xcvr800GBase2Xdr4, '800gbase-2xdr4',
     'Override the media type to be 800GBASE-2XDR4' ),
   ( XcvrMediaType.xcvr800GBase2Plr4, '800gbase-2plr4',
     'Override the media type to be 800GBASE-2PLR4' ),
   ( XcvrMediaType.xcvr800GBaseCrActive8, '800gbase-cra8',
     'Override the media type to be 800GBASE-CRA8' ),
   ( XcvrMediaType.xcvr800GBaseZr, '800gbase-zr',
     'Override the media type to be 800GBASE-ZR' ),
   ( XcvrMediaType.xcvr800GOptical, '800g-optical',
     'Override the media type to be Optical800G' ),
   ( XcvrMediaType.xcvr200GOptical, '200g-optical',
     'Override the media type to be Optical200G' ),
]
qsfpDdMediaTypeEnumToClis, qsfpDdTokensToHelp = \
   convertMediaTypeCliData( qsfpDdMediaTypeCliData )

qsfpCmisMediaTypeCliData = [
   ( XcvrMediaType.xcvr200GBaseFr4, '200gbase-fr4',
     'Override the media type to be 200GBASE-FR4' ),
   ( XcvrMediaType.xcvr200GBaseAr4, '200gbase-ar4',
     'Override the media type to be 200GBASE-AR4' ),
   ( XcvrMediaType.xcvr200GBaseSr4, '200gbase-sr4',
     'Override the media type to be 200GBASE-SR4' ),
   ( XcvrMediaType.xcvr400GBaseCrActive4, '400gbase-cra4',
     'Override the media type to be 400GBASE-CRA4' ),
   ( XcvrMediaType.xcvr400GBaseCr4, '400gbase-cr4',
     'Override the media type to be 400GBASE-CR4 CA-L' ),
   ( XcvrMediaType.xcvr400GBaseCr4N, '400gbase-cr4n',
     'Override the media type to be 400GBASE-CR4 CA-N' ),
   ( XcvrMediaType.xcvr400GBaseCr4S, '400gbase-cr4s',
     'Override the media type to be 400GBASE-CR4 CA-S' ),
   ( XcvrMediaType.xcvr400GBaseAr4, '400gbase-ar4',
     'Override the media type to be 400GBASE-AR4' ),
]

qsfpCmisMediaTypeEnumToClis, qsfpCmisTokensToHelp = \
   convertMediaTypeCliData( qsfpCmisMediaTypeCliData )

dsfpMediaTypeCliData = [
   ( XcvrMediaType.xcvr100GBaseCr2, '100gbase-cr2',
     'Override the media type to be 100GBASE-CR2 CA-L' ),
   ( XcvrMediaType.xcvr100GBaseCr2N, '100gbase-cr2n',
     'Override the media type to be 100GBASE-CR2 CA-N' ),
   ( XcvrMediaType.xcvr100GBaseCr2S, '100gbase-cr2s',
     'Override the media type to be 100GBASE-CR2 CA-S' ),
   ( XcvrMediaType.xcvr100GBaseSr2, '100gbase-sr2',
     'Override the media type to be 100GBASE-SR2' ),
   ( XcvrMediaType.xcvr100GBaseAr2, '100gbase-ar2',
     'Override the media type to be 100GBASE-AR2' ),
   ( XcvrMediaType.xcvr100GOptical, '100g-optical',
     'Override the media type to be Optical100G' ),
]

dsfpMediaTypeEnumToClis, dsfpTokensToHelp = \
   convertMediaTypeCliData( dsfpMediaTypeCliData )

sfpDdMediaTypeCliData = [
   ( XcvrMediaType.xcvr100GBaseSr2, '100gbase-sr2',
     'Override the media type to be 100GBASE-SR2' ),
]

sfpDdMediaTypeEnumToClis, sfpDdTokensToHelp = \
   convertMediaTypeCliData( sfpDdMediaTypeCliData )

mediaTypeEnumToClis = sfpMediaTypeEnumToClis.copy()
mediaTypeEnumToClis.update( qsfpMediaTypeEnumToClis )
mediaTypeEnumToClis.update( cfp2MediaTypeEnumToClis )
mediaTypeEnumToClis.update( osfpMediaTypeEnumToClis )
mediaTypeEnumToClis.update( qsfpDdMediaTypeEnumToClis )
mediaTypeEnumToClis.update( qsfpCmisMediaTypeEnumToClis )
mediaTypeEnumToClis.update( dsfpMediaTypeEnumToClis )
mediaTypeEnumToClis.update( sfpDdMediaTypeEnumToClis )
mediaTypeCliToEnums = { v: k for k, v in mediaTypeEnumToClis.items() }

# Used in the CLI - maps the media type token to the help for that token.
tokensToHelp = sfpTokensToHelp.copy()
tokensToHelp.update( qsfpTokensToHelp )
tokensToHelp.update( cfp2TokensToHelp )
tokensToHelp.update( osfpTokensToHelp )
tokensToHelp.update( qsfpDdTokensToHelp )
tokensToHelp.update( qsfpCmisTokensToHelp )
tokensToHelp.update( dsfpTokensToHelp )
tokensToHelp.update( sfpDdTokensToHelp )
nonOverridableMediaTypeEnums = {
   XcvrMediaType.xcvr10GBaseLrm, XcvrMediaType.xcvr10GBaseMfg,
   XcvrMediaType.xcvr10GBaseCrActive, XcvrMediaType.xcvrUnknown,
   XcvrMediaType.xcvr25GUnknown, XcvrMediaType.xcvr100GUnknown,
   XcvrMediaType.xcvr400GCopperUnknown, XcvrMediaType.xcvr400GOpticalUnknown,
   XcvrMediaType.xcvr40GBaseMfg, XcvrMediaType.xcvr10GBaseT,
   XcvrMediaType.xcvr5GBaseT, XcvrMediaType.xcvr2p5GBaseT,
   XcvrMediaType.xcvr40GBaseSr4d, XcvrMediaType.xcvr100GBaseClr4,
   XcvrMediaType.xcvr100GDwdmDco, XcvrMediaType.xcvr400GDwdmDco,
   XcvrMediaType.xcvr100GBaseMfg, XcvrMediaType.xcvr2x100GBaseMfg,
   XcvrMediaType.xcvrAmpZr, XcvrMediaType.xcvr800GCopperUnknown,
   XcvrMediaType.xcvr800GOpticalUnknown, XcvrMediaType.xcvr200GCopperUnknown,
   XcvrMediaType.xcvr200GOpticalUnknown, XcvrMediaType.xcvr200GBypass,
   XcvrMediaType.xcvr100GOpticalUnknown, XcvrMediaType.xcvr100GCopperUnknown,
   XcvrMediaType.xcvr25GBaseMfg,
   }

# existedMediaTypeEnums contains both overridable and non-overridable mediaType
existedMediaTypeEnums = nonOverridableMediaTypeEnums.copy()
existedMediaTypeEnums.update( mediaTypeEnumToClis )
