#!/usr/bin/env python3
# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import enum

from TypeFuture import TacLazyType

# ------------------------------------------------------------------------------
#
# Helper functions for TestPattern Cli commands
#
# ------------------------------------------------------------------------------

# host, media, generator and checker kws are prepended with 'cli'
# to distinguish between them and the tac names
prbsCliHostKw = 'system'
prbsCliMediaKw = 'line'
prbsCliGeneratorKw = 'transmitter'
prbsCliCheckerKw = 'receiver'
fecEnableKw = 'fec'
testKw = 'test'
patternKw = 'pattern'
configuredKw = 'configured'
operationalKw = 'operational'
noneKw = "none"

SideEnum = TacLazyType( "Xcvr::TestPatternSide" )
PatternTypeEnum = TacLazyType( "Xcvr::TestPatternType" )
ClockSourceEnum = TacLazyType( "Hardware::Phy::TestPatternClockSource" )

# These are the patterns that we will not create software to support
# at the moment, since they require extra work
aristaUnsupportedPatterns = { PatternTypeEnum.patternUser,
                              PatternTypeEnum.patternCustom }
aristaSupportedPatterns = ( set( PatternTypeEnum.attributes ) -
                            aristaUnsupportedPatterns )

# This function translates between the cli representation of names
# to the tac representation of names. In other words:
# ( system | line ) ( transmitter | receiver ) ->
# ( host | media ) ( generator | checker )
def prbsCliSideToTacSide( hostOrMedia, genOrCheck ):
   prbsSideDict = {
      prbsCliHostKw: { prbsCliGeneratorKw: SideEnum.hostGenerator,
                       prbsCliCheckerKw: SideEnum.hostChecker },
      prbsCliMediaKw: { prbsCliGeneratorKw: SideEnum.mediaGenerator,
                        prbsCliCheckerKw: SideEnum.mediaChecker }
                   }
   return prbsSideDict[ hostOrMedia ][ genOrCheck ]

# List of side/mode names as they are represented in the cli, as opposed to the
# internal representation from Xcvr::TestPatternSide
class CliSideEnum( enum.Enum ):
   __order__ = 'lineTransmitter lineReceiver systemTransmitter systemReceiver'
   lineTransmitter = f"{prbsCliMediaKw} {prbsCliGeneratorKw}"
   lineReceiver = f"{prbsCliMediaKw} {prbsCliCheckerKw}"
   systemTransmitter = f"{prbsCliHostKw} {prbsCliGeneratorKw}"
   systemReceiver = f"{prbsCliHostKw} {prbsCliCheckerKw}"

def prbsTacSideToCliSide( tacSideName ):
   prbsSideDict = {
      SideEnum.hostGenerator: CliSideEnum.systemTransmitter,
      SideEnum.hostChecker: CliSideEnum.systemReceiver,
      SideEnum.mediaGenerator: CliSideEnum.lineTransmitter,
      SideEnum.mediaChecker: CliSideEnum.lineReceiver }
   return prbsSideDict[ tacSideName ]
