# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from typing import Optional
from TypeFuture import TacLazyType

MediaType = TacLazyType( "Xcvr::MediaType" )
XcvrTypesApi = TacLazyType( "XcvrTypes::XcvrTypesApi" )

def mediaTypeString( mediaTypeEnum: MediaType,
                     wavelength: Optional[ int ] = None ) -> str:
   """
   Returns the string-representation of a given media-type enum.

   There is some special handling for the 10G-DWDM-ER and 10G-DWDM-ZR media-types.
   The strings for those media-types rely on the laser wavelength parameter from
   the EEPROM. We are going to assume that if the user is looking for the string
   representation for either of those two types that they will also specify the
   wavelength. Conversely, if no wavelength is specified, we will assert that the
   requested string is not for either of those two media-types.

   Parameters
   ----------
   mediaTypeEnum : Xcvr::MediaType for which the function returns the string form of

   wavelength : wavelength to be used to generate the string for 10G-DWDM-[EZ]R
   """
   mediaTypeRequiresWavelength = mediaTypeEnum in ( MediaType.xcvr10GBaseDwdmEr,
                                                    MediaType.xcvr10GBaseDwdmZr )
   if wavelength is not None:
      assert mediaTypeRequiresWavelength
   else:
      assert not mediaTypeRequiresWavelength
      wavelength = 0
   return XcvrTypesApi.mediaTypeStringHelper( mediaTypeEnum, wavelength )
