#!/usr/bin/env python3
# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# This is not actually a SuperServer Plugin, but we define a subClass of 
# LinuxService here. This class serves as a base class for all the services 
# based on xinetd. Currently, ssh and telnet are provided by xinetd.

import os
import QuickTrace
import SuperServer
import Tracing
import Tac

t0 = Tracing.trace0
t1 = Tracing.trace1
__defaultTraceHandle__ = Tracing.Handle( "XinetdLib" )
qt0 = QuickTrace.trace0

def includeDir():
   return '/etc/xinetd.d'

def confFile():
   return '/etc/xinetd.conf'

class XinetdService( SuperServer.SystemdService ): # pylint: disable-msg=W0223
   """ Base class for services based on xinetd """

   notifierTypeName = "*"
   started_ = False
   # 3 second delay before we reload xinetd, which is when the configuration
   # would take effect. 3 seconds seemed to be a reasonable choice to bunch up
   # few events under circumstanes like agent restart or startup config or copy 
   # to running config. xinetd seem to take this much time to process about 1500
   # entries under /etc/xinetd.d directory (which equates to about 500 VRF's
   # configured.
   reloadTimeout_ = 3
   xinetdConfWritten_ = False
   serviceReloadTimer_ = None

   def __init__( self, serviceName, config, serviceFile ):
      t0( 'XinetdService::__init__', serviceName )
      lsn = 'xinetd'

      SuperServer.SystemdService.__init__( self, serviceName, lsn, config, 
                                           serviceFile )

      if XinetdService.serviceReloadTimer_ is None:
         XinetdService.serviceReloadTimer_ = Tac.ClockNotifiee(
            handler=XinetdService._reloadXinetdHandler,
            timeMin=Tac.endOfTime )

      # Write the xinetd configuration at the beginning of time
      t1( 'xinetdConfWritten_ is', XinetdService.xinetdConfWritten_ )
      if not XinetdService.xinetdConfWritten_:
         if not self.writeConfigFile( confFile(), self.xinetdConf(), 
                                       saveBackupConfig=False ):
            self.sync()
         else:
            XinetdService.xinetdConfWritten_ = True

      # NOTE - We only ever start xinetd. We never stop it. This is so
      # that multiple SuperServer plugins for different xinetd services
      # can happily coexist. This means that, once xinetd is started,
      # it is never stopped. Since it doesn't consume many resources,
      # especially with all services disabled, this really isn't a problem.

   def startService( self ):
      # Override the default implementation of startService, which
      # calls 'service xinetd start'. We want to make sure we are reloading
      # xinetd.  Normally xinetd is not stopped, it needs to reload the
      # configuration file in order to start/stop the telnet service
      if not XinetdService.started_:
         self._runServiceCmd( "start" ) # In case xinetd isn't already running
         XinetdService.started_ = True
      self.starts += 1

   @classmethod
   def _reloadXinetdHandler( cls ):
      t0( '_reloadXinetdHandler' )
      cls.serviceReloadTimer_.timeMin = Tac.endOfTime

      if ( not cls.started_ and
           not SuperServer.LinuxService.runServiceCommand(
              # pylint: disable-next=protected-access
              SuperServer.SystemdService._serviceCmd ( "xinetd", "start" ) ) ):
         cls.scheduleReloadService()
         return

      cls.started_ = True
      # Though we may have issued a start, xinetd may already have been
      # running (say SuperServer alone crashed) and the start maybe a no-op
      # so issue a reload too
      if not SuperServer.LinuxService.runServiceCommand(
            # pylint: disable-next=protected-access
            SuperServer.SystemdService._serviceCmd ( "xinetd", "reload" ) ):
         cls.scheduleReloadService()

   @classmethod
   def scheduleReloadService( cls ):
      if cls.serviceReloadTimer_.timeMin == Tac.endOfTime:
         t0( 'Starting reload timer' )
         cls.serviceReloadTimer_.timeMin = Tac.now() + cls.reloadTimeout_
      else:
         t1( 'Skipping issuing reload' )

   def restartService( self ):
      XinetdService.scheduleReloadService()

   @classmethod
   def _serviceProcessWarm( cls ):
      # If we have a reload pending, return false. Else return true.
      # This way an xinetd service will report warm only after xinetd
      # has gotten a chance to process the latest config update.
      return cls.serviceReloadTimer_.timeMin == Tac.endOfTime

   def writeConfigFile( self, configFilename, config, saveBackupConfig=False,
                        writeHeader=None, updateInPlace=False ):
      # pylint: disable-next=consider-using-f-string
      t0( 'XinetdService::writeConfigFile with %s, sbc is %s and uip is %s' %
          ( configFilename, saveBackupConfig, updateInPlace ) )

      assert not saveBackupConfig
      # Xinetd will require reloading the config file if the config file
      # has changed
      XinetdService.scheduleReloadService()

      # If new config is empty, delete existing conf file
      if not config:
         try:
            os.unlink( configFilename )
            # pylint: disable-next=consider-using-f-string
            qt0( "Empty config, deleted existing file %s" % configFilename )
         except OSError:
            pass
         return True
      # Since xinetd gets its configuration from all files in /etc/xinetd.d/,
      # regardless of name, we cannot write backup files as xinetd will interpret
      # them as new services!
      return SuperServer.LinuxService.writeConfigFile( self, configFilename, config,
                                                       saveBackupConfig=False,
                                                       updateInPlace=updateInPlace )

   def stopService( self ):
      # Override the default implementation of stopService, which
      # calls 'service xinetd stop'. We want to first kill all telnet
      # sessions started by xinetd, but don't need to kill xinetd itself.
      # xinetd is a pretty trivial program, especially with all servers
      # disabled, so it's really not a big deal to leave it running. This
      # is also necessary at the point that we use some of the other
      # services provided by xinetd.
      XinetdService.scheduleReloadService()
      # NOTE - the above command is safe (ie returns with no error code)
      # even when xinetd isn't running.

      self.stops += 1

   def xinetdConf( self ):
      # Just use the default xinetd configuration
      t0( 'xinetd conf ' )
      conf =  """
# This is the master xinetd configuration file. Settings in the
# default section will be inherited by all service configurations
# unless explicitly overridden in the service configuration. See
# xinetd.conf in the man pages for a more detailed explanation of
# these attributes.

defaults
{
# The next two items are intended to be a quick access place to
# temporarily enable or disable services.
#
#       enabled         =
#       disabled        =

# Define general logging characteristics.
        log_type        = SYSLOG daemon info 
        log_on_failure  = HOST
        log_on_success  = PID HOST DURATION EXIT

# Define access restriction defaults
#
#       no_access       =
#       only_from       =
#       max_load        = 0
        cps             = 50 10
        instances       = 50
        per_source      = 10

# Address and networking defaults
#
#       bind            =
#       mdns            = yes
        v6only          = no

# setup environmental attributes
#
#       passenv         =
        groups          = yes
        umask           = 002

# Generally, banners are not used. This sets up their global defaults
#
#       banner          =
#       banner_fail     =
#       banner_success  =
}

includedir %s
"""
      return conf % includeDir()
