# Copyright (c) 2023, Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import annotations

from typing import (
   Generic,
   NamedTuple,
   TypeVar,
)

from ArPyUtils.Types import ArException

class Version( NamedTuple ):
   major: int
   minor: int

   def compatible( self, other: Version ) -> bool:
      '''Checks if the other version is compatible with this version.

      Compatibility is when both versions have matching major number and where this
      version has a minor number greater than or equal to the other version.
      '''
      return other.major == self.major and other.minor <= self.minor

Item = TypeVar( 'Item' )

class Registry( Generic[ Item ] ):
   '''A generic registry used to register items against a particular version.

   The registry can then be used to retrieve items that are compatible with certain
   generations.
   '''

   def __init__( self ):
      self.registry: dict[ Version, Item ] = {}

   def register( self, version: Version, item: Item ):
      '''Register an item with a particular version.'''
      if version in self.registry:
         raise ArException(
            f'Registering two items with the same version {version}',
            newItem=item,
            oldItem=self.registry[ version ] )

      self.registry[ version ] = item

   def find( self, version: Version ) -> Item | None:
      '''Retrieves the highest compatible item from the registry.'''
      compatibleVersions = ( itemVersion for itemVersion in self.registry
                             if itemVersion.compatible( version ) )
      maxCompatibleVersion = max( compatibleVersions, default=None )
      if not maxCompatibleVersion:
         return None

      return self.registry[ maxCompatibleVersion ]
