# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import weakref

from ArPyUtils.Types import ArException
import BothTrace
import Tac
import Tracing
from TypeFuture import TacLazyType
from ZeroTouchL1Lib.InfluenceGroup.Loader import (
   ZeroTouchInfluenceGroupLoader,
)
from ZeroTouchL1Lib.Sku.Loader import (
   ZeroTouchSkuLoader,
)
from ZeroTouchL1Lib.Watchers import (
   EntMibWatcher,
)

__defaultTraceHandle__ = Tracing.Handle( "ZeroTouch.L1" )
TERROR = BothTrace.trace1
TNOTE = BothTrace.trace3
TVERBOSE = BothTrace.trace7

ZTP_PORT_ROLES = frozenset( ( "Switched", "SwitchedBootstrap", "Management" ) )

AliasDesc = TacLazyType( 'ZeroTouch::L1::PortAliasDescriptor' )
EthIntfId = TacLazyType( "Arnet::EthIntfId" )
MgmtIntfId = TacLazyType( "Arnet::MgmtIntfId" )
FileSystemConstants = TacLazyType( "ZeroTouch::L1::FileSystemConstants" )

class ZeroTouchL1DefinitionManager:
   def __init__( self, entMib, influenceGroupSliceDir ):
      self.entMib = entMib
      self.influenceGroupSliceDir = influenceGroupSliceDir

      self.influenceGroupDefDir = Tac.newInstance(
            "ZeroTouch::L1::InfluenceGroupDefinitionDir", "influenceGroupDefDir" )
      self.skuDefDir = Tac.newInstance( "ZeroTouch::L1::SkuDefinitionDir",
                                        "skuDefinitionDir" )

      # We want to make sure to load in any/all card definitions before we setup the
      # reactors to handle card insertions
      self.loadL1Definitions()

      # Create a watcher to handle card insertions/deletions; will handle processing
      # any pre-existing cards as well.
      # Pass a weak backpointer to let the reactor call back when FruReady changes.
      self.watcher = EntMibWatcher( entMib, weakref.proxy( self ) )

   # TODO: See BUG838213; Add support for a path that lives on the USB drive and
   #       react to insertion to repopulate the influence groups if the file exists
   def getInfluenceGroupPaths( self ):
      paths = [ FileSystemConstants.influenceGroupDefinitionLibraryPath() ]
      if extraPath := os.getenv( "ZEROTOUCH_EXTRA_INFLUENCE_GROUP_YAML", None ):
         paths.append( extraPath )
      return paths

   # TODO: See BUG838213; Add support for a path that lives on the USB drive and
   #       react to insertion to repopulate the influence groups if the file exists
   def getSkuPaths( self ):
      paths = [ FileSystemConstants.skuDefinitionLibraryPath() ]
      if extraPath := os.getenv( "ZEROTOUCH_EXTRA_SKU_YAML", None ):
         paths.append( extraPath )
      return paths

   def loadL1Definitions( self ):
      # Load the influence group definitions first
      groupLoader = ZeroTouchInfluenceGroupLoader( self.influenceGroupDefDir )
      for path in self.getInfluenceGroupPaths():
         groupLoader.load( path )

      # Next load the sku definitions. We pass in the sku name if this is a fixed
      # system so that the loader can skip loading other definitions. The notable
      # exception is if this the fixedSystem supports card slots, since we need to
      # load those definitions as well.
      fixedModelName = getattr( self.entMib.fixedSystem, "modelName", None )
      if getattr( self.entMib.fixedSystem, "cardSlotsSupported", True ):
         fixedModelName = None
      skuLoader = ZeroTouchSkuLoader( self.skuDefDir, self.influenceGroupDefDir,
                                      skuBaseName=fixedModelName )
      for path in self.getSkuPaths():
         skuLoader.load( path )

   def getEntMibForSlice( self, sliceName ):
      # Have to special-case handling FixedSystem since its at the root level
      if sliceName == "FixedSystem" and self.entMib.fixedSystem:
         return self.entMib.fixedSystem

      # Handle any cards in removable slots
      if self.entMib.root:
         for slot in self.entMib.root.cardSlot.values():
            if not slot.card:
               continue

            cardName = slot.card.tag + slot.card.label
            if slot.card.hasUplink:
               # Uplink cards have a different name than what the tag implies
               cardName = f'Linecard{slot.card.label}'

            if sliceName == cardName:
               return slot.card

      # If we couldn't find anything then the best we can do is return nothing
      return None

   def clearL1InfluenceGroupsForCard( self, sliceName ):
      if sliceName in self.influenceGroupSliceDir.groups:
         TNOTE( "Clearing influence groups for", sliceName )
         del self.influenceGroupSliceDir.groups[ sliceName ]

   def createL1InfluenceGroupsForCard( self, sliceName ):
      card = self.getEntMibForSlice( sliceName )
      # We don't ever expect to encounter this case, but we should try our best to
      # keep the agent limping along since we can't take user input
      if not card:
         TERROR( "No entmib entry found for card", sliceName )
         self.clearL1InfluenceGroupsForCard( sliceName )
         return

      # We only want to consider cards that have interfaces that ZTP wants to use
      # This essentially means only ethernet-capable front-panel ports
      roles = ZTP_PORT_ROLES
      if 'ZEROTOUCH_IGNORE_MANAGEMENT' in os.environ:
         roles -= { "Management" }
      intfIds = [ port.intfId for port in card.port.values() if port.role in roles ]
      if not intfIds:
         TNOTE( "Not using", sliceName,
                "for ZTP as it defines no front panel switching interfaces" )
         return

      # Fetch the correct sku definition for the card
      # TODO: See BUG840070; Handle autogenerating a definition for skus without one
      skuDefinition = self.skuDefDir.getSkuDefinition( card.modelName )
      if not skuDefinition:
         TERROR( "Didn't find a matching sku definition for", sliceName, "(",
                 card.modelName, ")" )
         return
      TNOTE( "Found sku definition match", skuDefinition.name, "for", sliceName )

      # If we somehow missed the removal of a card we need to potentially clear the
      # old group data before we can create the new groups
      if prevSliceGroups := self.influenceGroupSliceDir.groups.get( sliceName ):
         if prevSliceGroups.skuDefinition.name == skuDefinition.name:
            # However, if the sku definition hasn't changed, then we've already
            # initialized the groups for this card, so don't clear the groups
            TVERBOSE( "Existing influence groups found for", sliceName,
                      "with the same sku definition (", skuDefinition.name,
                      "); skipping population." )
            return
         else:
            TNOTE( "Existing influence groups found for", sliceName,
                   "with a different sku definition (",
                   prevSliceGroups.skuDefinition.name,
                   ") than currently inserted card (", skuDefinition.name, ")" )
            self.clearL1InfluenceGroupsForCard( sliceName )

      # Management ports aren't defined in the normal way, so we craft a custom alias
      # for each management interface based on the intfId and its xcvr type.
      # Any management ports that don't have a xcvrSlot are treated as an rj45 port
      managementAlias = {}
      def getManagementAlias( intfId ):
         return ( managementAlias.get( intfId ) or
                  AliasDesc( intfId, "RJ45Port", "Intf" ) )
      for xcvrSlot in card.xcvrSlot.values():
         for port in xcvrSlot.port.values():
            intfId = port.intfId
            if MgmtIntfId.isMgmtIntfId( intfId ):
               # For now we only support SFP management xcvrs
               if xcvrSlot.slotType != "sfp":
                  raise ArException( "Encountered unexpected Management port type",
                                     intfId=intfId, xcvrType=xcvrSlot.slotType )
               managementAlias[ intfId ] = AliasDesc( intfId, "SinglePort", "Intf" )

      # Once we've gotten through all of the skip cases, we can process all of
      # interfaces on the card via the sku definition to produce the actual
      # influence groups we have on the system.
      sliceGroups = self.influenceGroupSliceDir.newGroups( sliceName )
      sliceGroups.skuDefinition = skuDefinition
      for intfId in intfIds:
         # Fetch the alias information for the interface and create its group
         if MgmtIntfId.isMgmtIntfId( intfId ):
            aliasDesc = getManagementAlias( intfId )
         elif EthIntfId.isEthIntfId( intfId ):
            aliasDesc = skuDefinition.portAlias.get( EthIntfId.port( intfId ) )
         else:
            TNOTE( "Skipping unsupported interface", intfId, "on", sliceName )
            continue
         if not aliasDesc:
            raise ArException( "Failed to find a matching port alias for interface",
                               intfId=intfId,
                               validPorts=list( skuDefinition.portAlias ) )
         group = sliceGroups.newGroup( aliasDesc.groupId )
         # We need to setup the pointer to the group definition on this specific
         # groups instance, but we have the same data provided for each interface
         # in the group. We check to confirm that this is the same for all of the
         # interfaces in the group and just rely on idempotency to set it.
         if ( group.groupDefinition and
              group.groupDefinition.name != aliasDesc.groupType ):
            raise ArException( "Inconsistent influence group type found for group",
                               groupId=aliasDesc.groupId,
                               existingGroupType=group.groupDefinition.name,
                               conflictingIntfId=intfId,
                               conflictingGroupType=aliasDesc.groupType )
         group.groupDefinition = \
               self.influenceGroupDefDir.groupDefinition[ aliasDesc.groupType ]

         # Find the right alias name for the interface and mark it as a member of
         # the group. The port alias doesn't include any lane information for the
         # interface, so add it in if the interface actually has lanes
         intfAlias = f"{aliasDesc.alias}"
         # TODO: Management ports don't support lanes yet; this will need to change
         #       if and when we support QSFP management ports
         if ( EthIntfId.isEthIntfId( intfId ) and
              ( lane := EthIntfId.lane( intfId ) ) ):
            intfAlias += f"/{lane}"
         group.memberIntfAlias[ intfId ] = intfAlias

      # The last step is to mark the entire card as initialized so that the agent
      # knows we've finished setting up all of the group data
      sliceGroups.initialized = True
