# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BothTrace
import CliCommon
import EapiClientLib
import Tac
import Tracing
from LoopbackIntfUtils import randomLoopbackName

__defaultTraceHandle__ = Tracing.Handle( "ZtnSwitchConfigSaver" )

bt5 = BothTrace.tracef5
t8 = Tracing.trace8

class SwitchConfigSaver:
   """ With L3 ZTN,
   * DMF controller generated config doesn't contain Management IP and static
     routes required by the switch for maintaining connectivity.
   * ZTN agent runs clean-config before applying config received from the
     controller. This leads to loss of Management1 intf config. So, such user
     entered configuration should be preserved by the ZTN agent.
   * At the same time, 'show running-config' should not show the Ma1 config
     while 'startup-config' should contain it to survive switch reboot. Adding
     the corresponding config mount paths as 'Config root filters' will solve the
     former problem. But the later problem should be handled by the ZTN agent.
   # NOTE: Initially, we'll be ignoring the last problem of running-config filter
   # until the filter is actually enabled.

   This class is responsible obtaining Ma1's IP config and route configuration
   from the current running-config """

   def __init__( self, modular=False ):
      self.sourceIntf = None # alternative "management" interface used in stest
      self.modular = modular
      # used only in btest to use Loopback0 as DmfLoopback instead
      # of randomizing it
      self.test = False

   def sendEapiRequest( self, cmds ):
      eapiClient = EapiClientLib.EapiClient( sysname=Tac.sysname(), disableAaa=True,
            privLevel=CliCommon.MAX_PRIV_LVL )
      cfgs = []
      try:
         output = eapiClient.runCmds( 1, cmds, 'text' )
         t8( f'{output}' )
         for result in output[ 'result' ]:
            cfgs += [ _f for _f in result[ 'output' ].split( '\n' ) if _f ]
            cfgs.append( '!' )
      except EapiClientLib.EapiException as e:
         t8( 'Failed to run commands due to error', e )
      return cfgs

   def getGwIp( self, cmds ):
      gwRoute = list( filter(
         lambda config:'ip route 0.0.0.0/0' in config, cmds ) )
      if not gwRoute:
         t8( 'Failed to get GW route' )
         return ""

      gwIp = gwRoute[ 0 ].split()[ -1 ]
      return gwIp

   def findIntfWithConfig( self, config, searchInterface="", skipInterface="" ):
      # Find interface with given config other than skipInterface
      cmd = r"show running-config section interface " + searchInterface
      t8( 'Running cmd:', cmd )
      cfgs = self.sendEapiRequest( [ cmd ] )

      for i, cfg in enumerate( cfgs ):
         if config in cfg:
            j = i
            cmd = cfgs[ j ]
            while "interface " not in cmd:
               j -= 1
               cmd = cfgs[ j ]
            if cfgs[ j ] == f"interface {skipInterface}":
               continue
            return cfgs[ j ].split()[ -1 ]
      return ""

   def getCurrentMgmtRedundantIntf( self ):
      # To find management redundant interface, we first interface
      # with `ip proxy-arp` configuration. We expect this configuration
      # to be present under just two interfaces - management redundant
      # interface and Management1. Once we find index of `ip proxy-arp`,
      # we walk back in cfgs list until we hit "interface XXXX" line.
      # If XXXX here is Management1, we continue to look further or
      # if it is not Management1, we know that we have found our
      # management redundant interface
      return self.findIntfWithConfig( "ip proxy-arp", skipInterface="Management1" )

   def getLoopbackIntfsInUse( self ):
      cmd = r"show running-config section interface Loopback"
      t8( "Running cmd:", cmd )
      cfgs = self.sendEapiRequest( [ cmd ] )

      interfaceLoopbackLines = list( filter(
         lambda config: 'interface Looopback' in config, cfgs ) )

      if not interfaceLoopbackLines:
         return []

      return [ l.split()[ -1 ] for l in interfaceLoopbackLines ]

   def getDmfLoopbackIntf( self ):
      return self.findIntfWithConfig( "description DmfLoopback",
                                      searchInterface="Loopback" )

   def getMgmtIpPrefixAndIndex( self, cfgs, startIndex=0 ):
      mgmtIpPrefix = ""
      mgmtIpPrefixIndex = 0
      for i in range( startIndex, len( cfgs ) ):
         if "ip address" in cfgs[ i ]:
            mgmtIpPrefix = cfgs[ i ].lstrip().split()[ -1 ]
            mgmtIpPrefixIndex = i
            return mgmtIpPrefix, mgmtIpPrefixIndex
      return mgmtIpPrefix, mgmtIpPrefixIndex

   def addMgmtRedundancyConfig( self, cmds, requestedMgmtRedundantIntf ):
      bt5()

      if self.test:
         loopbackIntf = "Loopback0"
      else:
         loopbackIntfsInUse = self.getLoopbackIntfsInUse()
         loopbackIntf = randomLoopbackName()
         while loopbackIntf in loopbackIntfsInUse:
            loopbackIntf = randomLoopbackName()

      t8( f"Requested management redundant interface: {requestedMgmtRedundantIntf}" )
      # Change Ma1 to available loopback interface
      runCfgCmds = [ cfg.replace( 'interface Management1',
                                  f'interface {loopbackIntf}' ) for cfg in cmds ]
      # insert custom "DmfLoopback" description for this loopback intf config
      loopbackIntfIndex = runCfgCmds.index( f'interface {loopbackIntf}' )
      mgmtIpPrefix, mgmtIpPrefixIndex = self.getMgmtIpPrefixAndIndex(
            runCfgCmds, startIndex=loopbackIntfIndex )
      mgmtIp, mgmtIpPrefixMask = mgmtIpPrefix.split( '/' )
      runCfgCmds[ mgmtIpPrefixIndex ] = f'   ip address {mgmtIp}/32'
      runCfgCmds.insert( loopbackIntfIndex + 1,
            '   description DmfLoopback '
            f'( Original Management1 Mask: {mgmtIpPrefixMask} )' )

      # Add Ma1 and redundant interface configs
      runCfgCmds.append( 'interface Management1' )
      runCfgCmds.append( '   ip routing address required disabled' )
      runCfgCmds.append( '   ip proxy-arp' )
      runCfgCmds.append( '!' )

      runCfgCmds.append( f'interface {requestedMgmtRedundantIntf}' )
      runCfgCmds.append( '   ip routing address required disabled' )
      runCfgCmds.append( '   ip proxy-arp' )
      runCfgCmds.append( '!' )

      # Get GW IP. We expect default route to be present, otherwise
      # we can't extract GW IP and mgmt redundancy may not work
      gwIp = self.getGwIp( cmds )
      if not gwIp:
         t8( 'Failed to get GW IP' )
         return ( [], 'switch gateway IP address is not configured' )

      # Add ip route configs
      for intf in [ 'Management1', requestedMgmtRedundantIntf ]:
         runCfgCmds.append( f'ip route {gwIp}/32 {intf}' )
      runCfgCmds.append( '!' )

      return ( runCfgCmds, '' )

   def removeMgmtRedundancyConfig( self, cmds,
         currentMgmtRedundantIntf, dmfLoopbackIntf ):
      bt5()

      runCfgCmds = cmds[ : ]

      def findIntfConfig( cfgs, interface ):
         # Returns indices in cfgs between which interface
         # config is present. Returns -1, -1 if it cannot
         # find the config
         for i, cfg in enumerate( runCfgCmds ):
            if cfg == f"interface {interface}":
               j = i
               cmd = runCfgCmds[ j ]
               while cmd != '!':
                  j += 1
                  cmd = runCfgCmds[ j ]
               return i, j
         return -1, -1

      # Remove Ma1 related configs
      indexStart, indexEnd = findIntfConfig( runCfgCmds, "Management1" )
      if indexStart >= 0 and indexEnd >= 0:
         del runCfgCmds[ indexStart : indexEnd + 1 ]

      # Remove currentMgmtRedundantIntf related configs
      indexStart, indexEnd = findIntfConfig( runCfgCmds, currentMgmtRedundantIntf )
      if indexStart >= 0 and indexEnd >= 0:
         del runCfgCmds[ indexStart : indexEnd + 1 ]

      gwIp = self.getGwIp( cmds )
      if not gwIp:
         t8( 'Failed to get GW IP' )
         return ( [], 'Switch gateway IP address is not configured' )

      # Remove IP routes
      for intf in [ 'Management1', currentMgmtRedundantIntf ]:
         runCfgCmds.remove( f"ip route {gwIp}/32 {intf}" )

      # Replace DmfLoopback with Management1
      runCfgCmds = [ cfg.replace( f'interface {dmfLoopbackIntf}',
                                  'interface Management1' ) for cfg in runCfgCmds ]
      # Replace the mgmt IP /32 mask with original mask
      mgmtIntfIndex = runCfgCmds.index( 'interface Management1' )
      originalMgmtMask = '32'
      for i in range( mgmtIntfIndex, len( runCfgCmds ) ):
         if "DmfLoopback" in runCfgCmds[ i ]:
            try:
               originalMgmtMask = \
                     [ s for s in runCfgCmds[ i ].split() if s.isdigit() ][ 0 ]
               break
            except IndexError:
               t8( 'Failed to get original management interface mask' )
               return ( [], 'management IP address mask is not known' )

      mgmtIpPrefix, mgmtIpPrefixIndex = self.getMgmtIpPrefixAndIndex(
            runCfgCmds, startIndex=mgmtIntfIndex )
      mgmtIp = mgmtIpPrefix.split( '/' )[ 0 ]

      runCfgCmds[ mgmtIpPrefixIndex ] = \
            f"   ip address {mgmtIp}/{originalMgmtMask}"
      # Get rid of "description DmfLoopback" config cmd
      runCfgCmds.remove( "   description DmfLoopback "
                         f"( Original Management1 Mask: {originalMgmtMask} )" )

      return ( runCfgCmds, '' )

   def maybeReplaceMgmtRedundantIntf( self, cmds, currentMgmtRedundantIntf,
         requestedMgmtRedundantIntf ):
      bt5()

      t8( f"Replacing {currentMgmtRedundantIntf} with "
          f"{requestedMgmtRedundantIntf}" )
      runCfgCmds = [ cfg.replace( currentMgmtRedundantIntf,
                                  requestedMgmtRedundantIntf ) for cfg in cmds ]
      return runCfgCmds

   def generateConfig( self, requestedMgmtRedundantIntf="" ):
      bt5()
      cmds = []

      # Management interfaces config
      # This will be Management1 on fixed platform, Management0, Management1/1,
      # Management2/1, etc. on modular platform
      cmds.append( r"show running-config section ^interface\sManagement" )
      if self.sourceIntf:
         cmds.append( rf"show running-config section ^interface\s{self.sourceIntf}" )

      # Ignore requestedMgmtRedundantIntf for modular chassis
      if self.modular:
         requestedMgmtRedundantIntf = ""

      currentMgmtRedundantIntf = ""
      dmfLoopbackIntf = self.getDmfLoopbackIntf()
      mgmtRedundancyConfigLoaded = bool( dmfLoopbackIntf )

      if mgmtRedundancyConfigLoaded:
         # TODO: save currentMgmtRedundantIntf in status
         # and try to get from there first and only make EAPI call if not
         # not set in status
         currentMgmtRedundantIntf = self.getCurrentMgmtRedundantIntf()
         assert currentMgmtRedundantIntf, \
               "Management redundancy loaded but redundant intf not found"

         cmds.append( r"show running-config interfaces " + currentMgmtRedundantIntf )
         cmds.append( r"show running-config interfaces " + dmfLoopbackIntf )

      # Static IP and IPv6 routes
      cmds.append( r"show running-config section ip\sroute" )
      cmds.append( r"show running-config section ipv6\sroute" )

      # Speed groups configuration
      cmds.append( r"show running-config section hardware speed-group" )

      # DNS config
      cmds.append( r"show running-config section ^dns" )
      cmds.append( r"show running-config section ^ip\sname-server" )

      # TerminAttr config
      cmds.append( r"show running-config section ^daemon TerminAttr" )

      # Trace settings
      cmds.append( r"show running-config section trace\s" )

      # Misc
      cmds.append( r"show running-config section alias" )
      cmds.append( r"show running-config section agent shutdown" ) # jira/BSC-14405

      t8( 'Running cmds:', cmds )
      runCfgCmds = self.sendEapiRequest( cmds )
      errMsg = ""

      if requestedMgmtRedundantIntf and mgmtRedundancyConfigLoaded:
         # it is possible that redundant intf has changed. If so, replace it
         # with requestedMgmtRedundantIntf`
         runCfgCmds = self.maybeReplaceMgmtRedundantIntf( runCfgCmds,
               currentMgmtRedundantIntf, requestedMgmtRedundantIntf )

      elif not requestedMgmtRedundantIntf and mgmtRedundancyConfigLoaded:
         # if controller has not enabled mgmt redundany config but the switch
         # has it, we need to remove mgmt redundancy config on the switch
         runCfgCmds, errMsg = self.removeMgmtRedundancyConfig( runCfgCmds,
               currentMgmtRedundantIntf, dmfLoopbackIntf )

      elif requestedMgmtRedundantIntf and not mgmtRedundancyConfigLoaded:
         # if controller has enabled mgmt redundancy config but the switch
         # does not have it, we need to add mgmt redundancy config on the switch
         runCfgCmds, errMsg = self.addMgmtRedundancyConfig( runCfgCmds,
               requestedMgmtRedundantIntf )

      else:
         # do nothing. We don't have mgmt redundancy config loaded neither
         # controller has requested it
         pass

      return ( runCfgCmds, errMsg )
