"""ManifestV2.py

"""

class Manifest:

    VERSION = 2

    def __init__(self, data):
        self.data = data
        self.checksumAlgorithm = 'md5'

    @property
    def swiUrl(self):
        return self.data.get('swi_url', None)

    @property
    def swiChecksum(self):
        return self.data.get('swi_md5', None)

    @property
    def swiKey(self):
        if not self.swiChecksum:
            return None
        return "%s:%s" % (self.checksumAlgorithm.upper(), self.swiChecksum,)

    @property
    def configUrl(self):
        return self.data.get('startup_config_url', None)

    @property
    def configChecksum(self):
        return self.data.get('startup_config_md5', None)

    @property
    def configKey(self):
        if not self.configChecksum:
            return None
        return "%s:%s" % (self.checksumAlgorithm.upper(), self.configChecksum,)

    @property
    def controllers(self):
        for ip in self.data.get('controllers_ips', []):
            yield ['https', ip, 8843,]

    def toJson(self):
        return self.data

    @classmethod
    def fromJson(cls, data):

        mver = data.get('ztn-version', None)
        if mver is not None:
            raise ValueError("invalid manifest version")

        ob = cls(data)

        if ob.swiUrl is None and ob.swiChecksum is None:
            pass
        elif ob.swiUrl is not None and ob.swiChecksum is not None:
            # maybe validate checksum format
            pass
        else:
            raise ValueError("invalid URL/checksum combination")

        if ob.configUrl is None and ob.configChecksum is None:
            pass
        elif ob.configUrl is not None and ob.configChecksum is not None:
            # maybe validate checksum format
            pass
        else:
            raise ValueError("invalid URL/checksum combination")

        ctls = list(ob.controllers)
        if not ob.swiUrl and not ob.configUrl and not ctls:
            raise ValueError("empty manifest")


        return ob

    @classmethod
    def fromChecksums(cls,
                      swiUrl=None, swiChecksum=None, configUrl=None,
                      configChecksum=None):
        """Synthesize a manifest.

        Does not include controller IPs.
        """
        mf = {}

        if swiUrl and swiChecksum:
            mf['swi_url'] = swiUrl
            mf['swi_md5'] = swiChecksum
        elif not swiUrl and not swiChecksum:
            pass
        else:
            raise ValueError("invalid swi URL specifier")

        if configUrl and configChecksum:
            mf['startup_config_url'] = configUrl
            mf['startup_config_md5'] = configChecksum
        elif not configUrl and not configChecksum:
            pass
        else:
            raise ValueError("invalid startup-config URL specifier")

        return cls.fromJson(mf)
