"""ManifestV3.py

"""

import http.client

class Manifest:
    """Parse a V3 SWL manifest.

    Here we make a short-cut and require sha256 checksums.
    """

    VERSION = 3

    def __init__(self, data):
        self.data = data
        self.checksumAlgorithm = 'sha256'

    @property
    def swiUrl(self):
        for url, _ in self.fromJsonEntries(self.data.get('images', [])):
            return url
        return None

    @property
    def swiChecksum(self):
        for _, ck in self.fromJsonEntries(self.data.get('images', [])):
            return ck
        return None

    @property
    def swiKey(self):
        ck = self.swiChecksum
        if not ck:
            return None
        return "%s:%s" % (self.checksumAlgorithm.upper(), ck,)

    @property
    def configUrl(self):
        for url, _ in self.fromJsonEntries(self.data.get('startup-config', [])):
            return url
        return None

    @property
    def configChecksum(self):
        for _, ck in self.fromJsonEntries(self.data.get('startup-config', [])):
            return ck
        return None

    @property
    def configKey(self):
        ck = self.configChecksum
        if not ck:
            return None
        return "%s:%s" % (self.checksumAlgorithm.upper(), ck,)

    @property
    def controllers(self):
        for ipe in self.data.get('controllers', []):
            ip = ipe.get('address', None)
            if ip is None:
                raise ValueError("missing controller IP")
            for se in ipe.get('services', []):
                name = se.get('name', None)
                proto = se.get('protocol', None)
                port = se.get('port', None)

                if name is None:
                    raise ValueError("invalid service name")

                if proto == 'http':
                    if port is None:
                        port = http.client.HTTP_PORT
                elif proto == 'https':
                    if port is None:
                        port = http.client.HTTPS_PORT
                else:
                    raise ValueError("invalid protocol")

                if not isinstance(port, int):
                    raise ValueError("invalid port")

                if name == 'ztn':
                    yield [proto, ip, port,]

    def toJson(self):
        return self.data

    @classmethod
    def fromJsonEntries(cls, data):

        if not isinstance(data, list):
            raise ValueError("invalid JSON contents")

        for ent in data:
            url = ent.get('url', None)
            if url is None:
                continue
            cks = ent.get('hashes', {})
            if not isinstance(cks, dict):
                raise ValueError("invalid/missing checksums in JSON contents")
            for algo, ck in cks.items():
                if algo == 'sha256':
                    yield (url, ck,)

    @classmethod
    def fromJson(cls, data):

        mver = data.get('ztn-version', None)
        if mver != 3:
            raise ValueError("invalid manifest version")

        ob = cls(data)

        cl = list(ob.controllers)
        if not (ob.swiUrl or ob.configUrl or cl):
            raise ValueError("empty manifest")

        return ob

    @classmethod
    def fromChecksums(cls,
                      swiUrl=None, swiChecksum=None,
                      configUrl=None, configChecksum=None):
        """Synthesize a manifest. Doesn't include controller IPs. """

        mf = {'ztn-version' : 3,}

        if swiUrl and swiChecksum:
            mf['images'] = [
                {'url' : swiUrl,
                 'hashes' : {'sha256' : swiChecksum,},},
                ]
        elif not (swiUrl or swiChecksum):
            pass
        else:
            raise ValueError("invalid swi URL specifier")

        if configUrl and configChecksum:
            mf['startup-config'] = [
                {'url' : configUrl,
                 'hashes' : {'sha256' : configChecksum,},},
                ]
        elif not (configUrl or configChecksum):
            pass
        else:
            raise ValueError("invalid startup-config URL specifier")

        return cls.fromJson(mf)
