#!/usr/bin/env python3
# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=import-error
import commands
from netaddr import IPAddress
from cli_parser.option_parser import OptionParser
import pybess
import sys
sys.path.append( '../test' )
UNSPECIFIED = 0xFFFFFFFF

ipProtocolNumberToName = {
      0 : 'ip',
      1 : 'icmp',
      2 : 'igmp',
      6 : 'tcp',
      17 : 'udp',
      41 : 'ipv6',
      46 : 'rsvp',
      47 : 'gre',
      50 : 'esp',
      51 : 'ah',
      58 : 'icmpv6',
      89 : 'ospf',
      103 : 'pim',
      112 : 'vrrp',
      124 : 'isis',
      132 : 'sctp',
}

targets = {
   'INTERFACE_ACL' : 0,
   'QOS' : 1,
   'CPU_CLASSIFY' : 2,
   'MIRROR' : 3,
   'NAT' : 4,
   'OPENFLOW' : 5,
   'AEGIS_L4XFORM' : 6,
   'AEGIS_POLICY' : 7,
   'CLASSIFICATION' : 8,
}

class ShowArAclRulesOptionParser( OptionParser ):

   def register_options( self ):
      self.add_option( "-target", "string" )
      self.add_option( "-id", "uint" )
      self.add_option( "-h", "action", action=self.help )

   @staticmethod
   def help():
      help_text = [
            "Usage:",
            " show aracl rules -target <ACL_TARGET, 'string'> -id <ACL_ID, 'uint'>",
            " Example: show aracl rules -target interface_acl -id 1",
            "",
            "Options for ACL_TARGET:",
            "INTERFACE_ACL : ACL_TARGET_DP",
            "QOS : ACL_TARGET_POLICY_QOS",
            "CPU_CLASSIFY : ACL_TARGET_CPU_CLASSIFY",
            "MIRROR : ACL_TARGET_MIRROR",
            "NAT : ACL_TARGET_NAT\n",
            "Note - ACL_TARGET_OPENFLOW, ACL_TARGET_AEGIS_L4XFORM,"
                  " ACL_TARGET_AEGIS_POLICY, ACL_TARGET_CLASSIFICATION"
                  " are currently unsupported.",
            ]
      print( "\n".join( help_text ) )

   def get_acl_target( self ):
      return self.get_value( "-target" )

   def get_acl_id( self ):
      return self.get_value( "-id" )

def _show_aracl_rules_parse_options( cli, opts ):
   if opts is None:
      opts = []

   args = {}
   try:
      parser = ShowArAclRulesOptionParser( opts )
      parser.parse()
      args[ 'acl_target' ] = parser.get_acl_target()
      args[ 'acl_id' ] = parser.get_acl_id()

      # empty params indicates that no value has been specified for -target
      # or -id. Note: params would not be empty in case wrong option has
      # been mentioned.
      if parser.params == []:
         args[ 'acl_target' ] = UNSPECIFIED
         args[ 'acl_id' ] = UNSPECIFIED

   except Exception as error:
      if opts == [ '-h' ]:
         return None
      elif '-id' in opts:
         raise cli.BindError( '"Invalid acl_id. Please enter a valid acl_id"' )
      raise \
         cli.BindError( '"Invalid Input (run show aracl rules -h for more help)"' )\
         from error

   return args

# Making checks for the args before passing it to .cpp file
def _show_aracl_rules_args_preproc( cli, args ):
   cmd_args = {}
   if args[ 'acl_target' ] is None and args[ 'acl_id' ] is None:
      raise cli.BindError(
         '"Invalid Input (run show aracl rules -h for more help)"' )

   # Making some checks for valid acl_target

   # args['acl_target'] could be none when only -id has been mentioned
   # in the command (Example: show aracl rules -id 3).
   # In this case args would be {'acl_target' : None, 'acl_id' : 3}
   # Thus we need to iterate over all targets to find acl_id 3
   if args[ 'acl_target' ] is None:
      cmd_args[ 'acl_target' ] = UNSPECIFIED
   elif args[ 'acl_target' ] != UNSPECIFIED \
         and args[ 'acl_target' ].upper() not in targets:
      raise cli.BindError(
         '"Invalid Target (run show aracl rules -h for more help)"' )

   # In case where both acl_target and acl_id (show aracl rules) has not been
   # mentioned, our args would be {'acl_target' : -1, 'acl_id' : -1}
   # which should remain same.
   elif args[ 'acl_target' ] == UNSPECIFIED:
      cmd_args[ 'acl_target' ] = args[ 'acl_target' ]
   else:
      cmd_args[ 'acl_target' ] = targets[ args[ 'acl_target' ].upper() ]

   # Making some checks for valid acl_id
   if args[ 'acl_id' ] is None:
      cmd_args[ 'acl_id' ] = UNSPECIFIED
   else:
      cmd_args[ 'acl_id' ] = args[ 'acl_id' ]

   return cmd_args

@commands.cmd( 'show aracl rules [RULES_CMD_OPTS...]' )
def show_aracl_rules( cli, opts ):

   def getIpStr( ip ):
      octets = [ ( ip >> 24 ) & 0xff, ( ip >> 16 ) & 0xff, ( ip >> 8 ) & 0xff,
                                                                  ip & 0xff ]
      ipStr = ".".join( map( str, octets ) )
      return ipStr

   def action( permit ):
      return "permit" if permit else "deny"

   def ProtoName( Proto ):
      return Proto if Proto not in ipProtocolNumberToName \
                   else ipProtocolNumberToName[ Proto ]

   mods = [ mod.name for mod in cli.bess.list_modules().modules
            if mod.mclass == "ArAclDb" ]

   if not mods:
      return

   args = _show_aracl_rules_parse_options( cli, opts )
   cmd_args = _show_aracl_rules_args_preproc( cli, args )

   # Get Acl Targets to Enum value from Data path
   try:
      ret = cli.bess.run_module_command( mods[ 0 ], 'getAclTargetInfoResponse',
                                         'EmptyArg', {} )
      targetToEnumMap = {}
      for targetEnumInfo in ret.acl_target_info:
         targetToEnumMap.update(
               { targetEnumInfo.target_name : targetEnumInfo.target_id } )
      enumToTargetMap = dict(
               ( reversed( item ) for item in targetToEnumMap.items() ) )
   except pybess.bess.BESS.Error as error:
      cli.fout.write( error.errmsg )
      return

   # To do: Please refer to BUG974970
   # The below check must be removed whenever the command starts to
   # support the targets mentioned in the bug.
   if targetToEnumMap[ 'ACL_TARGET_OPENFLOW' ] <= cmd_args[ 'acl_target' ] \
      <= targetToEnumMap[ 'ACL_TARGET_CLASSIFICATION' ]:
      cli.fout.write( 'This command currently does not support ACL_TARGET_OPENFLOW,'
                     ' ACL_TARGET_AEGIS_L4XFORM, ACL_TARGET_AEGIS_POLICY,'
                     ' and ACL_TARGET_CLASSIFICATION\n' )
      return

   ret = None

   try:
      ret = cli.bess.run_module_command( mods[ 0 ], 'getArAclDbRules',
                                        'ShowAclArg', cmd_args )
   except pybess.bess.BESS.Error as error:
      cli.fout.write( error.errmsg )
      return

   # Printing the rules on the bessctl command line.
   for target in ret.target_data:
      if len( target.acl_id_data ) != 0:
         cli.fout.write( f'ACL_TARGET: {enumToTargetMap[target.acl_target]}\n' )
         cli.fout.write( ' Total rules configured for this target = '
                         f'{target.total_rules}\n' )
      for id_ in target.acl_id_data:
         cli.fout.write( f'  IP Access List ACL_ID: {id_.acl_id}\n' )
         cli.fout.write( '   Total Rules configured for this acl_id'
                         f' = {id_.num_rules}\n' )

         for rule in id_.rule:
            cli.fout.write( f'    {rule.counter_value} ' )
            if target.acl_target in [ targetToEnumMap[ 'ACL_TARGET_DP' ],
                                      targetToEnumMap[ 'ACL_TARGET_MIRROR' ] ]:
               cli.fout.write( f'{action(rule.permit)} ' )

            cli.fout.write( f'{ProtoName(rule.proto)} ' )
            if getIpStr( rule.src_ip ) == '0.0.0.0' and \
               IPAddress( getIpStr( rule.src_ip_mask ) ).netmask_bits() == 0:
               cli.fout.write( "any " )
            else:
               cli.fout.write( f'{getIpStr(rule.src_ip)}/' )
               cli.fout.write(
                  f'{IPAddress( getIpStr(rule.src_ip_mask) ).netmask_bits()} ' )
            if rule.src_port_lower != 0 or rule.src_port_upper != 65535:
               cli.fout.write( f'{rule.src_port_lower}-' )
               cli.fout.write( f'{rule.src_port_upper} ' )
            if getIpStr( rule.dst_ip ) == '0.0.0.0' and \
               IPAddress( getIpStr( rule.dst_ip_mask ) ).netmask_bits() == 0:
               cli.fout.write( "any " )
            else:
               cli.fout.write( f'{getIpStr( rule.dst_ip )}/' )
               cli.fout.write(
                  f'{IPAddress( getIpStr(rule.dst_ip_mask) ).netmask_bits()} ' )
            if rule.dst_port_lower != 0 or rule.dst_port_upper != 65535:
               cli.fout.write( f'{rule.dst_port_lower}-' )
               cli.fout.write( f'{rule.dst_port_upper} ' )
            if rule.ttl_lower != 0 or rule.ttl_upper != 255:
               cli.fout.write( f'ttl:{rule.ttl_lower}-' )
               cli.fout.write( f'{rule.ttl_upper} ' )
            if rule.tos != 0:
               cli.fout.write( f'tos:{rule.tos} ' )
            if rule.len_lower != 0 or rule.len_upper != 65535:
               cli.fout.write( f'length:{rule.len_lower}-' )
               cli.fout.write( f'{rule.len_upper} ' )
            if rule.icmp_type != 0:
               cli.fout.write( f'icmp_type:{rule.icmp_type} ' )
            if rule.icmp_code != 0:
               cli.fout.write( f'icmp_code:{rule.icmp_code} ' )
            if rule.tcp_flag != 0:
               cli.fout.write( f'tcp_flag:{rule.tcp_flag} ' )

            if target.acl_target == targetToEnumMap[ 'ACL_TARGET_DP' ]:
               if rule.log:
                  cli.fout.write( 'log ' )
               if rule.implicit_deny:
                  cli.fout.write( '(default)' )

            if target.acl_target == targetToEnumMap[ 'ACL_TARGET_POLICY_QOS' ]:
               cli.fout.write( f'class_index:{rule.policy_class_index} ' )

            if target.acl_target == targetToEnumMap[ 'ACL_TARGET_CPU_CLASSIFY' ]:
               cli.fout.write( f'tc:{rule.tc_data} ' )

            if target.acl_target == targetToEnumMap[ 'ACL_TARGET_NAT' ]:
               cli.fout.write( f'entry_id:{rule.nat_entry_id} ' )
               cli.fout.write( f'flags:{rule.nat_flags} ' )
               cli.fout.write( f'pool_id:{rule.nat_pool_id} ' )
               cli.fout.write( f'conn_mark:{rule.nat_conn_mark} ' )
               cli.fout.write( f'profile_id:{rule.nat_profile_id} ' )
               cli.fout.write( f'twice_addr:{rule.nat_twice_addr} ' )
               cli.fout.write( f'twice_port:{rule.nat_twice_port} ' )

            cli.fout.write( '\n' )
         cli.fout.write( '\n' )

@commands.cmd( 'show aracl rules -h' )
def show_aracl_rules_help( cli ):
   opts = [ '-h' ]
   _show_aracl_rules_parse_options( cli, opts )

@commands.var_attrs( '[RULES_CMD_OPTS...]' )
def aracl_rules_var_attrs():
   return ( 'opts', '[OPTIONS](run show aracl rules -target <acl_target>'
           '-id <acl_id> ), -h for more help', [] )
