# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
# pylint: disable=import-error
# pylint: disable=broad-except
import commands
from bessctlPluginLib import (
      getIntfShortNameFromId,
      getFlowKeyStr,
)

def show_internet_exit_counters_header( cli ):
   cli.fout.write( "Internet Exit Counters:\n" )
   cli.fout.write( "-----------------------\n" )

def show_internet_exit_counters( cli, stats ):
   cli.fout.write( f"No of policies allocated : {stats.policy_alloc_cnt}\n" )
   cli.fout.write( f"No of policies freed : {stats.policy_free_cnt}\n" )
   cli.fout.write( f"No of flows allocated : {stats.flow_alloc_cnt}\n" )
   cli.fout.write( f"No of flows freed : {stats.flow_free_cnt}\n" )
   cli.fout.write( "No of policy allocation failures : " +
         f"{stats.policy_alloc_fail_cnt}\n" )
   cli.fout.write( "No of policy addition failures : " +
         f"{stats.policy_add_fail_cnt}\n" )
   cli.fout.write( "No of successful exit additions to policies : " +
         f"{stats.policy_add_exit_success_cnt}\n" )
   cli.fout.write( "No of failed exit additions to policies : " +
         f"{stats.policy_add_exit_fail_cnt}\n" )
   cli.fout.write( "No of policy deletion failures : " +
         f"{stats.policy_del_fail_cnt}\n" )
   cli.fout.write( "No of successful exit deletions from policies : " +
         f"{stats.policy_del_exit_success_cnt}\n" )
   cli.fout.write( "No of policy deletion failures due to missing policy : " +
         f"{stats.policy_del_policy_not_found_cnt}\n" )
   cli.fout.write( "No of policy deletion failures due to missing exit : " +
         f"{stats.policy_del_exit_not_found_cnt}\n" )
   cli.fout.write( "No of policies added to policy database : " +
         f"{stats.policy_added_to_db_cnt}\n" )
   cli.fout.write( "No of policies deleted from policy database : " +
         f"{stats.policy_deleted_from_db_cnt}\n" )
   cli.fout.write( "No of flow allocation failures : " +
         f"{stats.flow_alloc_fail_cnt}\n" )
   cli.fout.write( "No of packets detected without flow cache entries : " +
         f"{stats.no_flow_cache_cnt}\n" )
   cli.fout.write( "No of packets dropped due to non availability of usable exits : "
         + f"{stats.no_usable_exit_cnt}\n" )
   cli.fout.write( "No of exit reselection failures : " +
         f"{stats.exit_reselect_fail_cnt}\n" )
   cli.fout.write( "No of packets dropped due to invalid result type : " +
         f"{stats.invalid_result_type_cnt}\n" )
   cli.fout.write( "No of selected exits that are not usable : " +
         f"{stats.selected_exit_not_usable_cnt}\n" )
   cli.fout.write( "No of flow user objects set by other thread : " +
         f"{stats.set_flow_user_obj_race_cnt}\n" )
   cli.fout.write( "No of flows released due to invalid policy : " +
         f"{stats.freed_flows_with_invalid_policy_cnt}\n" )
   cli.fout.write( "No of flows assigned remote exits : " +
         f"{stats.remote_exit_select_cnt}\n" )
   cli.fout.write( "No of packets sent for remote exit reselection : " +
         f"{stats.remote_exit_reselect_cnt}\n" )

def getIeModName( cli ):
   for m in cli.bess.list_modules().modules:
      if m.mclass in ( 'InternetExit', 'SiEdge' ):
         return m.name
   return None

@commands.cmd( 'show internet-exit counters',
     'Show Internet Exit counters' )
def show_internet_exit_counters_cmd( cli ):
   modName = getIeModName( cli )
   if not modName:
      return
   stats = cli.bess.run_module_command( modName, 'getCounters', 'EmptyArg', {} )
   show_internet_exit_counters_header( cli )
   show_internet_exit_counters( cli, stats )
   cli.fout.write( '\n' )

ieResultType = {
      1 : "via_intf",
      2 : "via_fw",
      3 : "fib-lookup",
      4 : "remote",
      }

def show_internet_exit_policy( cli, policyInfo ):

   def get_result_type_str( result_type ):
      return ieResultType[ result_type ] if result_type in ieResultType else \
             'invalid'

   def get_nh_index_str( nh_index ):
      return 'N/A' if nh_index == 0xFFFFFFFF else f'{nh_index}'

   avtInfo = policyInfo.avt_info
   cli.fout.write( f'vni {avtInfo.vni} avt_id {avtInfo.avt_id}\n' )
   for ieExit in policyInfo.exit_info:
      cli.fout.write(
         f'  index {ieExit.exit_index:2d} '
         f'result_type {get_result_type_str( ieExit.result_type ):10s} '
         f'o_gate {ieExit.o_gate:4d} '
         f'nh_index {get_nh_index_str( ieExit.nh_index ):4s} '
         f'exit_intf {getIntfShortNameFromId( ieExit.exit_intf_id ):10s}\n' )

def show_internet_exit_policy_helper( cli, cmd, paramType, params,
      printHeader=True ):
   def _vniAvtIdSetkey( item ):
      return ( item.avt_info.vni, item.avt_info.avt_id )

   modName = getIeModName( cli )
   if not modName:
      return

   try:
      response = cli.bess.run_module_command( modName, cmd, paramType, params )
   except Exception as error:
      # pylint: disable=no-member
      print( error.errmsg )
      return

   if not response.policy_info:
      return
   if printHeader:
      cli.fout.write( 'Internet exit policy lookup table:\n' )
   for policyInfo in sorted( response.policy_info, key=_vniAvtIdSetkey ):
      show_internet_exit_policy( cli, policyInfo )
   cli.fout.write( '\n' )

@commands.cmd( 'show internet-exit policy VNI AVTID',
      'Show internet exit policy information for specific VNI and AVTID' )
def show_ie_policy_lookup( cli, vni, avt_id ):
   show_internet_exit_policy_helper( cli, 'getPolicyInfo', 'VniAvtArg',
         { 'vni' : vni, 'avt_id' : avt_id }, printHeader=False )

@commands.cmd( 'show internet-exit policy all',
      'Show internet exit policy information for all vni/avt' )
def show_ie_policy_lookup_all( cli ):
   show_internet_exit_policy_helper( cli, 'getAllPoliciesInfo', 'EmptyArg', {} )

def show_ie_flow_info_helper( cli, modName, reqArgs ):
   try:
      response = cli.bess.run_module_command( modName,
                     'getIeFlowInfo', 'InternetExitGetFlowReqArg', reqArgs )
   except Exception as error:
      # pylint: disable=no-member
      print( error.errmsg )
      return 0

   for entry in response.flow_info:
      avtIdStr = f'avt_id {entry.avt_id}'
      intfNameStr = f'exit_intf {getIntfShortNameFromId( entry.exit_intf_id )}'
      cli.fout.write(
         f'{getFlowKeyStr( entry.flow_key )}\n'
         f'   {avtIdStr} {intfNameStr}\n' )

   return response.next_iter

@commands.cmd( 'show internet-exit flow-info',
      'Show internet exit flowcache information' )
def show_ie_flow_info( cli ):
   modName = getIeModName( cli )
   if not modName:
      return

   reqArgs = {
      'max_iterate' : 100,
      'start_iter' : 0x0, # start from beginning
   }

   cli.fout.write( 'Internet Exit flowcache info:\n' )
   while True:
      # get the next_iter from where we have to continue
      next_iter = show_ie_flow_info_helper( cli, modName, reqArgs )
      if next_iter == 0x0:
         # we are done iterating through all flows
         break
      reqArgs[ 'start_iter' ] = next_iter
   cli.fout.write( '\n' )
