# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import commands
import socket

def _show_ipsec_sa_stats( cli, args ):
   ret = cli.bess.run_module_command( 'IpsecManagementModule',
                                      'dumpStatsForSA',
                                      'IpsecDumpSAStatsReqArg', args )
   for entry in ret.saStatsEntries:
      cli.fout.write( "Spi 0x%x: pkts %d bytes %d replayErr %d "
                      "integrityErr %d addTime %u, useTime %u "
                      "core %d dstip 0x%x encapid %d "
                      "replayWindErr %d pathUdpErr %d \n" %
                      ( entry.sa_spi, entry.pkts, entry.bytes,
                        entry.replayErrors, entry.integrityErrors,
                        entry.addTime, entry.useTime, entry.coreid,
                        socket.ntohl( entry.dst_addr ), entry.encap_id,
                        entry.replayWindowErrors, entry.pathUdpHdrError ) )
   cli.fout.write( '\n' )

def _show_ipsec_sa_stats_ingress( cli, cpulist ):
   args = {}
   args[ 'sa_direction' ] = 1
   if cpulist:
      args[ 'cpuList' ] = cpulist
   else:
      args[ 'cpuList' ] = 0xffffffff
   cli.fout.write( 'show ipsec-sa-stats ingress:\n' )
   _show_ipsec_sa_stats( cli, args )

@commands.cmd( 'show ipsec-sa-stats ingress [CPULIST]',
     'show ipsec SA stats for ingress: cpulist is 0-16 or 0xffffffff' )
def show_ipsec_sa_stats_ingress( cli, cpulist ):
   _show_ipsec_sa_stats_ingress( cli, cpulist )

def _show_ipsec_sa_stats_egress( cli, cpulist ):
   args = {}
   args[ 'sa_direction' ] = 0
   if cpulist:
      args[ 'cpuList' ] = cpulist
   else:
      args[ 'cpuList' ] = 0xffffffff
   cli.fout.write( 'show ipsec-sa-stats egress:\n' )
   _show_ipsec_sa_stats( cli, args )

@commands.cmd( 'show ipsec-sa-stats egress [CPULIST]',
     'show ipsec SA stats for egress: cpulist is 0-16 or 0xffffffff' )
def show_ipsec_sa_stats_egress( cli, cpulist ):
   _show_ipsec_sa_stats_egress( cli, cpulist )

@commands.cmd( 'show ipsec-sa-stats [CPULIST]',
     'show ipsec SA stats for all/specified CPULIST in the datapath' )
def show_ipsec_sa_stats( cli, cpulist ):
   _show_ipsec_sa_stats_ingress( cli, cpulist )
   _show_ipsec_sa_stats_egress( cli, cpulist )

def _show_ipsec_sa_state( cli, args ):
   ret = cli.bess.run_module_command( 'IpsecManagementModule',
                                      'dumpStateForSA',
                                      'IpsecDumpSAStateReqArg', args )
   for sa in ret.saStateEntries:
      cli.fout.write( "Spi 0x%x: seq %d srcip 0x%x dstip 0x%x calgo %d "
                     "aalgo %d ckeylen %d akeylen %d flags 0x%x dig_len "
                     "%d ivlen %d blksize %d replay_window %d \n" %
                     ( sa.sa_spi, sa.sa_seq, socket.ntohl( sa.src_addr ),
                       socket.ntohl( sa.dst_addr ), sa.cipher_algorithm,
                       sa.auth_algorithm, sa.cipher_key_len,
                       sa.auth_key_len, sa.sa_flags, sa.sa_digest_len,
                       sa.sa_iv_len, sa.sa_block_size, sa.replay_window ) )
   cli.fout.write( '\n' )

def _show_ipsec_sa_state_ingress( cli ):
   args = {}
   args[ 'sa_direction' ] = 1
   cli.fout.write( 'show ipsec-sa-state ingress:\n' )
   _show_ipsec_sa_state( cli, args )

@commands.cmd( 'show ipsec-sa-state ingress',
     'show ipsec SA state for ingress' )
def show_ipsec_sa_state_ingress( cli ):
   _show_ipsec_sa_state_ingress( cli )

def _show_ipsec_sa_state_egress( cli ):
   args = {}
   args[ 'sa_direction' ] = 0
   cli.fout.write( 'show ipsec-sa-state egress:\n' )
   _show_ipsec_sa_state( cli, args )

@commands.cmd( 'show ipsec-sa-state egress',
     'show ipsec SA state for egress' )
def show_ipsec_sa_state_egress( cli ):
   _show_ipsec_sa_state_egress( cli )

@commands.cmd( 'show ipsec-sa-state',
     'show ipsec SA state' )
def show_ipsec_sa_state( cli ):
   _show_ipsec_sa_state_ingress( cli )
   _show_ipsec_sa_state_egress( cli )

def _show_ipsec_module_stats( cli, args ):
   ret = cli.bess.run_module_command( 'IpsecManagementModule',
                                      'dumpModuleStatsAll',
                                      'IpsecDumpModuleStatsReqArg', args )
   cli.fout.write( 'show ipsec-module-stats:\n' )
   for entry in ret.moduleStatsEntries:
      cli.fout.write( "core %d: pktsIn/Out %d %d EncrBadSpi %d "
         "dropped %d decr %d encr %d rdrops %d natFail %d InvEncap %d "
         "SaLookupErr %d fcAddError %d cryptoErrIn/Out %d %d badProto %d "
         "pktsEspInvIn/Out %d %d pktsHwCryptoError %d pktsHwCryptoDone %d "
         "pktsHwCryptoDecr %d pktsHwCryptoEncr %d \n" %
         ( entry.coreid, entry.pktsInput, entry.pktsOutput,
           entry.pktsEncryptSpiInvalid, entry.pktsDropped, entry.pktsDecrypted,
           entry.pktsEncrypted, entry.pktsReplayDropped,
           entry.pktsNattEncapFailed, entry.pktsEncapIdInvalid,
           entry.pktsSaLookupErrors, entry.pktsFlowCacheAddErrors,
           entry.pktsInCryptoQErrors, entry.pktsOutCryptoQErrors, entry.badProto,
           entry.pktsInEspInvalid, entry.pktsOutEspInvalid, entry.pktsHwCryptoError,
           entry.pktsHwCryptoDecryptDone, entry.pktsHwCryptoDecrypted,
           entry.pktsHwCryptoEncrypted ) )
   cli.fout.write( '\n' )

@commands.cmd( 'show ipsec-module-stats [CPULIST]',
     'show ipsec module stats [CPULIST]: cpulist is 0-16 or 0xffffffff' )
def show_ipsec_module_stats( cli, cpulist ):
   args = {}
   args[ 'cpuList' ] = 0xffffffff
   if cpulist:
      args[ 'cpuList' ] = cpulist
   _show_ipsec_module_stats( cli, args )

def _show_ipsec_crypto_stats( cli, args ):
   ret = cli.bess.run_module_command( 'IpsecManagementModule',
                                      'dumpCryptodevStats',
                                      'IpsecDumpCryptodevStatsRespArg', args )
   cli.fout.write( 'show ipsec-crypto-stats:\n' )
   for entry in ret.cdevStatsEntries:
      if entry.deviceName == "":
         continue
      cli.fout.write( "%s (%d): enq %d deq %d enqErr %d deqErr %d\n" %
         ( entry.deviceName, entry.deviceId, entry.enqueuedCount,
           entry.dequeuedCount, entry.enqueueErrCount,
           entry.dequeueErrCount ) )
   cli.fout.write( '\n' )

@commands.cmd( 'show ipsec-crypto-stats',
     'show ipsec crypto stats' )
def show_ipsec_crypto_stats( cli ):
   args = {}
   _show_ipsec_crypto_stats( cli, args )
