# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
from .option_validator_factory import OptionValidatorFactory

class OptionParser(object):

   def __init__(self, params):
      self.params = params
      self.param_len = len(params) if params else 0
      self.pos = 0
      self.remaining = []
      self.options = {}
      self.register_options()

   @staticmethod
   def get_validation_error_message(option, entry):
      return "{} option expected {} parameter".format(option, entry["type_"])

   @staticmethod
   def get_invalid_choice_error_message(option, entry):
      return "{} can only be {}".format(option, str(entry["valid_values_"]))

   def register_options(self):
      pass

   def add_option(self, name_, type_, valid_values=None, action=None):
      self.options[name_] = {
               "type_" : type_,
               "value_" : None,
               "valid_values_": valid_values,
               "action_": action
            }

   def verify(self):
      for key, entry in self.options.items():
         if entry["value_"] is not None and entry["valid_values_"] is not None:
            if entry["value_"] not in entry["valid_values_"]:
               error_message = self.get_invalid_choice_error_message(key, entry)
               raise Exception(error_message)

   def parse(self):
      while self.pos < self.param_len:
         next_token = self.params[self.pos]
         if next_token in self.options:
            entry = self.options.get(next_token)
            if entry["type_"] == "action" and entry["action_"] is not None:
               entry["action_"]()
               raise Exception()
            if entry["type_"] == "bool":
               entry["value_"] = True
               self.pos += 1
               continue
            validator = OptionValidatorFactory.create(entry["type_"])
            validator.set_params(self.params, self.pos)
            try:
               validator.run()
            except:
               raise Exception(self.get_validation_error_message(next_token, entry))
            entry["value_"] = validator.get_value()
            self.pos = validator.next_pos()
         else:
            self.remaining.append(next_token)
            self.pos += 1
      self.verify()

   def get_value(self, option, default=None):
      value = self.options[option]["value_"]
      if value is None:
         return default
      return value

   def get_remaining(self):
      return " ".join(self.remaining)

