#!/bin/sh

# Copyright (C) 2020-2021 Internet Systems Consortium, Inc. ("ISC")
#
# This Source Code Form is subject to the terms of the Mozilla Public
# License, v. 2.0. If a copy of the MPL was not distributed with this
# file, You can obtain one at http://mozilla.org/MPL/2.0/.

# shellcheck disable=SC1091
# SC1091: Not following: ... was not specified as input (see shellcheck -x).

# Exit with error if commands exit with non-zero and if undefined variables are
# used.
set -eu

# shellcheck disable=SC2034
# SC2034: ... appears unused. Verify use (or export if used externally).
prefix="/usr"

# Include utilities. Use installed version if available and
# use build version if it isn't.
if [ -e ${prefix}/share/kea/scripts/admin-utils.sh ]; then
    . "${prefix}/share/kea/scripts/admin-utils.sh"
else
    . "/bld/kea-dhcp/Artools-rpmbuild/kea-2.0.0/src/bin/admin/admin-utils.sh"
fi

VERSION=$(mysql_version "$@")

if [ "$VERSION" != "9.3" ]; then
    printf 'This script upgrades 9.3 to 9.4. '
    printf 'Reported version is %s. Skipping upgrade.\n' "${VERSION}"
    exit 0
fi

mysql "$@" <<EOF

# Starting from this version we allow specifying multiple IP reservations
# for the same address in certain DHCP configurations. The server may check
# uniqueness of the IP addresses on its own. This is no longer checked at
# the database level to facilitate the use cases when a single host may
# get the same reserved IP address via different interfaces.

# Replace the unique index with non-unique index so the queries for
# hosts by IPv4 address are still efficient.
DROP INDEX key_dhcp4_ipv4_address_subnet_id ON hosts;
CREATE INDEX key_dhcp4_ipv4_address_subnet_id_identifier
    ON hosts (ipv4_address ASC, dhcp4_subnet_id ASC);

# Replace the unique index with non-unique index so the queries for
# hosts by IPv6 address are still efficient.
DROP INDEX key_dhcp6_address_prefix_len ON ipv6_reservations;
CREATE INDEX key_dhcp6_address_prefix_len
    ON ipv6_reservations (address ASC, prefix_len ASC);

# Stop using a trigger to delete entries dependent on hosts table.
# Use cascade action instead. This works better with complex delete
# statements.
DROP TRIGGER IF EXISTS host_BDEL;

# Replace existing constraint to set cascade actions.
ALTER TABLE ipv6_reservations DROP FOREIGN KEY fk_ipv6_reservations_Host;
ALTER TABLE ipv6_reservations ADD CONSTRAINT fk_ipv6_reservations_Host
    FOREIGN KEY (host_id)
        REFERENCES hosts(host_id)
        ON DELETE CASCADE ON UPDATE CASCADE;

ALTER TABLE dhcp4_options ADD CONSTRAINT fk_dhcp4_options_Host
    FOREIGN KEY (host_id)
        REFERENCES hosts(host_id)
        ON DELETE CASCADE ON UPDATE CASCADE;

ALTER TABLE dhcp6_options ADD CONSTRAINT fk_dhcp6_options_Host
    FOREIGN KEY (host_id)
        REFERENCES hosts(host_id)
        ON DELETE CASCADE ON UPDATE CASCADE;

# Update the schema version number
UPDATE schema_version
SET version = '9', minor = '4';

# This line concludes database upgrade to version 9.4.

EOF
