# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

'''
This patch creates /mnt/flash/skipFpgaUpgrade (if it doesn't already exist)
to force FPGA upgrade skips. The patch is needed when upgrading from releases
[ 4.20.5, 4.26.1 ] to SWIM formatted images w/ release 4.26.2+ so that
the ASU fast-boot reload doesn't attempt to unsquash FPGA files from
the target image's rootfs-i386.sqsh. Since SWIM does not have rootfs-i386.sqsh,
we need the FPGA upgrade skip or else the reload fast-boot command may
attempt to unzip/unsquash non-existing files and fail.
The /mnt/flash/skipFpgaUpgrade file is created with a cookie so that
on the up path of the ASU reload, we can remove the file in
NorCalInit to allow subsequent reloads to perform FPGA upgrades.
'''

from __future__ import absolute_import, division, print_function

import os
import zipfile

import AsuPatchBase

class AsuSkipFpgaUpgradePatch( AsuPatchBase.AsuPatchBase ):
   def __init__( self, initName ):
      self.fs = lambda:None
      self.skipCreated = False
      self.skipFpgaUpgradeFile = '/mnt/flash/skipFpgaUpgrade'
      self.asuPatchCookie = 'generatedByAsuPatch'
      AsuPatchBase.AsuPatchBase.__init__( self, initName )

   def initArgs( self, *args, **kwargs ):
      if 'asuReloadData' not in kwargs:
         return 1
      for argName, argVal in kwargs.items():
         if argName == 'asuReloadData':
            asuReloadContext = argVal
            self.fs = asuReloadContext.fs
      return 0

   def cleanup( self ):
      if self.skipCreated:
         try:
            os.remove( self.skipFpgaUpgradeFile )
         except OSError as err:
            print( "Error cleaning up FPGA Upgrade Skip File:" )
            print( err )
            return 1
      return 0

   def check( self ):
      return 0

   def reboot( self ):
      if os.path.exists( self.skipFpgaUpgradeFile ):
         # An overriding permanent skip file is already present:
         # keep it permanent (the asu generated one will be removed after upgrade).
         return 0

      # The destination image will be non-SWIM (single rootfs)
      # until we transition to using the SWIM Format as the default.
      # The patch is not applicable here since FPGA upgrades are supported
      # when ASU reloading to non-SWIM images
      swiZip = zipfile.ZipFile( self.fs.realFilename_ )
      if 'swimSqshMap' not in swiZip.namelist():
         return 0

      # Create the upgrade skip file and write the cookie to the file
      with open( self.skipFpgaUpgradeFile, 'w' ) as skipFile:
         skipFile.write( self.asuPatchCookie )
      self.skipCreated = True
      return 0

_asuSkipFpgaUpgradePatch = AsuSkipFpgaUpgradePatch( "AsuSkipFpgaUpgradePatch" )

def execute( stageVal, *args, **kwargs ):
   return _asuSkipFpgaUpgradePatch.execute( stageVal, *args, **kwargs )
