#!/usr/bin/env python
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
from __future__ import absolute_import, division, print_function
import Tac
import time
import sys
import os
import syslog
import Cell
import PyClient
import datetime as dt
from CliPlugin.AsuReloadCli import _asuData

# Force reload the latest AsuPatchBase from swi
# pylint: disable-msg=import-error
# pylint: disable-msg=wrong-import-position
sys.modules.pop( 'AsuPatchBase', None )
import AsuPatchBase

# waitFor and Timeout implementations carried over from ArPyUtils so as to
# not rely on the older versions' implementation of the same
class Timeout( Exception ):
   pass

def isFlashVfat():
   try:
      Tac.run( [ "df", "-t", "vfat", "/mnt/flash" ],
               stdout=Tac.DISCARD, stderr=Tac.DISCARD )
      return True
   except Tac.SystemCommandError:
      return False

def getIntFromEnv( envVar, default ):
   try:
      return int( os.environ.get( envVar, default ) )
   except ValueError:
      return int( default )

def savePreReloadLogs():
   # see BUG323324 for reasoning behind saving some logging information
   # memory values in this pre-reload area of code should be expressed in kibibytes
   filesToSync = set()
   try:
      preReloadLogsSaveFile = os.environ.get( "PRE_RELOAD_LOGS_SAVE_FILE",
                                              "pre_reload_logs.tgz" )
      preReloadLogsDestDir = os.environ.get( "PRE_RELOAD_LOGS_DEST_DIR",
                                             "/mnt/flash/debug" )
      try:
         os.makedirs( preReloadLogsDestDir )
      except OSError as e:
         if not os.path.isdir( preReloadLogsDestDir ):
            raise e
      preReloadLogsErrorPath = getDebugLogPath()
      preReloadLogsSavePath = os.path.join( preReloadLogsDestDir,
                                            preReloadLogsSaveFile )
      Tac.run( [ "timeout", "-k", "3", "3", "rm", "-f",
                 preReloadLogsSavePath ],
               asRoot=True, ignoreReturnCode=True,
               stdout=Tac.DISCARD, stderr=Tac.DISCARD )
      preReloadSaveBuffer = getIntFromEnv( "PRE_RELOAD_SAVE_BUFFER", "1024" )

      out = Tac.run( [ "df", "--block-size=1024", preReloadLogsDestDir ],
                       asRoot=True, stdout=Tac.CAPTURE, stderr=Tac.DISCARD )
      available = int( out.split( "\n" )[ 1 ].split()[ 3 ] )
      preReloadSaveMinSize = getIntFromEnv( "PRE_RELOAD_SAVE_MIN_SIZE", "512" )
      saveSize = ( available - preReloadSaveBuffer )
      if saveSize < preReloadSaveMinSize:
         filesToSync.add( preReloadLogsErrorPath )
         errMsg = ( "%sKiB available on /mnt/flash, too little space to"
                    " save pre-reload logs" % available )
         debugSyslog( syslog.LOG_NOTICE, errMsg )
      else:
         preReloadLogs = "/var/log/messages"
         if not isFlashVfat():
            # avoid writing too much on VFAT before reload
            preReloadLogs += " /var/log/agents/ /var/log/qt"
         preReloadLogs = os.environ.get( "PRE_RELOAD_LOGS", preReloadLogs )
         preReloadSaveTimeout = getIntFromEnv( "PRE_RELOAD_SAVE_TIMEOUT", "9" )
         preReloadSaveCmdZero = ( "timeout -k %d %d bash -c" %
                                  ( preReloadSaveTimeout, preReloadSaveTimeout ) )
         preReloadSaveCmdOne = ( "tar -zvc %s | head --bytes=%dK > %s" %
                                 ( preReloadLogs, saveSize, preReloadLogsSavePath ) )
         # Do this before running the command since it might raise an exception
         filesToSync.add( preReloadLogsSavePath )
         Tac.run( preReloadSaveCmdZero.split() + [ preReloadSaveCmdOne ],
                  asRoot=True, stdout=Tac.DISCARD, stderr=Tac.DISCARD )
   except Exception as e: # pylint: disable-msg=broad-except
      filesToSync.add( preReloadLogsErrorPath )
      debugSyslog( syslog.LOG_NOTICE, e )
   # end pre-reload log save

   # We just wrote a file to flash before reload; make sure we fsync the file
   # to avoid corruption.
   filesToSync = [ fname for fname in filesToSync if os.path.isfile( fname ) ]
   if filesToSync:
      Tac.run( [ 'SyncFile' ] + filesToSync, asRoot=True,
               ignoreReturnCode=True )

def waitFor( func, timeout=600.0, description=None,
             checkAfter=0.1, firstCheck=0.1, maxDelay=5 ):
   """
   An alternative to Tac.waitFor(), for packages which cannot depend on tacc.
   - there are no warnings, it raises 'Timeout' exception on timeout
   - instead of maxDelay we have a checkAfter, seconds to elapse before
     evaluating the condition
   - sleep is not an option, we always sleep
   - when func() returns something which evaluates to True, that value will
     be returned
   - we always wait before calling func, first execution of func is after
     checkAfter seconds.
   """
   expiry = time.time() + timeout
   if firstCheck:
      time.sleep( firstCheck )
   checkWait = checkAfter
   while True:
      lastChance = time.time() >= expiry
      result = func()
      if result:
         return result
      elif lastChance:
         raise Timeout( "Timed out waiting for %s" % description )
      timeLeft = max( expiry - time.time(), 0 )
      checkWait = min( checkWait, timeLeft )
      time.sleep( checkWait )
      checkWait = min( maxDelay, checkWait * 2 )

def getDebugLogPath():
   preReloadLogsSaveErrorFile = os.environ.get(
      "PRE_RELOAD_LOGS_ERROR_FILE", "pre_reload_asupatch_logs" )
   preReloadLogsDestDir = os.environ.get(
      "PRE_RELOAD_LOGS_DEST_DIR", "/mnt/flash/debug" )
   preReloadLogsErrorPath = os.path.join(
      preReloadLogsDestDir, preReloadLogsSaveErrorFile )
   return preReloadLogsErrorPath

def debugSyslog( level, log ):
   DEBUGSYSLOG = False
   msg = "%s: %s" % (
       str( dt.datetime.now() ), str( log )
   )
   if DEBUGSYSLOG:
      syslog.syslog( level, msg )
   else:
      preReloadLogsErrorPath = getDebugLogPath()
      Tac.run( [ "mkdir", "-p",
                 os.path.dirname( preReloadLogsErrorPath ) ],
               asRoot=True, ignoreReturnCode=True,
               stdout=Tac.DISCARD, stderr=Tac.DISCARD )
      try:
         with open( preReloadLogsErrorPath, "a" ) as errorFile:
            errorFile.write( msg + "\n" )

      except Exception as e: # pylint: disable-msg=broad-except
         print( e )

class MonitorStageAndCopyLogsThread( AsuPatchBase.AsuPatchBaseThread ):

   def __init__( self, sysdbPath ):
      pcl = PyClient.PyClient( sysdbPath, "Sysdb" )
      self.root = pcl.agentRoot()
      super( # pylint: disable=super-with-arguments
         MonitorStageAndCopyLogsThread, self
      ).__init__()

   def addDynamicStageDependency( self, agentName, stageName ):
      debugSyslog(
         syslog.LOG_NOTICE,
         'Adding new dependency on %s' % ( stageName ) )
      stageDir = self.root.entity[
         Cell.path( "stageAgentStatus/shutdown" )
      ]
      if agentName not in stageDir:
         stageDir.newEntity( "Stage::AgentStatus", agentName )
      asuPatchAgent = stageDir[ agentName ]
      stageRequest = Tac.newInstance(
         "Stage::StageRequest", agentName, stageName, 30
      )
      asuPatchAgent.stageRequest.addMember( stageRequest )

   def markAsuPatchAgentStageComplete( self, agentName, stageName ):
      asuPatchAgent = self.root.entity[
         Cell.path( "stageAgentStatus/shutdown/%s" % ( agentName ) )
      ]
      stageKey = Tac.Value(
         "Stage::AgentStatusKey", agentName, stageName, "default"
      )
      asuPatchAgent.complete[ stageKey ] = True

   def isStageStarted( self, stage ):
      shutProgressEntity = self.root.entity[
          Cell.path( "stage/shutdown/progress" )
      ].progress
      if 'default' in shutProgressEntity:
         return stage in shutProgressEntity[ 'default' ].stage
      return False

   def waitForStageAndStart( self, stage, func ):
      debugSyslog( syslog.LOG_NOTICE, "Waiting for stage completion.." )
      try:
         waitFor(
            lambda: self.isStageStarted( stage ) or self.isSSUPatchThreadStopped(),
            description="Wait till end of %s starts" % ( stage ),
            maxDelay=1, timeout=600
         )
      except Timeout:
         syslog.syslog(
            syslog.LOG_ERR,
            'Timed out waiting for Stage progression to hit %s' % ( stage )
         )
         return

      if self.isSSUPatchThreadStopped():
         return
      debugSyslog( syslog.LOG_NOTICE, "Starting payload now.." )
      func()

   def getAllLeafStages( self ):
      stageColl = self.root.entity[
          Cell.path( "stage/shutdown/status" )
      ].stage
      allDeps = []
      for stg in stageColl:
         stage = stageColl[ stg ]
         allDeps += stage.dependency.keys()
      return list( set( stageColl.keys() ) - set( allDeps ) )

   def addNewStage( self, stageName, deps ):
      dummyAgent = "__dummyInternal1__"
      agentConfig = self.root.entity[
         Cell.path( "stageInput/shutdown/%s" % ( dummyAgent ) ) ]
      event = agentConfig.newEvent( stageName, dummyAgent )

      # Add stage dependency for the dynamic stage
      for dep in deps:
         event.dependency[ dep ] = True

      # Dummy stage is set to complete by default
      event.complete[ 'default' ] = True

   def clearDebugTraces( self ):
      Tac.run(
         [ "timeout", "-k", "3", "3", "rm", "-f", getDebugLogPath() ],
         asRoot=True, ignoreReturnCode=True,
         stdout=Tac.DISCARD, stderr=Tac.DISCARD
      )

   def run( self ):
      shutPathLogCopyStage = 'SaveShutPathLogs'
      shutPathLogCopyAgent = "AsuPatchForShutPathLogs"

      self.clearDebugTraces()
      deps = self.getAllLeafStages()
      if shutPathLogCopyStage not in deps:
         debugSyslog(
            syslog.LOG_NOTICE,
            "Adding %s that depends on: %s" % (
               shutPathLogCopyStage, str( deps ) )
         )
         self.addNewStage( shutPathLogCopyStage, deps )
      else:
         debugSyslog(
            syslog.LOG_NOTICE,
            "Leaf stage %s already exists" % ( shutPathLogCopyStage )
         )
      debugSyslog(
          syslog.LOG_NOTICE,
          "Engaging %s dependency on %s for copying shutpath logs" % (
             shutPathLogCopyAgent, shutPathLogCopyStage
          )
      )
      self.addDynamicStageDependency( shutPathLogCopyAgent, shutPathLogCopyStage )
      self.waitForStageAndStart( shutPathLogCopyStage, savePreReloadLogs )
      if self.isSSUPatchThreadStopped():
         debugSyslog( syslog.LOG_NOTICE, "Exit patch thread early" )
         return

      debugSyslog(
          syslog.LOG_NOTICE,
          "Agent %s now marking %s stage as complete" % (
             shutPathLogCopyAgent, shutPathLogCopyStage
          )
      )
      self.markAsuPatchAgentStageComplete(
         shutPathLogCopyAgent, shutPathLogCopyStage
      )

class StageCompletionLogsCopy( AsuPatchBase.AsuPatchBase ):
   def check( self ):
      thread = MonitorStageAndCopyLogsThread( self.em.sysname() )
      thread.start()
      if hasattr( _asuData, 'asuPatchThreads' ):
         _asuData.asuPatchThreads.append( thread )
      return 0

   def reboot( self ):
      pass

# This method is executed by AsuPatch in the shutdown path by the src image
def execute( stageVal, *args, **kwargs ):
   obj = StageCompletionLogsCopy( 'StageCompletionLogsCopy' )
   return obj.execute( stageVal, *args, **kwargs )

# This block is used if this script is triggered by the event handler
if __name__ == "__main__":
   if int( os.environ.get( 'EVENT_COUNT', 0 ) ) > 0:
      logCopyObj = StageCompletionLogsCopy( 'StageCompletionLogsCopy' )
      logCopyObj.check()
   else:
      shutPathLogCopyStageMain = 'SaveShutPathLogs'
      shutPathLogCopyAgentMain = "AsuPatchForShutPathLogs"
      debugSyslog(
          syslog.LOG_NOTICE,
          "On Init, engaging pseudo agent dependency on %s" % (
             shutPathLogCopyStageMain
          )
      )
      copyLogThreadObj = MonitorStageAndCopyLogsThread( "ar" )
      depsMain = copyLogThreadObj.getAllLeafStages()
      if shutPathLogCopyStageMain not in depsMain:
         copyLogThreadObj.addNewStage( shutPathLogCopyStageMain, depsMain )
      copyLogThreadObj.addDynamicStageDependency(
         shutPathLogCopyAgentMain, shutPathLogCopyStageMain
      )
