# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac

import re
import sys

# Force reload the latest AsuPatchBase from swi
# pylint: disable-msg=import-error
# pylint: disable-msg=wrong-import-position
sys.modules.pop( 'AsuPatchBase', None )
import AsuPatchBase

class ValidationError( Exception ):
   pass

class StrataLacpPkt( AsuPatchBase.AsuPatchBase ):
   def check( self ):
      return 0

   def reboot( self ):
      try:
         self.setLacpPackets()
      except ValueError:
         self.log( 'Unable to verify LACP packet configuration' )
         return 1
      except KeyboardInterrupt:
         self.log( 'User requested abort during LACP packet configuration' )
         return 1
      except ValidationError:
         self.log( 'Unable to set up LACP packet configuration' )
         return 1
      else:
         return 0

   def cleanup( self ):
      return 0

   def setLacpPackets( self ):
      cmd = [ 'python', '-m' 'Acons', 'Sysdb' ]
      # The original numbers are 15 and 165 for numLacpPdus and numPlaceHolder
      # packets respectively. This gives us 1 LACP packet for every 11
      # placeholder packets. The chip gets setup for a shaper on the queue
      # to only send 1 packet per second on the going down path. This gives us
      # 180 seconds of packets before we run. These new numbers are to boost
      # the time we have to prevent LACP from timing out by increasing the
      # number of packets enqueued before going down.
      expectNumLacpPdus = 20
      expectNumPlaceHolder = 228
      setLacpPktValues = ( 'cd /ar/Sysdb/asu/hardware/status\n' +
         '_.numLacpPdusPerPort=%d\n_.numPlaceholderPktsPerPort=%d' %
         ( expectNumLacpPdus, expectNumPlaceHolder ) )
      output = Tac.run( cmd, input=setLacpPktValues, asRoot=True,
                        stdout=Tac.CAPTURE, stderr=Tac.CAPTURE )
      checkLacpPktValues = "cd /ar/Sysdb/asu/hardware/status\nls -l"
      output = Tac.run( cmd, input=checkLacpPktValues, asRoot=True,
                        stdout=Tac.CAPTURE, stderr=Tac.CAPTURE )
      m = re.search( r'numLacpPdusPerPort\s*:\s(?P<numLacpPdus>\d+)', output )
      numLacpPdus = int( m.group( 'numLacpPdus' ) )
      m = re.search( r'numPlaceholderPktsPerPort\s*:\s(?P<numPlaceholder>\d+)',
                     output )
      numPlaceholder = int( m.group( 'numPlaceholder' ) )
      if numLacpPdus != expectNumLacpPdus or numPlaceholder != expectNumPlaceHolder:
         raise ValidationError

def execute( stageVal, *args, **kwargs ):
   obj = StrataLacpPkt( 'StrataLacpPkt' )
   return obj.execute( stageVal, *args, **kwargs )
